<?php
/**
 * Verification script for database sessions
 * Run this script to verify that the database session handler is set up correctly
 */

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';

echo "=== Database Session Handler Verification ===\n\n";

// Check if table exists
try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $stmt = $conn->query("SHOW TABLES LIKE 'user_sessions'");
    if ($stmt->rowCount() === 0) {
        echo "❌ ERROR: Table 'user_sessions' does not exist!\n";
        echo "   Please run: mysql -u root -p personal_expenses < database/migration_add_user_sessions.sql\n\n";
        exit(1);
    } else {
        echo "✓ Table 'user_sessions' exists\n";
    }
    
    // Check table structure
    $stmt = $conn->query("DESCRIBE user_sessions");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $columnNames = array_column($columns, 'Field');
    
    $requiredColumns = ['session_id', 'user_id', 'data', 'expires_at', 'last_activity', 'created_at'];
    $missingColumns = array_diff($requiredColumns, $columnNames);
    
    if (!empty($missingColumns)) {
        echo "❌ ERROR: Missing columns: " . implode(', ', $missingColumns) . "\n";
        exit(1);
    } else {
        echo "✓ Table structure is correct\n";
    }
    
    // Check if there are any sessions
    $stmt = $conn->query("SELECT COUNT(*) as count FROM user_sessions WHERE expires_at > NOW()");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    echo "✓ Active sessions in database: " . $result['count'] . "\n\n";
    
    // Test session handler
    echo "Testing session handler...\n";
    try {
        $handler = new DatabaseSessionHandler();
        echo "✓ DatabaseSessionHandler instantiated successfully\n";
        
        // Test read/write
        $testSessionId = 'test_' . time();
        $testData = 'user_id|i:1;test|s:4:"data";';
        
        $writeResult = $handler->write($testSessionId, $testData);
        if ($writeResult) {
            echo "✓ Session write test passed\n";
            
            $readResult = $handler->read($testSessionId);
            if ($readResult === $testData) {
                echo "✓ Session read test passed\n";
            } else {
                echo "❌ Session read test failed (data mismatch)\n";
            }
            
            // Cleanup
            $handler->destroy($testSessionId);
            echo "✓ Session destroy test passed\n";
        } else {
            echo "❌ Session write test failed\n";
        }
    } catch (Exception $e) {
        echo "❌ ERROR testing session handler: " . $e->getMessage() . "\n";
        exit(1);
    }
    
    echo "\n=== All checks passed! Database session handler is ready to use. ===\n";
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    exit(1);
}
