/**
 * Service Worker for Personal Expenses PWA
 * Minimal service worker to enable PWA installation
 * 
 * Note: Update CACHE_NAME when shortcuts change to force service worker update
 */

// Cache name includes timestamp to force update when shortcuts change
// This will be updated dynamically by JavaScript when shortcuts change
const CACHE_NAME = 'personal-expenses-v' + (self.registration ? self.registration.scope : '1');
const urlsToCache = [
    '/',
    '/index.html',
    '/assets/css/style.css',
    '/assets/js/api.js',
    '/assets/js/app.js',
    '/Logo.png'
];

// Install event - cache resources
self.addEventListener('install', (event) => {
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then((cache) => {
                return cache.addAll(urlsToCache);
            })
            .catch((error) => {
                console.log('Cache install failed:', error);
            })
    );
    self.skipWaiting(); // Activate immediately
});

// Activate event - clean up old caches
self.addEventListener('activate', (event) => {
    event.waitUntil(
        caches.keys().then((cacheNames) => {
            return Promise.all(
                cacheNames.map((cacheName) => {
                    if (cacheName !== CACHE_NAME) {
                        return caches.delete(cacheName);
                    }
                })
            );
        })
    );
    return self.clients.claim(); // Take control of all pages immediately
});

// Fetch event - network first, fallback to cache
self.addEventListener('fetch', (event) => {
    // Skip non-GET requests
    if (event.request.method !== 'GET') {
        return;
    }
    
    // Skip API requests (they need to be fresh)
    if (event.request.url.includes('/api/')) {
        return;
    }
    
    // Never cache manifest.php - always fetch fresh to get updated shortcuts
    if (event.request.url.includes('/manifest.php')) {
        event.respondWith(
            fetch(event.request, { cache: 'no-store' })
        );
        return;
    }
    
    event.respondWith(
        fetch(event.request)
            .then((response) => {
                // Clone the response
                const responseToCache = response.clone();
                
                // Cache successful responses
                if (response.status === 200) {
                    caches.open(CACHE_NAME).then((cache) => {
                        cache.put(event.request, responseToCache);
                    });
                }
                
                return response;
            })
            .catch(() => {
                // Network failed, try cache
                return caches.match(event.request);
            })
    );
});
