<?php
/**
 * Dynamic Web App Manifest Generator
 * Generates manifest.json with shortcuts for the first two fidelity cards
 * 
 * Note: Android shortcuts are updated when the PWA is reinstalled/updated.
 * The manifest is regenerated on each request with the current top 2 fidelity cards
 * based on display_order, click_count, and name.
 * 
 * IMPORTANT: This manifest must be accessible without authentication for Chrome
 * to be able to read it during PWA installation.
 */

ob_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';

ini_set('display_errors', 0);
ob_clean();

// Note: config.php might have already started the session
// We need to check if session is already active before trying to start it

header('Content-Type: application/manifest+json');
// No cache for manifest - always fetch fresh to get updated shortcuts
// This ensures shortcuts are always up-to-date when user is authenticated
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Base manifest (always returned, even without auth)
// Add version/timestamp to force manifest refresh when shortcuts change
// Use a hash of the top 2 card IDs to change version when order changes
$manifestVersion = 0;
if (isset($_SESSION['user_id']) || (isset($_GET['token']) && !empty($_GET['token']))) {
    // Calculate version based on top 2 cards to force refresh when order changes
    try {
        $temp_user_id = null;
        if (isset($_SESSION['user_id'])) {
            $temp_user_id = $_SESSION['user_id'];
        } else if (isset($_GET['token'])) {
            // Extract user_id from token (we'll verify it later)
            $token = urldecode($_GET['token']);
            $token_data = base64_decode($token, true);
            if ($token_data !== false) {
                $parts = explode(':', $token_data);
                if (count($parts) === 3) {
                    $temp_user_id = intval($parts[0]);
                }
            }
        }
        
        if ($temp_user_id) {
            $db = new Database();
            $conn = $db->getConnection();
            $query = "
                SELECT fc.id
                FROM fidelity_cards fc
                LEFT JOIN user_fidelity_card_order ufco ON ufco.user_id = :user_id_owned_order AND ufco.fidelity_card_id = fc.id
                LEFT JOIN user_fidelity_card_clicks ufcc ON ufcc.user_id = :user_id_owned_clicks AND ufcc.fidelity_card_id = fc.id
                WHERE fc.user_id = :user_id_owned
                
                UNION ALL
                
                SELECT fc.id
                FROM fidelity_card_shares fcs
                JOIN fidelity_cards fc ON fcs.fidelity_card_id = fc.id
                LEFT JOIN user_fidelity_card_order ufco ON ufco.user_id = :user_id_shared_order AND ufco.fidelity_card_id = fc.id
                LEFT JOIN user_fidelity_card_clicks ufcc ON ufcc.user_id = :user_id_shared_clicks AND ufcc.fidelity_card_id = fc.id
                WHERE fcs.shared_with_user_id = :user_id_shared
                
                ORDER BY COALESCE(ufco.display_order, 999999) ASC, COALESCE(ufcc.click_count, 0) DESC, fc.name ASC
                LIMIT 2
            ";
            $stmt = $conn->prepare($query);
            $stmt->execute([
                ':user_id_owned' => $temp_user_id,
                ':user_id_owned_order' => $temp_user_id,
                ':user_id_owned_clicks' => $temp_user_id,
                ':user_id_shared' => $temp_user_id,
                ':user_id_shared_order' => $temp_user_id,
                ':user_id_shared_clicks' => $temp_user_id
            ]);
            $topCards = $stmt->fetchAll();
            if (count($topCards) >= 2) {
                // Create version hash from top 2 card IDs
                $manifestVersion = crc32($topCards[0]['id'] . ':' . $topCards[1]['id']);
            } else if (count($topCards) === 1) {
                $manifestVersion = crc32($topCards[0]['id'] . ':');
            }
        }
    } catch (Exception $e) {
        // If error, use timestamp as fallback
        $manifestVersion = time();
    }
}

$manifest = [
    'name' => 'Personal Expenses',
    'short_name' => 'Personal Expenses',
    'description' => 'Gestisci le tue spese personali',
    'start_url' => '/',
    'scope' => '/',
    'display' => 'standalone',
    'background_color' => '#ffffff',
    'theme_color' => '#667eea',
    'orientation' => 'portrait',
    'id' => '/?v=' . $manifestVersion, // Add version to force refresh when shortcuts change
    'icons' => [
        [
            'src' => '/Logo.png',
            'sizes' => '192x192',
            'type' => 'image/png',
            'purpose' => 'any maskable'
        ],
        [
            'src' => '/Logo.png',
            'sizes' => '512x512',
            'type' => 'image/png',
            'purpose' => 'any maskable'
        ]
    ]
];

// Try to get shortcuts if user is authenticated (optional)
// Note: config.php has already started the session, so we can directly check $_SESSION
// Also check for token in URL (for when cookies aren't sent)
try {
    // Debug: log session status (only in debug mode)
    // Also enable debug if token is present (to troubleshoot token issues)
    $debug = (isset($_GET['debug']) && $_GET['debug'] === '1') || isset($_GET['token']);
    if ($debug) {
        error_log('Manifest debug - Session status: ' . session_status());
        error_log('Manifest debug - Session ID: ' . (session_id() ?: 'none'));
        error_log('Manifest debug - Session name: ' . session_name());
        error_log('Manifest debug - Cookie session ID: ' . ($_COOKIE[session_name()] ?? 'none'));
        error_log('Manifest debug - Has user_id: ' . (isset($_SESSION['user_id']) ? 'yes' : 'no'));
        error_log('Manifest debug - User ID: ' . ($_SESSION['user_id'] ?? 'none'));
        error_log('Manifest debug - Token in URL: ' . (isset($_GET['token']) ? 'yes' : 'no'));
        error_log('Manifest debug - All session keys: ' . (isset($_SESSION) ? implode(', ', array_keys($_SESSION)) : 'none'));
    }
    
    $user_id = null;
    
    // First, try to get user_id from session (normal case - when user is browsing)
    if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
        $user_id = $_SESSION['user_id'];
    }
    // If no session, try to get user_id from secure token in URL (for Chrome manifest reading)
    // This happens when Chrome reads the manifest without sending cookies
    // SECURITY: We use a secure token instead of user_id directly to prevent unauthorized access
    // Note: The 'v' parameter is a hash of top 2 card IDs to force Chrome to reload manifest when shortcuts change
    else if (isset($_GET['token']) && !empty($_GET['token'])) {
        $token = urldecode($_GET['token']); // Decode URL encoding
        
        // Verify token is valid and extract user_id
        // Token format: base64(user_id:timestamp:hmac)
        // HMAC is calculated as: hash_hmac('sha256', user_id . timestamp, MANIFEST_TOKEN_SECRET)
        try {
            $token_data = base64_decode($token, true);
            if ($token_data === false) {
                if ($debug) {
                    error_log('Manifest debug - Token base64 decode failed');
                }
                throw new Exception('Invalid token format');
            }
            
            $parts = explode(':', $token_data);
            if (count($parts) !== 3) {
                if ($debug) {
                    error_log('Manifest debug - Token structure invalid, parts count: ' . count($parts));
                }
                throw new Exception('Invalid token structure');
            }
            
            list($token_user_id, $token_timestamp, $token_hmac) = $parts;
            
            if ($debug) {
                error_log('Manifest debug - Token parts: user_id=' . $token_user_id . ', timestamp=' . $token_timestamp);
            }
            
            // Verify token hasn't expired (24 hours)
            $token_age = time() - intval($token_timestamp);
            if ($token_age > 86400) { // 24 hours
                if ($debug) {
                    error_log('Manifest debug - Token expired, age: ' . $token_age . ' seconds');
                }
                throw new Exception('Token expired');
            }
            
            // Verify HMAC
            if (!defined('MANIFEST_TOKEN_SECRET') || MANIFEST_TOKEN_SECRET === 'your-secret-key-change-this-in-production') {
                throw new Exception('Manifest token secret not configured');
            }
            
            $expected_hmac = hash_hmac('sha256', $token_user_id . ':' . $token_timestamp, MANIFEST_TOKEN_SECRET);
            if ($debug) {
                error_log('Manifest debug - Expected HMAC: ' . $expected_hmac);
                error_log('Manifest debug - Token HMAC: ' . $token_hmac);
            }
            
            if (!hash_equals($expected_hmac, $token_hmac)) {
                if ($debug) {
                    error_log('Manifest debug - HMAC verification failed');
                }
                throw new Exception('Invalid token signature');
            }
            
            // Token is valid, get user_id
            $uid_param = intval($token_user_id);
            if ($uid_param > 0) {
                // Verify user exists and is not blocked
                $db = new Database();
                $conn = $db->getConnection();
                $stmt = $conn->prepare("SELECT id, is_blocked FROM users WHERE id = :user_id");
                $stmt->execute([':user_id' => $uid_param]);
                $user = $stmt->fetch();
                
                if ($user && !$user['is_blocked']) {
                    $user_id = $user['id'];
                    if ($debug) {
                        error_log('Manifest debug - User ID from secure token: ' . $user_id);
                    }
                } else {
                    if ($debug) {
                        error_log('Manifest debug - Invalid or blocked user ID from token: ' . $uid_param);
                    }
                }
            }
        } catch (Exception $e) {
            if ($debug) {
                error_log('Manifest debug - Token verification failed: ' . $e->getMessage());
                error_log('Manifest debug - Token: ' . substr($token, 0, 50) . '...');
            }
            // Invalid token, don't set user_id
        }
    }
    
    // Check if user is authenticated (without exiting on failure)
    if ($user_id) {
        try {
            // Verify user is not blocked
            $db = new Database();
            $conn = $db->getConnection();
            $stmt = $conn->prepare("SELECT is_blocked FROM users WHERE id = :user_id");
            $stmt->execute([':user_id' => $user_id]);
            $user = $stmt->fetch();
            
            if ($user && $user['is_blocked']) {
                // User is blocked, skip shortcuts
                throw new Exception('User account is blocked');
            }
            
            // User is authenticated and not blocked, proceed with shortcuts
            
            // Get first two fidelity cards (same query as index.php but limited to 2)
        $query = "
            SELECT 
                fc.*,
                NULL as sharing_mode,
                NULL as owner_name,
                NULL as owner_email,
                NULL as share_id,
                'owned' as card_status,
                COALESCE(ufco.display_order, 999999) as display_order,
                COALESCE(ufcc.click_count, 0) as click_count
            FROM fidelity_cards fc
            LEFT JOIN user_fidelity_card_order ufco ON ufco.user_id = :user_id_owned_order AND ufco.fidelity_card_id = fc.id
            LEFT JOIN user_fidelity_card_clicks ufcc ON ufcc.user_id = :user_id_owned_clicks AND ufcc.fidelity_card_id = fc.id
            WHERE fc.user_id = :user_id_owned
            
            UNION ALL
            
            SELECT 
                fc.*,
                'read' as sharing_mode,
                owner.name as owner_name,
                owner.email as owner_email,
                fcs.id as share_id,
                'shared' as card_status,
                COALESCE(ufco.display_order, 999999) as display_order,
                COALESCE(ufcc.click_count, 0) as click_count
            FROM fidelity_card_shares fcs
            JOIN fidelity_cards fc ON fcs.fidelity_card_id = fc.id
            JOIN users owner ON fcs.owner_user_id = owner.id
            LEFT JOIN user_fidelity_card_order ufco ON ufco.user_id = :user_id_shared_order AND ufco.fidelity_card_id = fc.id
            LEFT JOIN user_fidelity_card_clicks ufcc ON ufcc.user_id = :user_id_shared_clicks AND ufcc.fidelity_card_id = fc.id
            WHERE fcs.shared_with_user_id = :user_id_shared
            
            ORDER BY display_order ASC, click_count DESC, name ASC
            LIMIT 2
        ";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':user_id_owned' => $user_id,
            ':user_id_owned_order' => $user_id,
            ':user_id_owned_clicks' => $user_id,
            ':user_id_shared' => $user_id,
            ':user_id_shared_order' => $user_id,
            ':user_id_shared_clicks' => $user_id
        ]);
        $cards = $stmt->fetchAll();
        
        // Build shortcuts array
        $shortcuts = [];
        foreach ($cards as $index => $card) {
            // Use APP_URL from config if available, otherwise construct from request
            $baseUrl = defined('APP_URL') ? rtrim(APP_URL, '/') : '';
            if (empty($baseUrl)) {
                $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
                $baseUrl = $protocol . '://' . $host;
            }
            
            $shortcut = [
                'name' => $card['name'],
                'short_name' => mb_substr($card['name'], 0, 12) . (mb_strlen($card['name']) > 12 ? '...' : ''),
                'url' => '/#fidelity-card-' . $card['id'], // Relative URL (better for PWA)
                'description' => $card['store_name'] ?: $card['name']
            ];
            
            // Always add icon (required for shortcuts to appear)
            if (!empty($card['logo_path'])) {
                // Determine image type from extension
                $extension = strtolower(pathinfo($card['logo_path'], PATHINFO_EXTENSION));
                $imageType = 'image/png';
                if ($extension === 'jpg' || $extension === 'jpeg') {
                    $imageType = 'image/jpeg';
                } else if ($extension === 'webp') {
                    $imageType = 'image/webp';
                }
                
                $shortcut['icons'] = [
                    [
                        'src' => '/uploads/' . $card['logo_path'],
                        'sizes' => '192x192',
                        'type' => $imageType,
                        'purpose' => 'any'
                    ]
                ];
            } else {
                // Use default app icon as fallback
                $shortcut['icons'] = [
                    [
                        'src' => '/Logo.png',
                        'sizes' => '192x192',
                        'type' => 'image/png',
                        'purpose' => 'any'
                    ]
                ];
            }
            
            $shortcuts[] = $shortcut;
        }
        
            // Add shortcuts if we have any cards
            if (!empty($shortcuts)) {
                $manifest['shortcuts'] = $shortcuts;
                if ($debug) {
                    error_log('Manifest debug - Added ' . count($shortcuts) . ' shortcuts');
                }
            } else {
                if ($debug) {
                    error_log('Manifest debug - No shortcuts (no cards found)');
                }
            }
        } catch (Exception $e) {
            // If there's an error getting shortcuts, continue without them
            if ($debug) {
                error_log('Manifest shortcuts generation error: ' . $e->getMessage());
            }
        } catch (Error $e) {
            // Handle fatal errors gracefully
            if ($debug) {
                error_log('Manifest shortcuts generation fatal error: ' . $e->getMessage());
            }
        }
    } else {
        // User not authenticated, skip shortcuts (this is normal)
        if ($debug) {
            error_log('Manifest debug - User not authenticated, skipping shortcuts');
        }
    }
    
} catch (Exception $e) {
    // If auth fails or error occurs, continue with basic manifest without shortcuts
    // This is expected when Chrome reads the manifest before user login
    // Don't log expected "not authenticated" errors
    if ($debug || (strpos($e->getMessage(), 'not authenticated') === false && strpos($e->getMessage(), 'blocked') === false)) {
        error_log('Manifest shortcuts generation error: ' . $e->getMessage());
    }
} catch (Error $e) {
    // Handle fatal errors gracefully
    error_log('Manifest generation fatal error: ' . $e->getMessage());
}

// Add debug info to manifest if requested
if (isset($_GET['debug']) && $_GET['debug'] === '1') {
    $manifest['_debug'] = [
        'session_status' => session_status(),
        'has_user_id' => isset($_SESSION['user_id']),
        'user_id' => $_SESSION['user_id'] ?? null,
        'has_shortcuts' => isset($manifest['shortcuts']),
        'shortcuts_count' => isset($manifest['shortcuts']) ? count($manifest['shortcuts']) : 0
    ];
}

// Always output manifest (with or without shortcuts)
echo json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
