-- Personal Expenses Database Schema

-- Users table (linked to Google OAuth)
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    google_id VARCHAR(255) UNIQUE NOT NULL,
    email VARCHAR(255) NOT NULL,
    name VARCHAR(255),
    picture_url VARCHAR(500),
    default_currency VARCHAR(3) DEFAULT 'EUR',
    default_account_id INT,
    default_view VARCHAR(50) DEFAULT 'dashboard',
    is_admin BOOLEAN DEFAULT FALSE,
    is_blocked BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_google_id (google_id),
    INDEX idx_email (email),
    INDEX idx_is_admin (is_admin),
    INDEX idx_is_blocked (is_blocked)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tax deduction categories (system-wide)
CREATE TABLE tax_deduction_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    code VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default tax deduction categories
INSERT INTO tax_deduction_categories (code, name, description) VALUES
('ASILO', 'Asilo', 'Spese per asilo nido'),
('ASSICURAZIONE_VIDA', 'Assicurazione vita', 'Premi assicurazione sulla vita'),
('DONAZIONE', 'Donazione', 'Donazioni a enti benefici'),
('RISPARMIO_ENERGETICO', 'Risparmio energetico', 'Interventi di risparmio energetico'),
('GIARDINO', 'Giardino', 'Spese per giardino e manutenzione verde'),
('ALTRI_REDDITI', 'Altri redditi', 'Altre spese deducibili'),
('RISTRUTTURAZIONE', 'Ristrutturazione edilizia', 'Interventi di ristrutturazione edilizia'),
('SANITA', 'Sanita', 'Spese sanitarie'),
('SICUREZZA', 'Sicurezza', 'Interventi di sicurezza'),
('TASSE', 'Tasse', 'Pagamento tasse');

-- Account types
CREATE TABLE account_types (
    id INT AUTO_INCREMENT PRIMARY KEY,
    code VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    requires_tax_specification BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default account types
INSERT INTO account_types (code, name, requires_tax_specification) VALUES
('BANK_CURRENT', 'Bank Current Account', FALSE),
('DEPOSIT', 'Deposit Account', FALSE),
('SECURITIES', 'Securities Account', FALSE),
('OTHER', 'Other', TRUE);

-- Accounts table
CREATE TABLE accounts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    name VARCHAR(255) NOT NULL,
    type_id INT NOT NULL,
    base_currency VARCHAR(3) NOT NULL,
    opening_date DATE NOT NULL,
    id_number VARCHAR(255),
    expire_date DATE NULL,
    closure_date DATE NULL,
    tax_type ENUM('CURRENT_ACCOUNT_TAX', 'DEPOSIT_ACCOUNT_TAX') NULL,
    logo_path VARCHAR(500) NULL,
    show_in_dashboard BOOLEAN NOT NULL DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (type_id) REFERENCES account_types(id),
    INDEX idx_user_id (user_id),
    INDEX idx_type_id (type_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Prepaid cards table
CREATE TABLE prepaid_cards (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    name VARCHAR(255) NOT NULL,
    card_number VARCHAR(50),
    base_currency VARCHAR(3) NOT NULL DEFAULT 'EUR',
    opening_date DATE NOT NULL,
    expire_date DATE NULL,
    photo_path VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Securities table (for securities accounts)
CREATE TABLE securities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    symbol VARCHAR(50) NOT NULL,
    quantity DECIMAL(15, 4) NOT NULL,
    purchase_price DECIMAL(15, 4) NOT NULL,
    purchase_date DATE NOT NULL,
    currency VARCHAR(3) NOT NULL,
    transaction_type ENUM('PURCHASE', 'SALE') NULL,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE,
    INDEX idx_account_id (account_id),
    INDEX idx_symbol (symbol)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Expenses table
CREATE TABLE expenses (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    account_id INT NOT NULL,
    date DATE NOT NULL,
    description TEXT NOT NULL,
    amount DECIMAL(15, 2) NOT NULL,
    currency VARCHAR(3) NOT NULL,
    is_tax_deductible BOOLEAN DEFAULT FALSE,
    tax_category_id INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE RESTRICT,
    FOREIGN KEY (tax_category_id) REFERENCES tax_deduction_categories(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_account_id (account_id),
    INDEX idx_date (date),
    INDEX idx_tax_category_id (tax_category_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Expense attachments table
CREATE TABLE expense_attachments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    expense_id INT NOT NULL,
    type ENUM('PHOTO', 'DOCUMENT', 'LINK') NOT NULL,
    file_path VARCHAR(500) NULL,
    link_url TEXT NULL,
    link_text TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (expense_id) REFERENCES expenses(id) ON DELETE CASCADE,
    INDEX idx_expense_id (expense_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Securities attachments table
CREATE TABLE securities_attachments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    security_id INT NOT NULL,
    type ENUM('PHOTO', 'DOCUMENT', 'LINK') NOT NULL,
    file_path VARCHAR(500) NULL,
    link_url TEXT NULL,
    link_text TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (security_id) REFERENCES securities(id) ON DELETE CASCADE,
    INDEX idx_security_id (security_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Fund transfers table
CREATE TABLE fund_transfers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    from_account_id INT NOT NULL,
    to_account_id INT NOT NULL,
    amount DECIMAL(15, 2) NOT NULL,
    from_currency VARCHAR(3) NOT NULL,
    to_currency VARCHAR(3) NOT NULL,
    exchange_rate DECIMAL(15, 6) NOT NULL,
    converted_amount DECIMAL(15, 2) NOT NULL,
    transfer_date DATE NOT NULL,
    arrival_date DATE NULL,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (from_account_id) REFERENCES accounts(id) ON DELETE RESTRICT,
    FOREIGN KEY (to_account_id) REFERENCES accounts(id) ON DELETE RESTRICT,
    INDEX idx_user_id (user_id),
    INDEX idx_from_account_id (from_account_id),
    INDEX idx_to_account_id (to_account_id),
    INDEX idx_transfer_date (transfer_date),
    INDEX idx_arrival_date (arrival_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Transfer attachments table
CREATE TABLE transfer_attachments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    transfer_id INT NOT NULL,
    type ENUM('PHOTO', 'DOCUMENT', 'LINK') NOT NULL,
    file_path VARCHAR(500) NULL,
    link_url TEXT NULL,
    link_text TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (transfer_id) REFERENCES fund_transfers(id) ON DELETE CASCADE,
    INDEX idx_transfer_id (transfer_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Account balances (calculated view - could be a materialized table for performance)
-- This would be calculated from expenses and transfers
-- For now, we'll calculate it on the fly, but could add a balances table for caching

