/**
 * Main Application Logic
 */

let currentUser = null;

// Session Debug Configuration
// Enable/disable session debug logging in browser console
// Can be toggled via: localStorage.setItem('sessionDebugEnabled', 'true')
const SESSION_DEBUG_ENABLED = localStorage.getItem('sessionDebugEnabled') === 'true';

/**
 * Log session debug information to console
 * @param {string} event Event name
 * @param {object} data Additional data
 */
function logSessionDebug(event, data = {}) {
    if (!SESSION_DEBUG_ENABLED) {
        return;
    }
    
    const timestamp = new Date().toISOString();
    const logData = {
        timestamp,
        event,
        ...data
    };
    
    console.log('[SESSION DEBUG]', logData);
    
    // Store in localStorage for persistence across page reloads
    try {
        const logs = JSON.parse(localStorage.getItem('sessionDebugLogs') || '[]');
        logs.push(logData);
        // Keep only last 100 entries
        if (logs.length > 100) {
            logs.shift();
        }
        localStorage.setItem('sessionDebugLogs', JSON.stringify(logs));
    } catch (e) {
        console.error('Failed to store session debug log:', e);
    }
}

// Initialize app
document.addEventListener('DOMContentLoaded', async () => {
    await checkAuth();
    setupEventListeners();
    
    // Handle browser back button for modal
    // Track if modal is open to intercept back button
    let isFidelityCardModalOpen = false;
    let isPrepaidCardModalOpen = false;
    
    window.addEventListener('popstate', (e) => {
        const modalOverlay = document.getElementById('modal-overlay');
        if (modalOverlay && !modalOverlay.classList.contains('hidden')) {
            // Check if it's a fullscreen fidelity card modal
            const modalContent = document.querySelector('.modal-content');
            if (modalContent && modalContent.classList.contains('fullscreen')) {
                // User pressed back, browser already went back in history
                // Close the modal
                isFidelityCardModalOpen = false;
                window.fidelityCardClosingFromBackButton = true;
                window.fidelityCardHistoryState = false;
                closeModal();
                
                // Push forward to cancel the back navigation and stay on current page
                // This prevents the app from navigating away when user presses back
                setTimeout(() => {
                    if (window.history && window.history.pushState) {
                        window.history.pushState(null, '');
                    }
                }, 0);
            }
        }
    });
    
    // Handle hash changes (for Android shortcuts)
    window.addEventListener('hashchange', () => {
        if (currentUser) {
            handleUrlHash();
        }
    });
});

async function checkAuth() {
    logSessionDebug('checkAuth_called', {
        timestamp: new Date().toISOString(),
        has_current_user: !!currentUser,
        url: window.location.href
    });
    
    try {
        const session = await API.checkSession();
        logSessionDebug('checkSession_response', {
            authenticated: session.authenticated,
            has_user_id: !!session.user_id,
            user_id: session.user_id,
            is_admin: session.is_admin
        });
        
        if (session.authenticated) {
            currentUser = session;
            logSessionDebug('checkAuth_success', {
                user_id: session.user_id,
                email: session.email,
                name: session.name,
                is_admin: session.is_admin
            });
            showMainApp();
            // Show/hide admin menu based on user role
            updateAdminMenuVisibility();
            
            // Update manifest link with user ID so Chrome can read shortcuts
            updateManifestLink(session.user_id);
            
            // Update service worker to refresh manifest with shortcuts
            updateServiceWorkerForShortcuts();
            
            // Get user's default view preference
            let defaultView = 'dashboard';
            try {
                const profile = await API.getProfile();
                defaultView = profile.default_view || 'dashboard';
            } catch (error) {
                console.error('Error loading profile for default view:', error);
            }
            
            // Restore last viewed tab or use user's default view preference
            const lastView = localStorage.getItem('lastView') || defaultView;
            switchView(lastView, false); // false = don't save to localStorage (already saved)
            
            // Handle URL hash for shortcuts (e.g., #fidelity-card-123)
            handleUrlHash();
        } else {
            logSessionDebug('checkAuth_failed_not_authenticated', {
                reason: 'Session not authenticated',
                response: session
            });
            showLoginScreen();
        }
    } catch (error) {
        logSessionDebug('checkAuth_error', {
            error: error.message,
            error_stack: error.stack,
            error_name: error.name
        });
        showLoginScreen();
    }
}

function showLoginScreen() {
    document.getElementById('login-screen').classList.remove('hidden');
    document.getElementById('main-app').classList.add('hidden');
    
    // Reset manifest link when user logs out
    const manifestLink = document.querySelector('link[rel="manifest"]');
    if (manifestLink) {
        manifestLink.href = '/manifest.php';
    }
}

function showMainApp() {
    document.getElementById('login-screen').classList.add('hidden');
    document.getElementById('main-app').classList.remove('hidden');
    if (currentUser) {
        const userNameElement = document.getElementById('user-name');
        if (userNameElement) {
            // Set text content directly - badge is now a sibling, not a child
            userNameElement.textContent = currentUser.name || currentUser.email;
        }
        
        // Update admin menu visibility
        updateAdminMenuVisibility();
        
        // Load notifications
        loadNotifications();
        // Refresh notifications every 30 seconds
        setInterval(loadNotifications, 30000);
    }
}

function updateAdminMenuVisibility() {
    const adminMenuContainer = document.getElementById('user-menu-admin-container');
    if (adminMenuContainer && currentUser) {
        const shouldShow = currentUser.is_admin === true || currentUser.is_admin === 1 || currentUser.is_admin === '1';
        adminMenuContainer.style.display = shouldShow ? 'block' : 'none';
        console.log('Admin menu visibility:', shouldShow, 'is_admin:', currentUser.is_admin);
    } else {
        console.log('Admin menu container not found or currentUser not set');
    }
}

function setupEventListeners() {
    // Google login
    const googleLoginBtn = document.getElementById('google-login-btn');
    if (googleLoginBtn) {
        googleLoginBtn.addEventListener('click', () => {
            window.location.href = '/api/auth/google_login.php';
        });
    }


    // Navigation
    document.querySelectorAll('.nav-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const view = btn.dataset.view;
            switchView(view);
        });
    });

    // Swipe navigation for mobile devices
    setupSwipeNavigation();

    // User menu
    setupUserMenuListeners();
    
    // App title click handler
    setupAppTitleClick();
    
    // Notifications
    setupNotificationsListeners();

    // Modal close
    const modalClose = document.querySelector('.modal-close');
    if (modalClose) {
        modalClose.addEventListener('click', closeModal);
    }
    
    // Close modal on ESC key press
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' || e.keyCode === 27) {
            const modalOverlay = document.getElementById('modal-overlay');
            if (modalOverlay && !modalOverlay.classList.contains('hidden')) {
                closeModal();
            }
        }
    });
    
    // Prevent closing modal when clicking outside - user must use Cancel button or close button
    // document.getElementById('modal-overlay').addEventListener('click', (e) => {
    //     if (e.target.id === 'modal-overlay') {
    //         closeModal();
    //     }
    // });

    // Add buttons (only exist in main-app, not in login screen)
    const addAccountBtn = document.getElementById('add-account-btn');
    if (addAccountBtn) {
        addAccountBtn.addEventListener('click', () => showAddAccountModal());
    }
    
    const addExpenseBtn = document.getElementById('add-expense-btn');
    if (addExpenseBtn) {
        addExpenseBtn.addEventListener('click', () => showAddExpenseModal());
    }
    
    const addSecurityBtn = document.getElementById('add-security-btn');
    if (addSecurityBtn) {
        addSecurityBtn.addEventListener('click', () => showAddSecurityModal());
    }
    
    const sellSecurityBtn = document.getElementById('sell-security-btn');
    if (sellSecurityBtn) {
        sellSecurityBtn.addEventListener('click', () => showSellSecurityModal());
    }
    
    const addCardBtn = document.getElementById('add-card-btn');
    if (addCardBtn) {
        addCardBtn.addEventListener('click', () => showAddPrepaidCardModal());
    }
    
    const showArchivedCardsBtn = document.getElementById('show-archived-cards-btn');
    if (showArchivedCardsBtn) {
        showArchivedCardsBtn.addEventListener('click', () => {
            const isShowingArchived = showArchivedCardsBtn.textContent.includes('Torna');
            if (isShowingArchived) {
                // Return to active cards
                loadPrepaidCards();
                showArchivedCardsBtn.textContent = 'Carte Archiviate';
            } else {
                // Show archived cards
                loadArchivedPrepaidCards();
                showArchivedCardsBtn.textContent = 'Torna a Carte Attive';
            }
        });
    }
    
    const addFidelityCardBtn = document.getElementById('add-fidelity-card-btn');
    if (addFidelityCardBtn) {
        addFidelityCardBtn.addEventListener('click', () => showAddFidelityCardModal());
    }
    
    const addBrandBtn = document.getElementById('add-brand-btn');
    if (addBrandBtn) {
        addBrandBtn.addEventListener('click', () => showAddBrandModal());
    }
    
    // Expense filters
    const applyFiltersBtn = document.getElementById('apply-filters-btn');
    const clearFiltersBtn = document.getElementById('clear-filters-btn');
    if (applyFiltersBtn) {
        applyFiltersBtn.addEventListener('click', async () => {
            // Clear dashboard filter when manually applying filters
            window.currentExpenseFilter = null;
            // Disable button during loading
            applyFiltersBtn.disabled = true;
            applyFiltersBtn.textContent = 'Caricamento...';
            try {
                await loadExpenses();
            } finally {
                // Re-enable button
                applyFiltersBtn.disabled = false;
                applyFiltersBtn.textContent = 'Applica Filtri';
            }
        });
    }
    if (clearFiltersBtn) {
        clearFiltersBtn.addEventListener('click', async () => {
            clearExpenseFilters();
            // Disable button during loading
            clearFiltersBtn.disabled = true;
            clearFiltersBtn.textContent = 'Caricamento...';
            try {
                await loadExpenses();
            } finally {
                // Re-enable button
                clearFiltersBtn.disabled = false;
                clearFiltersBtn.textContent = 'Pulisci Filtri';
            }
        });
    }
    
    // Date preset dropdown
    const datePresetSelect = document.getElementById('filter-date-preset');
    if (datePresetSelect) {
        datePresetSelect.addEventListener('change', (e) => {
            const preset = e.target.value;
            if (preset) {
                applyDatePreset(preset);
            }
        });
    }
    
    // Clear preset dropdown when dates are manually changed
    const startDateInput = document.getElementById('filter-start-date');
    const endDateInput = document.getElementById('filter-end-date');
    if (startDateInput && datePresetSelect) {
        startDateInput.addEventListener('change', () => {
            datePresetSelect.value = '';
        });
    }
    if (endDateInput && datePresetSelect) {
        endDateInput.addEventListener('change', () => {
            datePresetSelect.value = '';
        });
    }
}

function switchView(viewName, saveToStorage = true) {
    // Save current view to localStorage (unless called during initial load)
    if (saveToStorage) {
        localStorage.setItem('lastView', viewName);
    }
    
    // Update nav buttons
    document.querySelectorAll('.nav-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.view === viewName);
    });

    // Update views
    document.querySelectorAll('.view').forEach(view => {
        view.classList.toggle('active', view.id === `view-${viewName}`);
    });

    // Load view data
    switch(viewName) {
        case 'dashboard':
            loadDashboard();
            break;
        case 'accounts':
            loadAccounts();
            break;
        case 'expenses':
            // Load filters first, then expenses (to ensure filters UI is ready)
            loadExpensesFilters().then(() => {
                loadExpenses();
            });
            break;
        case 'securities':
            loadSecurities();
            break;
        case 'prepaid-cards':
            loadPrepaidCards();
            break;
        case 'fidelity-cards':
            loadFidelityCards();
            break;
        case 'profile':
            loadProfile();
            break;
        case 'users':
            loadUsers();
            break;
        case 'notifications':
            loadNotificationsPage();
            break;
        case 'brands':
            loadBrands();
            break;
    }
}

/**
 * Setup swipe navigation for mobile devices
 * Allows swiping left/right to navigate between tabs
 */
function setupSwipeNavigation() {
    // Only enable on touch devices
    if (!('ontouchstart' in window || navigator.maxTouchPoints > 0)) {
        return;
    }

    const mainApp = document.getElementById('main-app');
    const appContent = document.querySelector('.app-content');
    
    if (!mainApp || !appContent) {
        return;
    }

    let touchStartX = null;
    let touchStartY = null;
    let touchEndX = null;
    let touchEndY = null;
    const minSwipeDistance = 50; // Minimum distance in pixels for a swipe
    const maxVerticalDistance = 100; // Maximum vertical movement to consider it a horizontal swipe

    /**
     * Get list of visible navigation views in order
     * @returns {string[]} Array of view names in order
     */
    function getVisibleViews() {
        const navButtons = Array.from(document.querySelectorAll('.nav-btn'))
            .filter(btn => {
                // Only include visible buttons (not hidden by display: none)
                const style = window.getComputedStyle(btn);
                return style.display !== 'none';
            })
            .map(btn => btn.dataset.view);
        return navButtons;
    }

    /**
     * Get the current active view name
     * @returns {string|null} Current view name or null
     */
    function getCurrentView() {
        const activeBtn = document.querySelector('.nav-btn.active');
        return activeBtn ? activeBtn.dataset.view : null;
    }

    /**
     * Navigate to next or previous view based on swipe direction
     * @param {string} direction 'left' or 'right'
     */
    function navigateSwipe(direction) {
        const views = getVisibleViews();
        const currentView = getCurrentView();
        
        if (!currentView || views.length === 0) {
            return;
        }

        const currentIndex = views.indexOf(currentView);
        if (currentIndex === -1) {
            return;
        }

        let targetIndex;
        if (direction === 'left') {
            // Swipe left = go to next view
            targetIndex = (currentIndex + 1) % views.length;
        } else {
            // Swipe right = go to previous view
            targetIndex = (currentIndex - 1 + views.length) % views.length;
        }

        const targetView = views[targetIndex];
        if (targetView) {
            switchView(targetView);
        }
    }

    // Prevent swipe when user is interacting with scrollable content
    function isScrollableElement(element) {
        if (!element) return false;
        
        // Check for specific scrollable containers
        if (element.classList.contains('summary-container') || 
            element.classList.contains('list-item-actions') ||
            element.classList.contains('notifications-dropdown') ||
            element.classList.contains('notifications-list') ||
            element.classList.contains('summary-table') ||
            element.tagName === 'TABLE' ||
            element.tagName === 'TBODY' ||
            element.tagName === 'THEAD') {
            // Check parent for scrollable container
            let parent = element.parentElement;
            while (parent && parent !== appContent) {
                if (parent.classList.contains('summary-container') || 
                    parent.classList.contains('list-item-actions')) {
                    return true;
                }
                parent = parent.parentElement;
            }
        }
        
        const style = window.getComputedStyle(element);
        const overflowY = style.overflowY;
        const overflowX = style.overflowX;
        
        // Check if element is scrollable horizontally
        if (overflowX === 'auto' || overflowX === 'scroll') {
            if (element.scrollWidth > element.clientWidth) {
                return true;
            }
        }
        
        // Check if element is scrollable vertically
        if (overflowY === 'auto' || overflowY === 'scroll') {
            if (element.scrollHeight > element.clientHeight) {
                return true;
            }
        }
        
        return false;
    }

    // Check if a modal is open
    function isModalOpen() {
        const modalOverlay = document.getElementById('modal-overlay');
        return modalOverlay && !modalOverlay.classList.contains('hidden');
    }

    let isScrollingHorizontally = false;
    let scrollStartX = null;
    let scrollStartY = null;

    appContent.addEventListener('touchstart', (e) => {
        // Don't handle swipe if modal is open
        if (isModalOpen()) {
            return;
        }

        // Check if user is interacting with scrollable content
        let target = e.target;
        let scrollableElement = null;
        while (target && target !== appContent) {
            if (isScrollableElement(target)) {
                scrollableElement = target;
                break;
            }
            target = target.parentElement;
        }

        if (scrollableElement) {
            // Check if element can scroll horizontally
            const canScrollX = scrollableElement.scrollWidth > scrollableElement.clientWidth;
            if (canScrollX) {
                scrollStartX = scrollableElement.scrollLeft;
                scrollStartY = scrollableElement.scrollTop;
                isScrollingHorizontally = false;
            }
            return;
        }

        touchStartX = e.touches[0].clientX;
        touchStartY = e.touches[0].clientY;
        isScrollingHorizontally = false;
    }, { passive: true });

    appContent.addEventListener('touchmove', (e) => {
        // Check if we're tracking a scrollable element
        if (scrollStartX !== null) {
            let target = e.target;
            let scrollableElement = null;
            while (target && target !== appContent) {
                if (isScrollableElement(target)) {
                    scrollableElement = target;
                    break;
                }
                target = target.parentElement;
            }

            if (scrollableElement) {
                const currentScrollX = scrollableElement.scrollLeft;
                const currentScrollY = scrollableElement.scrollTop;
                
                // If horizontal scroll changed, user is scrolling horizontally
                if (Math.abs(currentScrollX - scrollStartX) > 5) {
                    isScrollingHorizontally = true;
                }
                
                // If vertical scroll changed significantly, reset horizontal scroll tracking
                if (Math.abs(currentScrollY - scrollStartY) > 10) {
                    scrollStartX = null;
                    scrollStartY = null;
                }
            }
        }

        // Update touch end position during move
        if (touchStartX !== null) {
            touchEndX = e.touches[0].clientX;
            touchEndY = e.touches[0].clientY;
        }
    }, { passive: true });

    appContent.addEventListener('touchend', (e) => {
        // Reset scroll tracking
        scrollStartX = null;
        scrollStartY = null;

        if (touchStartX === null || touchEndX === null) {
            return;
        }

        // Don't handle swipe if modal is open
        if (isModalOpen()) {
            touchStartX = null;
            touchEndX = null;
            touchEndY = null;
            isScrollingHorizontally = false;
            return;
        }

        // Don't handle swipe if user was scrolling horizontally
        if (isScrollingHorizontally) {
            touchStartX = null;
            touchEndX = null;
            touchEndY = null;
            isScrollingHorizontally = false;
            return;
        }

        const deltaX = touchEndX - touchStartX;
        const deltaY = touchEndY !== null ? Math.abs(touchEndY - touchStartY) : 0;

        // Reset touch positions
        touchStartX = null;
        touchEndX = null;
        touchEndY = null;
        isScrollingHorizontally = false;

        // Check if it's a horizontal swipe (not vertical scroll)
        if (deltaY > maxVerticalDistance) {
            return; // Too much vertical movement, probably scrolling
        }

        // Check if swipe distance is sufficient
        if (Math.abs(deltaX) < minSwipeDistance) {
            return; // Swipe too short
        }

        // Determine swipe direction
        if (deltaX > 0) {
            // Swipe right = go to previous view
            navigateSwipe('right');
        } else {
            // Swipe left = go to next view
            navigateSwipe('left');
        }
    }, { passive: true });
}

// Dashboard
async function loadDashboard() {
    try {
        const accounts = await API.getAccountTotals();
        const container = document.getElementById('account-totals');
        
        // Check if container exists
        if (!container) {
            console.error('Dashboard container not found');
            return;
        }
        
        // Check if API returned an error object instead of an array
        if (accounts && typeof accounts === 'object' && accounts.error && !Array.isArray(accounts)) {
            throw new Error(accounts.error || 'Errore nel recupero degli account');
        }
        
        // Check if accounts is valid and not empty
        if (!accounts || !Array.isArray(accounts) || accounts.length === 0) {
            console.log('No accounts found, showing empty state');
            container.style.display = 'block';
            container.innerHTML = `
                <div style="text-align: center; padding: 3rem 1rem; color: #666; grid-column: 1 / -1; width: 100%;">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">📊</div>
                    <h3 style="margin-bottom: 0.5rem; color: #333;">Nessun account trovato</h3>
                    <p style="margin-bottom: 2rem; color: #888;">Inizia creando il tuo primo account per gestire le tue spese personali.</p>
                    <button class="btn btn-primary" onclick="switchView('accounts'); setTimeout(() => showAddAccountModal(), 100);">
                        Crea il tuo primo account
                    </button>
                </div>
            `;
            return;
        }
        
        // Reset display style in case it was changed
        container.style.display = '';
        
        console.log(`Loading dashboard with ${accounts.length} account(s)`);
        
        // Process accounts and convert securities currencies
        const processedAccounts = await Promise.all(accounts.map(async (account) => {
            if (account.type_code === 'SECURITIES' && account.securities_by_currency && account.securities_by_currency.length > 0) {
                // Convert securities values to base currency
                let totalBalance = 0;
                const exchangeRates = {}; // Store exchange rates for display
                
                const conversionPromises = account.securities_by_currency.map(async (sec) => {
                    const value = parseFloat(sec.total_value) || 0;
                    if (isNaN(value)) {
                        console.warn(`Invalid total_value for security:`, sec);
                        return { converted: 0, currency: sec.currency };
                    }
                    
                    if (sec.currency === account.base_currency) {
                        return { converted: value, currency: sec.currency };
                    } else {
                        try {
                            const rateData = await API.getExchangeRate(sec.currency, account.base_currency);
                            const rate = parseFloat(rateData.rate) || 1;
                            const converted = value * rate;
                            // Store exchange rate for this currency pair
                            exchangeRates[sec.currency] = rate;
                            return { converted: converted, currency: sec.currency };
                        } catch (error) {
                            console.error(`Error fetching exchange rate ${sec.currency} to ${account.base_currency}:`, error);
                            return { converted: 0, currency: sec.currency };
                        }
                    }
                });
                const conversionResults = await Promise.all(conversionPromises);
                totalBalance = conversionResults.reduce((sum, result) => {
                    const numVal = parseFloat(result.converted) || 0;
                    if (isNaN(numVal)) {
                        console.warn('NaN value in conversion:', result);
                        return sum;
                    }
                    return sum + numVal;
                }, 0);
                account.balance = totalBalance;
                account.exchangeRates = exchangeRates; // Store exchange rates for display
                console.log(`Securities account ${account.name}:`, {
                    securities_by_currency: account.securities_by_currency,
                    conversionResults: conversionResults,
                    totalBalance: totalBalance,
                    exchangeRates: exchangeRates
                });
            } else if (account.type_code === 'SECURITIES') {
                // Securities account with no securities
                account.balance = 0;
            }
            return account;
        }));
        
        // Check if there are no accounts (safety check after processing)
        if (!processedAccounts || processedAccounts.length === 0) {
            console.log('No accounts after processing, showing empty state');
            container.style.display = 'block';
            container.innerHTML = `
                <div style="text-align: center; padding: 3rem 1rem; color: #666; grid-column: 1 / -1; width: 100%;">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">📊</div>
                    <h3 style="margin-bottom: 0.5rem; color: #333;">Nessun account trovato</h3>
                    <p style="margin-bottom: 2rem; color: #888;">Inizia creando il tuo primo account per gestire le tue spese personali.</p>
                    <button class="btn btn-primary" onclick="switchView('accounts'); setTimeout(() => showAddAccountModal(), 100);">
                        Crea il tuo primo account
                    </button>
                </div>
            `;
            return;
        }
        
        // Reset display style in case it was changed
        container.style.display = '';
        
        container.innerHTML = processedAccounts.map(account => {
            const logoStyle = account.logo_path 
                ? `background-image: url('/uploads/${account.logo_path}'); background-size: cover; background-position: center; background-blend-mode: overlay; background-color: rgba(255, 255, 255, 0.5); cursor: pointer;`
                : 'cursor: pointer;';
            
            // Ensure balance is a number
            const balance = parseFloat(account.balance) || 0;
            let balanceDisplay = formatCurrency(balance, account.base_currency);
            let metaDisplay = '';
            
            if (account.type_code === 'SECURITIES') {
                // Show securities breakdown
                if (account.securities_by_currency && account.securities_by_currency.length > 0) {
                    const currencyTotals = account.securities_by_currency.map(sec => 
                        `${formatCurrency(parseFloat(sec.total_value) || 0, sec.currency)}`
                    ).join(' • ');
                    
                    // Add exchange rates line if there are foreign currencies
                    const foreignCurrencies = account.securities_by_currency.filter(sec => 
                        sec.currency !== account.base_currency
                    );
                    
                    if (foreignCurrencies.length > 0 && account.exchangeRates) {
                        const exchangeRatesLine = foreignCurrencies.map(sec => {
                            const rate = account.exchangeRates[sec.currency];
                            if (rate) {
                                return `${sec.currency}/${account.base_currency}: ${parseFloat(rate).toFixed(4)}`;
                            }
                            return null;
                        }).filter(r => r !== null).join(' • ');
                        
                        metaDisplay = `${currencyTotals}<br><span style="font-size: 0.85em; color: #888;">${exchangeRatesLine}</span>`;
                    } else {
                        metaDisplay = currencyTotals;
                    }
                } else {
                    metaDisplay = 'Nessun titolo';
                }
            } else {
                // Helper function to format delta with color
                const formatDelta = (delta) => {
                    const deltaValue = parseFloat(delta) || 0;
                    const sign = deltaValue >= 0 ? '+' : '';
                    const color = deltaValue >= 0 ? '#28a745' : '#dc3545';
                    return `<span style="color: ${color};">${sign}${formatCurrency(Math.abs(deltaValue), account.base_currency)}</span>`;
                };
                
                const lastMonthTotal = parseFloat(account.last_month_total) || 0;
                const lastMonthDelta = parseFloat(account.last_month_delta) || 0;
                const lastSemesterTotal = parseFloat(account.last_semester_total) || 0;
                const lastSemesterDelta = parseFloat(account.last_semester_delta) || 0;
                const lastYearTotal = parseFloat(account.last_year_total) || 0;
                const lastYearDelta = parseFloat(account.last_year_delta) || 0;
                
                metaDisplay = `
                    <div style="font-size: 0.9em; line-height: 1.6;">
                        <div><strong>Ultimo mese:</strong> ${formatCurrency(lastMonthTotal, account.base_currency)} ${formatDelta(lastMonthDelta)}</div>
                        <div><strong>Ultimo semestre:</strong> ${formatCurrency(lastSemesterTotal, account.base_currency)} ${formatDelta(lastSemesterDelta)}</div>
                        <div><strong>Ultimo anno:</strong> ${formatCurrency(lastYearTotal, account.base_currency)} ${formatDelta(lastYearDelta)}</div>
                    </div>
                `;
            }
            
            const isShared = account.account_status === 'shared';
            const isOwned = account.account_status === 'owned';
            const hasShares = account.has_shares === 1 || account.has_shares === true;
            const ownerInfo = account.owner_name || account.owner_email ? `Condiviso da ${account.owner_name || account.owner_email}` : '';
            const sharedIndicator = isShared ? `<span style="font-size: 0.8em; color: #666; margin-left: 0.5rem;">🔗</span>` : '';
            const sharedIconForOwner = isOwned && hasShares ? `<span style="font-size: 0.8em; color: #666; margin-left: 0.5rem;" title="Condivisioni attive">🔗</span>` : '';
            
            return `
            <div class="card" data-account-id="${account.id}" style="${logoStyle}" draggable="true">
                <div class="card-header">
                    <div class="card-title">
                        ${escapeHtml(account.name)}${sharedIndicator}${sharedIconForOwner}
                        ${isShared ? `<div style="font-size: 0.75em; color: #888; margin-top: 0.25rem;">${ownerInfo}</div>` : ''}
                    </div>
                    <span class="card-meta">${account.type_name}</span>
                </div>
                <div class="card-balance" onclick="event.stopPropagation(); window.filterByAccount(${account.id}, '${account.type_code}')">${balanceDisplay}</div>
                <div class="card-meta" onclick="event.stopPropagation(); window.filterByAccount(${account.id}, '${account.type_code}')">${metaDisplay}</div>
            </div>
            `;
        }).join('');
        
        // Initialize drag and drop after rendering
        initializeDragAndDrop();
    } catch (error) {
        console.error('Error loading dashboard:', error);
        console.error('Error stack:', error.stack);
        console.error('Error details:', {
            message: error.message,
            name: error.name,
            accounts: typeof accounts !== 'undefined' ? accounts : 'undefined'
        });
        const container = document.getElementById('account-totals');
        if (container) {
            const errorMessage = error.message || 'Errore sconosciuto';
            container.innerHTML = `
                <div style="text-align: center; padding: 3rem 1rem; color: #d32f2f;">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">⚠️</div>
                    <h3 style="margin-bottom: 0.5rem; color: #333;">Errore nel caricamento</h3>
                    <p style="margin-bottom: 1rem; color: #888;">Si è verificato un errore nel caricamento della dashboard.</p>
                    <p style="margin-bottom: 2rem; color: #999; font-size: 0.9em; font-family: monospace;">${escapeHtml(errorMessage)}</p>
                    <button class="btn btn-primary" onclick="loadDashboard()">Riprova</button>
                </div>
            `;
        }
    }
}

window.filterByAccount = function(accountId, accountTypeCode) {
    if (accountTypeCode === 'SECURITIES') {
        // Switch to securities view and filter by account
        window.currentSecuritiesFilter = { account_id: accountId };
        switchView('securities');
        // loadSecurities() will be called by switchView
    } else {
        // Switch to expenses view and filter by account
        // Set filter BEFORE switching view so loadExpensesFilters() can apply it
        window.currentExpenseFilter = { account_id: accountId };
        switchView('expenses');
        // loadExpenses() will be called by switchView after loadExpensesFilters() completes
    }
}

window.filterByPrepaidCard = function(prepaidCardId) {
    // Switch to expenses view and filter by prepaid card
    // Set filter BEFORE switching view so loadExpensesFilters() can apply it
    window.currentExpenseFilter = { prepaid_card_id: prepaidCardId };
    switchView('expenses');
    // loadExpenses() will be called by switchView after loadExpensesFilters() completes
}

// Keep old function name for backward compatibility
window.filterExpensesByAccount = function(accountId) {
    window.filterByAccount(accountId, null);
}

// Drag and Drop functionality for account cards
let draggedAccountId = null;
let draggedElement = null;
let isSavingOrder = false;

function handleDragStart(event, accountId) {
    draggedAccountId = accountId;
    draggedElement = event.currentTarget;
    event.currentTarget.style.opacity = '0.5';
    event.dataTransfer.effectAllowed = 'move';
    event.dataTransfer.setData('text/html', event.currentTarget.innerHTML);
}

function handleDragOver(event) {
    if (event.preventDefault) {
        event.preventDefault();
    }
    event.dataTransfer.dropEffect = 'move';
    return false;
}

function handleDrop(event, targetAccountId) {
    if (event.stopPropagation) {
        event.stopPropagation();
    }
    
    if (draggedAccountId === null || draggedAccountId === targetAccountId) {
        return false;
    }
    
    const container = document.getElementById('account-totals');
    if (!container) {
        return false;
    }
    
    const cards = Array.from(container.querySelectorAll('.card'));
    const draggedIndex = cards.findIndex(card => card.dataset.accountId == draggedAccountId);
    const targetIndex = cards.findIndex(card => card.dataset.accountId == targetAccountId);
    
    if (draggedIndex === -1 || targetIndex === -1) {
        return false;
    }
    
    // Reorder cards in DOM
    if (draggedIndex < targetIndex) {
        cards[targetIndex].after(draggedElement);
    } else {
        cards[targetIndex].before(draggedElement);
    }
    
    // Save new order
    saveAccountOrder();
    
    return false;
}

function handleDragEnd(event) {
    event.currentTarget.style.opacity = '';
    draggedAccountId = null;
    draggedElement = null;
}

async function saveAccountOrder() {
    // Prevent multiple simultaneous saves
    if (isSavingOrder) {
        console.log('Save already in progress, skipping');
        return;
    }
    
    isSavingOrder = true;
    
    try {
        const container = document.getElementById('account-totals');
        if (!container) {
            console.warn('Account totals container not found');
            return;
        }
        
        const cards = Array.from(container.querySelectorAll('.card'));
        if (cards.length === 0) {
            console.warn('No account cards found to save order');
            return;
        }
        
        const accountOrders = cards.map((card, index) => ({
            account_id: parseInt(card.dataset.accountId),
            display_order: index + 1
        }));
        
        console.log('Saving account order:', accountOrders);
        
        const result = await API.saveAccountOrder(accountOrders);
        console.log('Account order saved successfully:', result);
    } catch (error) {
        console.error('Error saving account order:', error);
        // Reload dashboard to restore original order
        loadDashboard();
    } finally {
        isSavingOrder = false;
    }
}

function initializeDragAndDrop() {
    const container = document.getElementById('account-totals');
    if (!container) {
        console.warn('Account totals container not found for drag and drop');
        return;
    }
    
    // Add drop handler to container (for dropping on empty areas)
    container.addEventListener('dragover', (e) => {
        e.preventDefault();
        e.dataTransfer.dropEffect = 'move';
        
        const afterElement = getDragAfterElement(container, e.clientX);
        const dragging = container.querySelector('.dragging');
        if (dragging) {
            if (afterElement == null) {
                container.appendChild(dragging);
            } else {
                container.insertBefore(dragging, afterElement);
            }
        }
    });
    
    container.addEventListener('drop', async (e) => {
        e.preventDefault();
        e.stopPropagation();
        console.log('Container drop event fired');
        
        if (draggedAccountId === null) {
            console.log('No dragged account ID, ignoring drop');
            return;
        }
        
        // Wait a bit to ensure DOM is updated
        await new Promise(resolve => setTimeout(resolve, 50));
        
        // Save new order
        await saveAccountOrder();
    });
    
    const cards = container.querySelectorAll('.card');
    console.log(`Initializing drag and drop for ${cards.length} cards`);
    
    cards.forEach(card => {
        // Remove old event listeners if any by cloning
        const newCard = card.cloneNode(true);
        card.parentNode.replaceChild(newCard, card);
        
        // Add drag and drop handlers
        newCard.addEventListener('dragstart', (e) => {
            const accountId = parseInt(newCard.dataset.accountId);
            draggedAccountId = accountId;
            draggedElement = newCard;
            newCard.style.opacity = '0.5';
            newCard.classList.add('dragging');
            e.dataTransfer.effectAllowed = 'move';
            e.dataTransfer.setData('text/html', '');
            console.log('Drag started for account:', accountId);
        });
        
        newCard.addEventListener('dragend', async (e) => {
            const wasDragging = draggedAccountId !== null;
            const accountId = draggedAccountId;
            newCard.style.opacity = '';
            newCard.classList.remove('dragging');
            console.log('Drag ended for account:', accountId);
            
            // Reset dragged state immediately
            draggedAccountId = null;
            draggedElement = null;
            
            // Save order when drag ends (drop event might not fire in all browsers)
            // Use a small delay to allow DOM to update
            if (wasDragging) {
                setTimeout(async () => {
                    // Only save if not already saving
                    if (!isSavingOrder) {
                        await saveAccountOrder();
                    }
                }, 150);
            }
        });
        
        newCard.addEventListener('dragover', (e) => {
            e.preventDefault();
            e.stopPropagation();
            e.dataTransfer.dropEffect = 'move';
            
            const afterElement = getDragAfterElement(container, e.clientX);
            const dragging = container.querySelector('.dragging');
            if (dragging && dragging !== newCard) {
                if (afterElement == null) {
                    container.appendChild(dragging);
                } else {
                    container.insertBefore(dragging, afterElement);
                }
            }
        });
        
        newCard.addEventListener('drop', async (e) => {
            e.preventDefault();
            e.stopPropagation();
            console.log('Card drop event fired for account:', newCard.dataset.accountId);
            
            if (draggedAccountId === null || draggedAccountId === parseInt(newCard.dataset.accountId)) {
                console.log('Invalid drop condition, ignoring');
                return;
            }
            
            // Wait a bit to ensure DOM is updated
            await new Promise(resolve => setTimeout(resolve, 50));
            
            // Save new order
            await saveAccountOrder();
        });
    });
}

function getDragAfterElement(container, x) {
    const draggableElements = [...container.querySelectorAll('.card:not(.dragging)')];
    
    return draggableElements.reduce((closest, child) => {
        const box = child.getBoundingClientRect();
        const offset = x - box.left - box.width / 2;
        
        if (offset < 0 && offset > closest.offset) {
            return { offset: offset, element: child };
        } else {
            return closest;
        }
    }, { offset: Number.NEGATIVE_INFINITY }).element;
}

// Accounts
async function loadAccounts() {
    try {
        const accounts = await API.getAccounts();
        const container = document.getElementById('accounts-list');
        
        if (!container) {
            console.error('Accounts list container not found');
            return;
        }
        
        container.innerHTML = accounts.map(account => {
            const isNoAccount = account.name === 'No Account';
            const isShared = account.account_status === 'shared';
            const isOwned = account.account_status === 'owned';
            const sharingMode = account.sharing_mode || null;
            const ownerInfo = account.owner_name || account.owner_email ? `Condiviso da ${account.owner_name || account.owner_email}` : '';
            
            const balanceButton = isNoAccount 
                ? '' 
                : `<button class="btn btn-secondary" onclick="viewAccountBalance(${account.id})">Bilancio</button>`;
            
            const shareButton = isOwned && !isNoAccount && account.name !== 'No Account'
                ? `<button class="btn btn-secondary" onclick="showAccountSharesModal(${account.id})" title="Gestisci condivisioni">🔗 Condividi</button>`
                : '';
            
            const hasShares = account.has_shares === 1 || account.has_shares === true;
            const sharedIcon = isOwned && !isNoAccount && hasShares
                ? `<span class="shared-icon" title="Condivisioni attive" style="margin-right: 0.5rem; font-size: 1.2em;">🔗</span>`
                : '';
            
            const stopSharingButton = isShared
                ? `<button class="btn btn-secondary" onclick="stopSharingAccount(${account.id})">Rimuovi Condivisione</button>`
                : '';
            
            const transferButton = !isNoAccount && (isOwned || sharingMode === 'write')
                ? `<button class="btn btn-secondary" onclick="showAddTransferModal(${account.id})">Trasferisci Fondi</button>`
                : '';
            
            const editButton = isOwned && !isNoAccount
                ? `<button class="btn btn-secondary" onclick="window.showEditAccountModal(${account.id})">Modifica</button>`
                : '';
            
            const showInDashboard = account.show_in_dashboard !== false && account.show_in_dashboard !== 0;
            const dashboardToggle = !isNoAccount
                ? `<label style="display: flex; align-items: center; gap: 8px; cursor: pointer; padding: 0.5rem 1rem; border-radius: 4px; transition: background-color 0.2s;" onmouseover="this.style.backgroundColor='#f0f0f0'" onmouseout="this.style.backgroundColor='transparent'">
                    <input type="checkbox" ${showInDashboard ? 'checked' : ''} onchange="toggleAccountDashboard(${account.id}, this.checked)" style="cursor: pointer;">
                    <span style="font-size: 0.9em; white-space: nowrap;">Dashboard</span>
                  </label>`
                : '';
            
            return `
            <div class="list-item">
                <div class="list-item-content">
                    <div class="list-item-title">
                        ${sharedIcon}${escapeHtml(account.name)}
                        ${isShared ? `<span style="color: #666; font-size: 0.9em; margin-left: 0.5rem;">(${ownerInfo})</span>` : ''}
                    </div>
                    <div class="list-item-meta">
                        ${account.type_name} • ${account.base_currency} • 
                        Aperto: ${formatDate(account.opening_date)}
                        ${isShared && sharingMode ? ` • ${sharingMode === 'write' ? 'Scrittura' : 'Sola lettura'}` : ''}
                    </div>
                </div>
                <div class="list-item-actions">
                    ${dashboardToggle}
                    ${shareButton}
                    ${transferButton}
                    ${balanceButton}
                    ${editButton}
                    ${stopSharingButton}
                </div>
            </div>
            `;
        }).join('');
        
        console.log('Accounts loaded:', accounts.length);
    } catch (error) {
        console.error('Error loading accounts:', error);
    }
}

async function toggleAccountDashboard(accountId, showInDashboard) {
    try {
        await API.toggleAccountDashboardVisibility(accountId, showInDashboard);
        // Reload dashboard to reflect changes
        loadDashboard();
    } catch (error) {
        console.error('Error toggling dashboard visibility:', error);
        alert('Errore: ' + error.message);
        // Reload accounts to reset checkbox state
        loadAccounts();
    }
}

async function showAddAccountModal() {
    try {
        const types = await API.getAccountTypes();
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Aggiungi Account</h2>
            <form id="add-account-form">
                <div class="form-group">
                    <label>Nome</label>
                    <input type="text" name="name" required>
                </div>
                <div class="form-group">
                    <label>Tipo Account</label>
                    <select name="type_id" id="account-type-select" required>
                        <option value="">Seleziona...</option>
                        ${types.map(t => `<option value="${t.id}" data-requires-tax="${t.requires_tax_specification}">${escapeHtml(t.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group" id="tax-type-group" style="display: none;">
                    <label>Tipo Tassa</label>
                    <select name="tax_type">
                        <option value="CURRENT_ACCOUNT_TAX">Tassa Conto Corrente</option>
                        <option value="DEPOSIT_ACCOUNT_TAX">Tassa Conto Deposito (0,2%)</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Valuta Base</label>
                    <input type="text" name="base_currency" value="EUR" maxlength="3" required>
                </div>
                <div class="form-group">
                    <label>Data Apertura</label>
                    <input type="date" name="opening_date" required>
                </div>
                <div class="form-group">
                    <label>Numero ID</label>
                    <input type="text" name="id_number">
                </div>
                <div class="form-group">
                    <label>Data Scadenza</label>
                    <input type="date" name="expire_date">
                </div>
                <div class="form-group">
                    <label>Data Chiusura</label>
                    <input type="date" name="closure_date">
                </div>
                <div class="form-group">
                    <label>Logo</label>
                    <input type="file" name="logo" accept="image/*">
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        document.getElementById('account-type-select').addEventListener('change', (e) => {
            const selected = e.target.options[e.target.selectedIndex];
            const requiresTax = selected.dataset.requiresTax === '1';
            document.getElementById('tax-type-group').style.display = requiresTax ? 'block' : 'none';
        });
        
        document.getElementById('add-account-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            // Check if logo file is provided
            const logoFile = formData.get('logo');
            const hasLogo = logoFile && logoFile.size > 0;
            
            if (hasLogo) {
                // Use FormData for file upload
                try {
                    await API.createAccountWithLogo(formData);
                    closeModal();
                    loadAccounts();
                    loadDashboard();
                } catch (error) {
                    alert('Errore: ' + error.message);
                }
            } else {
                // Use JSON for regular data
                const data = Object.fromEntries(formData);
                
                // Handle checkbox for show_in_dashboard
                data.show_in_dashboard = formData.get('show_in_dashboard') === 'on';
                
                // Convert empty strings, undefined, or null to null for optional date fields
                if (!data.expire_date || data.expire_date === '') {
                    delete data.expire_date;
                }
                if (!data.closure_date || data.closure_date === '') {
                    delete data.closure_date;
                }
                if (!data.id_number || data.id_number === '') {
                    delete data.id_number;
                }
                
                try {
                    await API.createAccount(data);
                    closeModal();
                    loadAccounts();
                    loadDashboard();
                } catch (error) {
                    alert('Errore: ' + error.message);
                }
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing add account modal:', error);
    }
}

window.showEditAccountModal = async function(accountId) {
    try {
        const types = await API.getAccountTypes();
        const accounts = await API.getAccounts();
        const account = accounts.find(a => a.id === accountId);
        
        if (!account) {
            alert('Account non trovato');
            return;
        }
        
        // Check if account has associated data (expenses for regular accounts, securities for securities accounts)
        let hasAssociatedData = false;
        let warningMessage = '';
        
        if (account.type_code === 'SECURITIES') {
            // For securities accounts, check for securities
            const securities = await API.getSecurities(accountId);
            hasAssociatedData = securities && securities.length > 0;
            warningMessage = hasAssociatedData ? 'Questo account non può essere eliminato perché ha movimenti titoli associati.' : '';
        } else {
            // For other accounts, check for expenses
            const expenses = await API.getExpenses({ account_id: accountId });
            hasAssociatedData = expenses && expenses.length > 0;
            warningMessage = hasAssociatedData ? 'Questo account non può essere eliminato perché ha spese associate.' : '';
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Modifica Account</h2>
            <form id="edit-account-form">
                <input type="hidden" name="id" value="${account.id}">
                <div class="form-group">
                    <label>Nome</label>
                    <input type="text" name="name" value="${escapeHtml(account.name)}" required>
                </div>
                <div class="form-group">
                    <label>Tipo Account</label>
                    <select name="type_id" id="edit-account-type-select" required>
                        <option value="">Seleziona...</option>
                        ${types.map(t => `<option value="${t.id}" data-requires-tax="${t.requires_tax_specification}" ${t.id == account.type_id ? 'selected' : ''}>${escapeHtml(t.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group" id="edit-tax-type-group" style="display: ${account.tax_type ? 'block' : 'none'};">
                    <label>Tipo Tassa</label>
                    <select name="tax_type">
                        <option value="CURRENT_ACCOUNT_TAX" ${account.tax_type === 'CURRENT_ACCOUNT_TAX' ? 'selected' : ''}>Tassa Conto Corrente</option>
                        <option value="DEPOSIT_ACCOUNT_TAX" ${account.tax_type === 'DEPOSIT_ACCOUNT_TAX' ? 'selected' : ''}>Tassa Conto Deposito (0,2%)</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Valuta Base</label>
                    <input type="text" name="base_currency" value="${escapeHtml(account.base_currency)}" maxlength="3" required>
                </div>
                <div class="form-group">
                    <label>Data Apertura</label>
                    <input type="date" name="opening_date" value="${account.opening_date}" required>
                </div>
                <div class="form-group">
                    <label>Numero ID</label>
                    <input type="text" name="id_number" value="${escapeHtml(account.id_number || '')}">
                </div>
                <div class="form-group">
                    <label>Data Scadenza</label>
                    <input type="date" name="expire_date" value="${account.expire_date || ''}">
                </div>
                <div class="form-group">
                    <label>Data Chiusura</label>
                    <input type="date" name="closure_date" value="${account.closure_date || ''}">
                </div>
                <div class="form-group">
                    <label>Logo</label>
                    <input type="file" name="logo" accept="image/*">
                    ${account.logo_path ? `
                        <div style="margin-top: 10px;">
                            <img src="/uploads/${account.logo_path}" style="max-width: 200px; max-height: 100px; border-radius: 5px;" alt="Current logo">
                            <div style="margin-top: 10px;">
                                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                                    <input type="checkbox" name="remove_logo" id="remove-logo-checkbox">
                                    <span>Rimuovi logo</span>
                                </label>
                            </div>
                        </div>
                    ` : ''}
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    ${hasAssociatedData ? '' : `<button type="button" class="btn btn-danger" onclick="deleteAccount(${account.id})" style="margin-right: auto;">Elimina</button>`}
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
                ${warningMessage ? `<div class="alert alert-warning" style="margin-top: 10px;">${warningMessage}</div>` : ''}
            </form>
        `;
        
        document.getElementById('edit-account-type-select').addEventListener('change', (e) => {
            const selected = e.target.options[e.target.selectedIndex];
            const requiresTax = selected.dataset.requiresTax === '1';
            document.getElementById('edit-tax-type-group').style.display = requiresTax ? 'block' : 'none';
        });
        
        document.getElementById('edit-account-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            // Check if remove logo checkbox is checked
            const removeLogo = formData.get('remove_logo') === 'on';
            
            // Check if logo file is provided
            const logoFile = formData.get('logo');
            const hasLogo = logoFile && logoFile.size > 0;
            
            if (hasLogo || removeLogo) {
                // Use FormData for file upload or logo removal
                if (removeLogo) {
                    formData.append('remove_logo', '1');
                }
                try {
                    await API.updateAccountWithLogo(formData);
                    closeModal();
                    loadAccounts();
                    loadDashboard();
                } catch (error) {
                    alert('Errore: ' + error.message);
                }
            } else {
                // Use JSON for regular data
                const data = Object.fromEntries(formData);
                data.id = parseInt(data.id);
                data.type_id = parseInt(data.type_id);
                
                // Handle checkbox for show_in_dashboard
                data.show_in_dashboard = formData.get('show_in_dashboard') === 'on';
                
                // Convert empty strings, undefined, or null to null for optional date fields
                if (!data.expire_date || data.expire_date === '') {
                    delete data.expire_date;
                }
                if (!data.closure_date || data.closure_date === '') {
                    delete data.closure_date;
                }
                if (!data.id_number || data.id_number === '') {
                    delete data.id_number;
                }
                
                try {
                    await API.updateAccount(data);
                    closeModal();
                    loadAccounts();
                    loadDashboard();
                } catch (error) {
                    alert('Errore: ' + error.message);
                }
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing edit account modal:', error);
        alert('Errore nel caricamento dell\'account: ' + error.message);
    }
}

window.deleteAccount = async function(accountId) {
    if (!confirm('Sei sicuro di voler eliminare questo account?')) {
        return;
    }
    
    try {
        await API.deleteAccount(accountId);
        closeModal();
        loadAccounts();
        loadDashboard();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

// Expenses Filters
async function loadExpensesFilters() {
    try {
        const [accounts, prepaidCards] = await Promise.all([
            API.getAccounts(),
            API.getPrepaidCards({ enabled_only: true })
        ]);
        
        const accountSelect = document.getElementById('filter-accounts');
        
        if (!accountSelect) return;
        
        // Filter out securities accounts
        const nonSecuritiesAccounts = accounts.filter(a => a.type_code !== 'SECURITIES')
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        // Filter enabled prepaid cards (owned + shared)
        const enabledPrepaidCards = prepaidCards.filter(c => c.enabled !== false)
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        // Clear existing options except "Tutti"
        accountSelect.innerHTML = '<option value="">Tutti</option>';
        
        // Add account options with prefix
        nonSecuritiesAccounts.forEach(account => {
            const option = document.createElement('option');
            option.value = account.id;
            option.textContent = `[Account] ${account.name}`;
            option.dataset.type = 'account';
            accountSelect.appendChild(option);
        });
        
        // Add prepaid card options with prefix
        enabledPrepaidCards.forEach(card => {
            const option = document.createElement('option');
            option.value = card.id;
            option.textContent = `[Prepagata] ${card.name}`;
            option.dataset.type = 'prepaid_card';
            accountSelect.appendChild(option);
        });
        
        // Apply filters from dashboard if present
        if (window.currentExpenseFilter) {
            // Clear all filters first when applying dashboard filter
            const startDateInput = document.getElementById('filter-start-date');
            const endDateInput = document.getElementById('filter-end-date');
            const keywordsInput = document.getElementById('filter-keywords');
            const deductibleCheckbox = document.getElementById('filter-deductible-only');
            
            // Clear date filters if not in dashboard filter
            if (startDateInput && !window.currentExpenseFilter.start_date) {
                startDateInput.value = '';
            }
            if (endDateInput && !window.currentExpenseFilter.end_date) {
                endDateInput.value = '';
            }
            // Clear keywords if not in dashboard filter
            if (keywordsInput && !window.currentExpenseFilter.description_search) {
                keywordsInput.value = '';
            }
            // Clear deductible if not in dashboard filter
            if (deductibleCheckbox && !window.currentExpenseFilter.is_tax_deductible) {
                deductibleCheckbox.checked = false;
            }
            
            // Apply account filter
            if (window.currentExpenseFilter.account_id) {
                const option = Array.from(accountSelect.options).find(opt => opt.value == window.currentExpenseFilter.account_id && opt.dataset.type === 'account');
                if (option) {
                    // Clear all selections first
                    Array.from(accountSelect.options).forEach(opt => opt.selected = false);
                    option.selected = true;
                }
            }
            
            // Apply prepaid card filter
            if (window.currentExpenseFilter.prepaid_card_id) {
                const option = Array.from(accountSelect.options).find(opt => opt.value == window.currentExpenseFilter.prepaid_card_id && opt.dataset.type === 'prepaid_card');
                if (option) {
                    // Clear all selections first
                    Array.from(accountSelect.options).forEach(opt => opt.selected = false);
                    option.selected = true;
                }
            }
            
            // Apply date filters
            if (window.currentExpenseFilter.start_date && startDateInput) {
                startDateInput.value = window.currentExpenseFilter.start_date;
            }
            
            if (window.currentExpenseFilter.end_date && endDateInput) {
                endDateInput.value = window.currentExpenseFilter.end_date;
            }
            
            // Apply keywords filter
            if (window.currentExpenseFilter.description_search && keywordsInput) {
                keywordsInput.value = window.currentExpenseFilter.description_search;
            }
            
            // Apply deductible filter
            if (window.currentExpenseFilter.is_tax_deductible && deductibleCheckbox) {
                deductibleCheckbox.checked = true;
            }
        }
    } catch (error) {
        console.error('Error loading expense filters:', error);
    }
}

function getExpenseFilters() {
    const accountSelect = document.getElementById('filter-accounts');
    const startDateInput = document.getElementById('filter-start-date');
    const endDateInput = document.getElementById('filter-end-date');
    const keywordsInput = document.getElementById('filter-keywords');
    const deductibleCheckbox = document.getElementById('filter-deductible-only');
    
    const filters = {};
    
    // Get selected account IDs and prepaid card IDs (can select both simultaneously)
    if (accountSelect) {
        const selectedItems = Array.from(accountSelect.selectedOptions)
            .map(opt => ({ value: opt.value, type: opt.dataset.type }))
            .filter(item => item.value !== '');
        
        if (selectedItems.length > 0) {
            const accountIds = selectedItems.filter(item => item.type === 'account').map(item => item.value);
            const prepaidCardIds = selectedItems.filter(item => item.type === 'prepaid_card').map(item => item.value);
            
            if (accountIds.length > 0) {
                filters.account_ids = accountIds.join(',');
            }
            if (prepaidCardIds.length > 0) {
                filters.prepaid_card_ids = prepaidCardIds.join(',');
            }
        }
    }
    
    // Get date range
    if (startDateInput && startDateInput.value) {
        filters.start_date = startDateInput.value;
    }
    if (endDateInput && endDateInput.value) {
        filters.end_date = endDateInput.value;
    }
    
    // Get keywords
    if (keywordsInput && keywordsInput.value.trim()) {
        filters.description_search = keywordsInput.value.trim();
    }
    
    // Get deductible filter
    if (deductibleCheckbox && deductibleCheckbox.checked) {
        filters.is_tax_deductible = '1';
    }
    
    return filters;
}

function clearExpenseFilters() {
    const accountSelect = document.getElementById('filter-accounts');
    const startDateInput = document.getElementById('filter-start-date');
    const endDateInput = document.getElementById('filter-end-date');
    const keywordsInput = document.getElementById('filter-keywords');
    const deductibleCheckbox = document.getElementById('filter-deductible-only');
    
    if (accountSelect) {
        // Clear all selections in multi-select
        Array.from(accountSelect.options).forEach(opt => opt.selected = false);
        // Select "Tutti" option
        if (accountSelect.options.length > 0) {
            accountSelect.options[0].selected = true;
        }
    }
    if (startDateInput) {
        startDateInput.value = '';
    }
    if (endDateInput) {
        endDateInput.value = '';
    }
    if (keywordsInput) {
        keywordsInput.value = '';
    }
    if (deductibleCheckbox) {
        deductibleCheckbox.checked = false;
    }
    
    // Clear date preset dropdown
    const datePresetSelect = document.getElementById('filter-date-preset');
    if (datePresetSelect) {
        datePresetSelect.value = '';
    }
    
    // Clear dashboard filter as well
    window.currentExpenseFilter = null;
}

// Helper function to format date as YYYY-MM-DD using local time (not UTC)
function formatDateLocal(date) {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
}

// Helper function to normalize decimal separator (comma to dot)
function normalizeDecimalSeparator(value) {
    if (typeof value === 'string') {
        return value.replace(',', '.');
    }
    return value;
}

// Helper function to setup decimal separator normalization for text inputs
function setupDecimalNormalization(inputElement) {
    if (!inputElement) return;
    
    // Allow both comma and dot, but convert comma to dot in real-time
    inputElement.addEventListener('input', function(e) {
        const value = e.target.value;
        // Remove any non-numeric characters except comma and dot
        let cleanedValue = value.replace(/[^\d,.]/g, '');
        
        // Count commas and dots
        const commaCount = (cleanedValue.match(/,/g) || []).length;
        const dotCount = (cleanedValue.match(/\./g) || []).length;
        
        // If both comma and dot exist, keep only the first one encountered
        if (commaCount > 0 && dotCount > 0) {
            const firstComma = cleanedValue.indexOf(',');
            const firstDot = cleanedValue.indexOf('.');
            if (firstComma < firstDot) {
                // Comma comes first, remove all dots
                cleanedValue = cleanedValue.replace(/\./g, '');
            } else {
                // Dot comes first, remove all commas
                cleanedValue = cleanedValue.replace(/,/g, '');
            }
        }
        
        // Convert comma to dot
        if (cleanedValue.includes(',')) {
            const cursorPosition = e.target.selectionStart;
            const newValue = cleanedValue.replace(/,/g, '.');
            e.target.value = newValue;
            // Restore cursor position
            const newPosition = cleanedValue.indexOf(',') < cursorPosition ? cursorPosition : cursorPosition;
            e.target.setSelectionRange(newPosition, newPosition);
        } else if (cleanedValue !== value) {
            e.target.value = cleanedValue;
        }
    });
    
    // Handle blur event to ensure final value is normalized
    inputElement.addEventListener('blur', function(e) {
        const value = e.target.value;
        if (value.includes(',')) {
            e.target.value = value.replace(',', '.');
        }
    });
    
    // Handle paste event
    inputElement.addEventListener('paste', function(e) {
        setTimeout(() => {
            const value = e.target.value;
            // Remove any non-numeric characters except comma and dot
            let cleanedValue = value.replace(/[^\d,.]/g, '');
            
            // Count commas and dots
            const commaCount = (cleanedValue.match(/,/g) || []).length;
            const dotCount = (cleanedValue.match(/\./g) || []).length;
            
            // If both comma and dot exist, keep only the first one encountered
            if (commaCount > 0 && dotCount > 0) {
                const firstComma = cleanedValue.indexOf(',');
                const firstDot = cleanedValue.indexOf('.');
                if (firstComma < firstDot) {
                    cleanedValue = cleanedValue.replace(/\./g, '');
                } else {
                    cleanedValue = cleanedValue.replace(/,/g, '');
                }
            }
            
            // Convert comma to dot
            if (cleanedValue.includes(',')) {
                const cursorPosition = e.target.selectionStart;
                const newValue = cleanedValue.replace(/,/g, '.');
                e.target.value = newValue;
                e.target.setSelectionRange(cursorPosition, cursorPosition);
            } else if (cleanedValue !== value) {
                e.target.value = cleanedValue;
            }
        }, 0);
    });
    
    // Prevent invalid characters on keydown (but allow comma and dot)
    inputElement.addEventListener('keydown', function(e) {
        // Allow: backspace, delete, tab, escape, enter
        if ([46, 8, 9, 27, 13].indexOf(e.keyCode) !== -1 ||
            // Allow: Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
            (e.keyCode === 65 && e.ctrlKey === true) ||
            (e.keyCode === 67 && e.ctrlKey === true) ||
            (e.keyCode === 86 && e.ctrlKey === true) ||
            (e.keyCode === 88 && e.ctrlKey === true) ||
            // Allow: home, end, left, right
            (e.keyCode >= 35 && e.keyCode <= 39)) {
            return;
        }
        // Allow comma (188) and dot (190) and numpad decimal point (110) - but only one decimal separator
        if (e.keyCode === 188 || e.keyCode === 190 || e.keyCode === 110) {
            const value = e.target.value;
            if (value.includes(',') || value.includes('.')) {
                e.preventDefault();
            }
            return;
        }
        // Ensure that it is a number and stop the keypress
        if ((e.shiftKey || (e.keyCode < 48 || e.keyCode > 57)) && (e.keyCode < 96 || e.keyCode > 105)) {
            e.preventDefault();
        }
    });
    
    // Also handle keypress event for better compatibility
    inputElement.addEventListener('keypress', function(e) {
        const char = String.fromCharCode(e.which || e.keyCode);
        const value = e.target.value;
        
        // Allow digits
        if (char >= '0' && char <= '9') {
            return;
        }
        
        // Allow comma and dot (including numpad decimal point) - but only one decimal separator
        if (char === ',' || char === '.') {
            if (value.includes(',') || value.includes('.')) {
                e.preventDefault();
            }
            return;
        }
        
        // Block all other characters
        e.preventDefault();
    });
}

function applyDatePreset(preset) {
    const startDateInput = document.getElementById('filter-start-date');
    const endDateInput = document.getElementById('filter-end-date');
    
    if (!startDateInput || !endDateInput) {
        return;
    }
    
    const today = new Date();
    const todayStr = formatDateLocal(today);
    let startDate = '';
    let endDate = todayStr;
    
    switch (preset) {
        case 'this-month':
            // First day of current month to today
            startDate = formatDateLocal(new Date(today.getFullYear(), today.getMonth(), 1));
            break;
            
        case 'last-three-months':
            // Three months ago to today
            const threeMonthsAgo = new Date(today);
            threeMonthsAgo.setMonth(today.getMonth() - 3);
            startDate = formatDateLocal(threeMonthsAgo);
            break;
            
        case 'this-year':
            // First day of current year to today
            startDate = formatDateLocal(new Date(today.getFullYear(), 0, 1));
            break;
            
        case 'past-year':
            // First day of last year to last day of last year
            const lastYear = today.getFullYear() - 1;
            startDate = formatDateLocal(new Date(lastYear, 0, 1));
            endDate = formatDateLocal(new Date(lastYear, 11, 31));
            break;
    }
    
    startDateInput.value = startDate;
    endDateInput.value = endDate;
}

// Expenses
async function loadExpenses() {
    try {
        const container = document.getElementById('expenses-list');
        if (!container) {
            console.error('Expenses list container not found');
            return;
        }
        
        // Show loading state
        container.innerHTML = '<div style="text-align: center; padding: 2rem; color: #666;">Caricamento spese...</div>';
        
        // Get filters from UI or use filter if set (from dashboard click)
        // Priority: dashboard filter (if set) > UI filters
        let filters = {};
        
        if (window.currentExpenseFilter) {
            // Use dashboard filter - it was set when clicking from dashboard or prepaid card
            filters = { ...window.currentExpenseFilter };
            // Convert account_id to account_ids format for API
            if (filters.account_id && !filters.account_ids) {
                filters.account_ids = filters.account_id.toString();
                delete filters.account_id;
            }
            // Convert prepaid_card_id to prepaid_card_ids format for API
            if (filters.prepaid_card_id && !filters.prepaid_card_ids) {
                filters.prepaid_card_ids = filters.prepaid_card_id.toString();
                delete filters.prepaid_card_id;
            }
        } else {
            // Use filters from UI (user manually applied them)
            filters = getExpenseFilters();
        }
        
        // Log filters for debugging
        console.log('Loading expenses with filters:', filters);
        
        const expenses = await API.getExpenses(filters);
        
        container.innerHTML = expenses.map(expense => {
            const isTransfer = expense.is_transfer === 1 || expense.is_transfer === true;
            const isPositive = parseFloat(expense.amount) >= 0;
            const amountColor = isPositive ? '#28a745' : '#dc3545';
            const amountSign = isPositive ? '+' : '';
            
            // Check if expense was created by another user
            const isCreatedByOther = expense.user_id && currentUser && expense.user_id != currentUser.user_id;
            const createdByInfo = isCreatedByOther && (expense.created_by_name || expense.created_by_email)
                ? ` <span style="color: #666; font-size: 0.85em; font-style: italic;">(inserita da ${escapeHtml(expense.created_by_name || expense.created_by_email)})</span>`
                : '';
            
            const attachmentsHtml = expense.attachments && expense.attachments.length > 0
                ? `<div style="margin-top: 10px; display: flex; gap: 10px; flex-wrap: wrap;">
                    ${expense.attachments.map(att => {
                        if (att.type === 'PHOTO') {
                            return `<a href="/uploads/${att.file_path}" target="_blank"><img src="/uploads/${att.file_path}" style="max-width: 60px; max-height: 60px; border-radius: 5px; border: 1px solid #ddd;"></a>`;
                        } else if (att.type === 'DOCUMENT') {
                            return `<a href="/uploads/${att.file_path}" target="_blank" style="display: inline-block; padding: 5px 10px; background: #f0f0f0; border-radius: 5px; text-decoration: none; color: #333;">📄 ${att.file_path.split('/').pop()}</a>`;
                        } else if (att.type === 'LINK') {
                            return `<a href="${escapeHtml(att.link_url)}" target="_blank" style="display: inline-block; padding: 5px 10px; background: #e3f2fd; border-radius: 5px; text-decoration: none; color: #1976d2;">🔗 ${escapeHtml(att.link_text || att.link_url)}</a>`;
                        }
                    }).join('')}
                   </div>`
                : '';
            
            const transferBadge = isTransfer ? '<span style="background: #667eea; color: white; padding: 2px 8px; border-radius: 3px; font-size: 0.8em; margin-left: 8px;">Trasferimento</span>' : '';
            const transferType = expense.transfer_type || '';
            const transferId = expense.transfer_id || null;
            const arrivalDateInfo = isTransfer && expense.arrival_date && expense.transfer_type === 'INCOMING' 
                ? ` • Arrivo: ${formatDate(expense.arrival_date)}`
                : '';
            const departureDateInfo = isTransfer && expense.arrival_date && expense.transfer_type === 'OUTGOING'
                ? ` • Arrivo: ${formatDate(expense.arrival_date)}`
                : '';
            
            let actionButtons = '';
            if (isTransfer && transferId) {
                if (transferType === 'OUTGOING') {
                    // Outgoing transfer: show edit/delete buttons only if created by current user
                    const canEditTransfer = !isCreatedByOther;
                    const editButton = canEditTransfer 
                        ? `<button class="btn btn-secondary" onclick="window.showEditTransferModal(${transferId})">Modifica</button>`
                        : '';
                    actionButtons = `
                        ${editButton}
                        <button class="btn btn-danger" onclick="window.deleteTransfer(${transferId})">Elimina</button>
                    `;
                } else if (transferType === 'INCOMING') {
                    // Incoming transfer: show "go to initial movement" button
                    actionButtons = `
                        <button class="btn btn-secondary" onclick="window.goToInitialTransferMovement(${transferId})">Vai al Movimento Iniziale</button>
                    `;
                }
            } else if (!isTransfer && expense.id) {
                // Regular expense
                actionButtons = `
                    <button class="btn btn-secondary" onclick="window.showEditExpenseModal(${expense.id})">Modifica</button>
                    <button class="btn btn-danger" onclick="window.deleteExpense(${expense.id})">Elimina</button>
                `;
            }
            
            // Generate unique ID for the list item based on expense/transfer ID
            const itemId = isTransfer && transferId 
                ? `expense-item-transfer-${transferId}-${transferType}`
                : expense.id 
                    ? `expense-item-${expense.id}`
                    : `expense-item-${Date.now()}-${Math.random()}`;
            
            return `
            <div class="list-item" id="${itemId}" style="${isTransfer ? 'background: #f8f9fa;' : ''}">
                <div class="list-item-content">
                    <div class="list-item-title">
                        ${escapeHtml(expense.description)}${transferBadge}${createdByInfo}
                    </div>
                    <div class="list-item-meta">
                        ${formatDate(expense.date)}${departureDateInfo}${arrivalDateInfo} • 
                        <span style="color: ${amountColor}; font-weight: bold;">${amountSign}${formatCurrency(Math.abs(expense.amount), expense.currency)}</span> • 
                        ${expense.source_name || expense.account_name || expense.prepaid_card_name || 'N/A'}
                        ${expense.is_tax_deductible ? ` • ${expense.tax_category_name || 'Deducibile'}` : ''}
                    </div>
                    ${attachmentsHtml}
                </div>
                <div class="list-item-actions">
                    ${actionButtons}
                </div>
            </div>
            `;
        }).join('');
        
        // Show message if no expenses found
        if (expenses.length === 0) {
            container.innerHTML = '<div style="text-align: center; padding: 2rem; color: #666;"><p>Nessuna spesa trovata con i filtri selezionati.</p></div>';
        }
    } catch (error) {
        console.error('Error loading expenses:', error);
        const container = document.getElementById('expenses-list');
        if (container) {
            container.innerHTML = `<div style="text-align: center; padding: 2rem; color: #dc3545;"><p>Errore nel caricamento delle spese: ${error.message}</p></div>`;
        }
    }
}

async function showAddExpenseModal() {
    try {
        const [accounts, taxCategories, profile, prepaidCards] = await Promise.all([
            API.getAccounts(),
            API.getTaxCategories(),
            API.getProfile(),
            API.getPrepaidCards({ enabled_only: true })
        ]);
        
        // Filter out securities accounts
        const nonSecuritiesAccounts = accounts.filter(a => a.type_code !== 'SECURITIES')
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        // Filter enabled prepaid cards
        const enabledPrepaidCards = prepaidCards.filter(c => c.enabled !== false)
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        // Check if accounts or prepaid cards exist
        if ((!nonSecuritiesAccounts || nonSecuritiesAccounts.length === 0) && (!enabledPrepaidCards || enabledPrepaidCards.length === 0)) {
            alert('Errore: Devi prima creare almeno un account (non di tipo Securities) o una carta prepagata abilitata prima di aggiungere una spesa.');
            return;
        }
        
        // Check current filters to pre-select type and value
        const currentFilters = getExpenseFilters();
        let preselectedType = 'account';
        let preselectedId = null;
        
        // If there's exactly one account selected in filters, use it
        if (currentFilters.account_ids) {
            const accountIds = currentFilters.account_ids.split(',').filter(id => id !== '');
            if (accountIds.length === 1) {
                preselectedType = 'account';
                preselectedId = parseInt(accountIds[0]);
            }
        }
        
        // If there's exactly one prepaid card selected in filters, use it (overrides account if both are selected)
        if (currentFilters.prepaid_card_ids) {
            const prepaidCardIds = currentFilters.prepaid_card_ids.split(',').filter(id => id !== '');
            if (prepaidCardIds.length === 1) {
                preselectedType = 'prepaid_card';
                preselectedId = parseInt(prepaidCardIds[0]);
            }
        }
        
        // If no filter is selected, use default account from profile
        if (!preselectedId) {
            const defaultAccountId = profile.default_account_id || null;
            // Make sure default account is not a securities account
            preselectedId = nonSecuritiesAccounts.find(a => a.id == defaultAccountId) ? defaultAccountId : null;
            preselectedType = 'account';
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Aggiungi Spesa</h2>
            <form id="add-expense-form">
                <div class="form-group">
                    <label>Data</label>
                    <input type="date" name="date" value="${getTodayDate()}" required>
                </div>
                <div class="form-group">
                    <label>Descrizione</label>
                    <textarea name="description" required></textarea>
                </div>
                <div class="form-group">
                    <label style="display: block; margin-bottom: 8px;">Tipo</label>
                    <div style="display: flex; gap: 20px; margin-bottom: 10px;">
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="radio" name="source_type" value="account" ${preselectedType === 'account' ? 'checked' : ''}>
                            <span>Account</span>
                        </label>
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="radio" name="source_type" value="prepaid_card" ${preselectedType === 'prepaid_card' ? 'checked' : ''}>
                            <span>Carte Prepagate</span>
                        </label>
                    </div>
                    <select name="${preselectedType === 'account' ? 'account_id' : 'prepaid_card_id'}" id="expense-source-select" required>
                        <option value="">Seleziona...</option>
                        ${preselectedType === 'account' 
                            ? nonSecuritiesAccounts.map(a => `<option value="${a.id}" ${a.id == preselectedId ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')
                            : enabledPrepaidCards.map(c => `<option value="${c.id}" ${c.id == preselectedId ? 'selected' : ''}>${escapeHtml(c.name)}</option>`).join('')
                        }
                    </select>
                    <input type="hidden" name="${preselectedType === 'account' ? 'prepaid_card_id' : 'account_id'}" id="expense-prepaid-card-id">
                </div>
                <div class="form-group">
                    <label>Tipo</label>
                    <select name="expense_type" id="expense-type-select">
                        <option value="expense">Uscita (Spesa)</option>
                        <option value="income">Entrata (Fondi Ricevuti)</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Importo</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="amount" min="0" required>
                    <small style="color: #666; font-size: 0.9em; display: block; margin-top: 4px;">L'importo sarà automaticamente negativo per le uscite e positivo per le entrate</small>
                </div>
                <div class="form-group">
                    <label>Valuta</label>
                    <input type="text" name="currency" value="EUR" maxlength="3" required>
                </div>
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" name="is_tax_deductible" id="tax-deductible">
                        <label for="tax-deductible">Spesa Deducibile</label>
                    </div>
                </div>
                <div class="form-group" id="tax-category-group" style="display: none;">
                    <label>Categoria Deducibile</label>
                    <select name="tax_category_id">
                        <option value="">Seleziona...</option>
                        ${taxCategories.map(c => `<option value="${c.id}">${escapeHtml(c.name)} - ${escapeHtml(c.description || '')}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Allegati</label>
                    <div id="attachments-list" style="margin-bottom: 10px;"></div>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="btn btn-secondary" onclick="addAttachmentFile('add-expense')">Aggiungi Foto/Documento</button>
                        <button type="button" class="btn btn-secondary" onclick="addAttachmentLink('add-expense')">Aggiungi Link</button>
                    </div>
                    <input type="file" id="add-expense-file-input" style="display: none;" accept="image/*,.pdf,.doc,.docx" multiple>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        // Initialize attachments array for this form
        window.currentAttachments = [];
        
        // Store accounts and prepaid cards for dynamic dropdown update
        const expenseAccounts = nonSecuritiesAccounts;
        const expensePrepaidCards = enabledPrepaidCards;
        
        // Handle source type change (Account/Prepaid Cards)
        const sourceTypeRadios = document.querySelectorAll('input[name="source_type"]');
        const sourceSelect = document.getElementById('expense-source-select');
        const prepaidCardIdInput = document.getElementById('expense-prepaid-card-id');
        
        function updateSourceDropdown() {
            const selectedType = document.querySelector('input[name="source_type"]:checked').value;
            sourceSelect.innerHTML = '<option value="">Seleziona...</option>';
            prepaidCardIdInput.value = '';
            
            if (selectedType === 'account') {
                const sortedAccounts = [...expenseAccounts].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
                sortedAccounts.forEach(a => {
                    const option = document.createElement('option');
                    option.value = a.id;
                    option.textContent = a.name;
                    // Select if it matches the preselected ID and type
                    if (preselectedType === 'account' && a.id == preselectedId) {
                        option.selected = true;
                    }
                    sourceSelect.appendChild(option);
                });
                sourceSelect.name = 'account_id';
                sourceSelect.required = true;
            } else {
                const sortedPrepaidCards = [...expensePrepaidCards].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
                sortedPrepaidCards.forEach(c => {
                    const option = document.createElement('option');
                    option.value = c.id;
                    option.textContent = c.name;
                    // Select if it matches the preselected ID and type
                    if (preselectedType === 'prepaid_card' && c.id == preselectedId) {
                        option.selected = true;
                    }
                    sourceSelect.appendChild(option);
                });
                sourceSelect.name = 'prepaid_card_id';
                sourceSelect.required = true;
            }
        }
        
        // Initial call to set up the dropdown correctly
        updateSourceDropdown();
        
        sourceTypeRadios.forEach(radio => {
            radio.addEventListener('change', updateSourceDropdown);
        });
        
        document.getElementById('tax-deductible').addEventListener('change', (e) => {
            document.getElementById('tax-category-group').style.display = e.target.checked ? 'block' : 'none';
        });
        
        document.getElementById('add-expense-file-input').addEventListener('change', (e) => {
            Array.from(e.target.files).forEach(file => {
                window.currentAttachments.push({
                    type: file.type.startsWith('image/') ? 'PHOTO' : 'DOCUMENT',
                    file: file,
                    tempId: 'temp_' + Date.now() + '_' + Math.random()
                });
            });
            updateAttachmentsList('add-expense');
            e.target.value = ''; // Reset input
        });
        
        // Setup decimal separator normalization for amount field
        const amountInput = document.querySelector('#add-expense-form input[name="amount"]');
        setupDecimalNormalization(amountInput);
        
        document.getElementById('add-expense-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const expenseType = formData.get('expense_type') || 'expense';
            const amount = parseFloat(normalizeDecimalSeparator(formData.get('amount')));
            const sourceType = formData.get('source_type') || 'account';
            
            const data = {
                date: formData.get('date'),
                description: formData.get('description'),
                amount: expenseType === 'income' ? amount : -amount, // Positive for income, negative for expense
                currency: formData.get('currency'),
                is_tax_deductible: formData.get('is_tax_deductible') === 'on',
                tax_category_id: formData.get('tax_category_id') ? parseInt(formData.get('tax_category_id')) : null
            };
            
            // Add account_id or prepaid_card_id based on source type
            if (sourceType === 'account') {
                data.account_id = parseInt(formData.get('account_id'));
            } else {
                data.prepaid_card_id = parseInt(formData.get('prepaid_card_id'));
            }
            
            try {
                const result = await API.createExpense(data);
                const expenseId = result.id;
                
                // Upload attachments
                if (window.currentAttachments && window.currentAttachments.length > 0) {
                    for (const attachment of window.currentAttachments) {
                        if (attachment.type === 'PHOTO' || attachment.type === 'DOCUMENT') {
                            await API.addExpenseAttachment(expenseId, attachment.type, attachment.file);
                        } else if (attachment.type === 'LINK') {
                            await API.addExpenseAttachment(expenseId, 'LINK', null, attachment.link_url, attachment.link_text);
                        }
                    }
                }
                
                window.currentAttachments = [];
                closeModal();
                loadExpenses();
                loadDashboard();
                
                // Check if we should suggest archiving the prepaid card
                if (result.suggest_archive_prepaid_card && data.prepaid_card_id) {
                    const cardId = result.suggest_archive_prepaid_card;
                    if (confirm('Il credito residuo di questa carta prepagata è ora a zero. Vuoi archiviarla?')) {
                        try {
                            await API.archivePrepaidCard(cardId, true);
                            // Reload prepaid cards list
                            loadPrepaidCards();
                            // Reload expense filters to remove archived card from dropdown
                            loadExpensesFilters();
                            // Reload expenses list to refresh the view
                            loadExpenses();
                            alert('Carta archiviata con successo.');
                        } catch (error) {
                            console.error('Error archiving card:', error);
                            alert('Errore durante l\'archiviazione della carta: ' + error.message);
                        }
                    }
                }
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing add expense modal:', error);
    }
}

window.showEditExpenseModal = async function(expenseId) {
    try {
        const [expenses, accounts, taxCategories, profile, prepaidCards] = await Promise.all([
            API.getExpenses(),
            API.getAccounts(),
            API.getTaxCategories(),
            API.getProfile(),
            API.getPrepaidCards({ enabled_only: true })
        ]);
        
        const expense = expenses.find(e => e.id === expenseId);
        
        if (!expense) {
            alert('Spesa non trovata');
            return;
        }
        
        // Filter out securities accounts
        const nonSecuritiesAccounts = accounts.filter(a => a.type_code !== 'SECURITIES');
        
        // Filter enabled prepaid cards (owned + shared)
        const enabledPrepaidCards = prepaidCards.filter(c => c.enabled !== false);
        
        // Determine if expense is associated with account or prepaid card
        const isPrepaidCardExpense = expense.prepaid_card_id || expense.prepaid_card_name;
        const sourceType = isPrepaidCardExpense ? 'prepaid_card' : 'account';
        
        // If expense is on a securities account (shouldn't happen, but handle it)
        if (expense.account_id && accounts.find(a => a.id === expense.account_id && a.type_code === 'SECURITIES')) {
            alert('Errore: Questa spesa è associata a un account Securities. Non può essere modificata.');
            return;
        }
        
        const defaultAccountId = profile.default_account_id || null;
        
        const modalBody = document.getElementById('modal-body');
        
        // Build the source selection HTML based on the original type (fixed, cannot be changed)
        let sourceSelectHtml = '';
        if (sourceType === 'account') {
            const sortedAccounts = [...nonSecuritiesAccounts].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
            sourceSelectHtml = `
                <div class="form-group">
                    <label>Account</label>
                    <select name="account_id" id="edit-expense-source-select" required>
                        <option value="">Seleziona...</option>
                        ${sortedAccounts.map(a => `<option value="${a.id}" ${a.id == expense.account_id ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
            `;
        } else {
            const sortedPrepaidCards = [...enabledPrepaidCards].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
            sourceSelectHtml = `
                <div class="form-group">
                    <label>Carta Prepagata</label>
                    <select name="prepaid_card_id" id="edit-expense-source-select" required>
                        <option value="">Seleziona...</option>
                        ${sortedPrepaidCards.map(c => `<option value="${c.id}" ${c.id == expense.prepaid_card_id ? 'selected' : ''}>${escapeHtml(c.name)}</option>`).join('')}
                    </select>
                </div>
            `;
        }
        
        modalBody.innerHTML = `
            <h2>Modifica Spesa</h2>
            <form id="edit-expense-form">
                <input type="hidden" name="id" value="${expense.id}">
                <input type="hidden" name="source_type" value="${sourceType}">
                <div class="form-group">
                    <label>Data</label>
                    <input type="date" name="date" value="${expense.date}" required>
                </div>
                <div class="form-group">
                    <label>Descrizione</label>
                    <textarea name="description" required>${escapeHtml(expense.description)}</textarea>
                </div>
                ${sourceSelectHtml}
                <div class="form-group">
                    <label>Tipo</label>
                    <select name="expense_type" id="edit-expense-type-select">
                        <option value="expense" ${parseFloat(expense.amount) < 0 ? 'selected' : ''}>Uscita (Spesa)</option>
                        <option value="income" ${parseFloat(expense.amount) >= 0 ? 'selected' : ''}>Entrata (Fondi Ricevuti)</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Importo</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="amount" id="edit-expense-amount" value="${Math.abs(expense.amount)}" min="0" required>
                    <small style="color: #666; font-size: 0.9em; display: block; margin-top: 4px;">L'importo sarà automaticamente negativo per le uscite e positivo per le entrate</small>
                </div>
                <div class="form-group">
                    <label>Valuta</label>
                    <input type="text" name="currency" value="${escapeHtml(expense.currency)}" maxlength="3" required>
                </div>
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" name="is_tax_deductible" id="edit-tax-deductible" ${expense.is_tax_deductible ? 'checked' : ''}>
                        <label for="edit-tax-deductible">Spesa Deducibile</label>
                    </div>
                </div>
                <div class="form-group" id="edit-tax-category-group" style="display: ${expense.is_tax_deductible ? 'block' : 'none'};">
                    <label>Categoria Deducibile</label>
                    <select name="tax_category_id">
                        <option value="">Seleziona...</option>
                        ${taxCategories.map(c => `<option value="${c.id}" ${c.id == expense.tax_category_id ? 'selected' : ''}>${escapeHtml(c.name)} - ${escapeHtml(c.description || '')}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Allegati</label>
                    <div id="edit-attachments-list" style="margin-bottom: 10px;">
                        ${expense.attachments && expense.attachments.length > 0 ? expense.attachments.map(att => `
                            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                                ${att.type === 'PHOTO' ? `<img src="/uploads/${att.file_path}" style="max-width: 50px; max-height: 50px; border-radius: 3px;">` : ''}
                                ${att.type === 'DOCUMENT' ? `<span>📄 ${att.file_path.split('/').pop()}</span>` : ''}
                                ${att.type === 'LINK' ? `<span>🔗 ${escapeHtml(att.link_text || att.link_url)}</span>` : ''}
                                <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="deleteExpenseAttachment(${att.id}, 'edit-expense')">Elimina</button>
                            </div>
                        `).join('') : '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>'}
                    </div>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="btn btn-secondary" onclick="addAttachmentFile('edit-expense')">Aggiungi Foto/Documento</button>
                        <button type="button" class="btn btn-secondary" onclick="addAttachmentLink('edit-expense')">Aggiungi Link</button>
                    </div>
                    <input type="file" id="edit-expense-file-input" style="display: none;" accept="image/*,.pdf,.doc,.docx" multiple>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        // Initialize attachments array for this form
        window.currentAttachments = [];
        window.currentExpenseId = expenseId;
        
        document.getElementById('edit-tax-deductible').addEventListener('change', (e) => {
            document.getElementById('edit-tax-category-group').style.display = e.target.checked ? 'block' : 'none';
        });
        
        document.getElementById('edit-expense-file-input').addEventListener('change', (e) => {
            Array.from(e.target.files).forEach(file => {
                window.currentAttachments.push({
                    type: file.type.startsWith('image/') ? 'PHOTO' : 'DOCUMENT',
                    file: file,
                    tempId: 'temp_' + Date.now() + '_' + Math.random()
                });
            });
            updateAttachmentsList('edit-expense');
            e.target.value = ''; // Reset input
        });
        
        // Setup decimal separator normalization for amount field
        const editAmountInput = document.querySelector('#edit-expense-form input[name="amount"]');
        setupDecimalNormalization(editAmountInput);
        
        document.getElementById('edit-expense-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const expenseType = formData.get('expense_type') || 'expense';
            const amount = parseFloat(normalizeDecimalSeparator(formData.get('amount')));
            const sourceType = formData.get('source_type'); // Fixed type from original expense (cannot be changed)
            
            const data = {
                id: parseInt(formData.get('id')),
                date: formData.get('date'),
                description: formData.get('description'),
                amount: expenseType === 'income' ? amount : -amount, // Positive for income, negative for expense
                currency: formData.get('currency'),
                is_tax_deductible: formData.get('is_tax_deductible') === 'on',
                tax_category_id: formData.get('tax_category_id') ? parseInt(formData.get('tax_category_id')) : null
            };
            
            // Add account_id or prepaid_card_id based on fixed source type
            if (sourceType === 'account') {
                data.account_id = parseInt(formData.get('account_id'));
            } else {
                data.prepaid_card_id = parseInt(formData.get('prepaid_card_id'));
            }
            
            try {
                const result = await API.updateExpense(data);
                
                // Upload new attachments
                if (window.currentAttachments && window.currentAttachments.length > 0) {
                    for (const attachment of window.currentAttachments) {
                        if (attachment.type === 'PHOTO' || attachment.type === 'DOCUMENT') {
                            await API.addExpenseAttachment(expenseId, attachment.type, attachment.file);
                        } else if (attachment.type === 'LINK') {
                            await API.addExpenseAttachment(expenseId, 'LINK', null, attachment.link_url, attachment.link_text);
                        }
                    }
                }
                
                window.currentAttachments = [];
                window.currentExpenseId = null;
                closeModal();
                loadExpenses();
                loadDashboard();
                
                // Check if we should suggest restoring the prepaid card
                if (result.suggest_restore_prepaid_card) {
                    const cardId = result.suggest_restore_prepaid_card;
                    if (confirm('Il credito residuo di questa carta prepagata è tornato positivo. Vuoi ripristinarla?')) {
                        try {
                            await API.archivePrepaidCard(cardId, false);
                            // Reload prepaid cards list
                            loadPrepaidCards();
                            // Reload expense filters to add restored card to dropdown
                            loadExpensesFilters();
                            alert('Carta ripristinata con successo.');
                        } catch (error) {
                            console.error('Error restoring card:', error);
                            alert('Errore durante il ripristino della carta: ' + error.message);
                        }
                    }
                }
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing edit expense modal:', error);
        alert('Errore nel caricamento della spesa: ' + error.message);
    }
}

window.deleteExpense = async function(expenseId) {
    if (!confirm('Sei sicuro di voler eliminare questa spesa?')) {
        return;
    }
    
    try {
        const result = await API.deleteExpense(expenseId);
        loadExpenses();
        loadDashboard();
        
        // Check if we should suggest restoring the prepaid card
        if (result.suggest_restore_prepaid_card) {
            const cardId = result.suggest_restore_prepaid_card;
            if (confirm('Il credito residuo di questa carta prepagata è tornato positivo. Vuoi ripristinarla?')) {
                try {
                    await API.archivePrepaidCard(cardId, false);
                    // Reload prepaid cards list
                    loadPrepaidCards();
                    // Reload expense filters to add restored card to dropdown
                    loadExpensesFilters();
                    alert('Carta ripristinata con successo.');
                } catch (error) {
                    console.error('Error restoring card:', error);
                    alert('Errore durante il ripristino della carta: ' + error.message);
                }
            }
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

// Transfers
async function showAddTransferModal(fromAccountId = null) {
    try {
        const [accounts, profile] = await Promise.all([
            API.getAccounts(),
            API.getProfile()
        ]);
        
        if (!accounts || accounts.length === 0) {
            alert('Errore: Devi prima creare almeno un account prima di aggiungere un trasferimento.');
            return;
        }
        
        // Use provided fromAccountId or default from profile
        const defaultFromAccountId = fromAccountId || profile.default_account_id || null;
        
        // Sort accounts alphabetically
        const sortedAccounts = [...accounts].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Nuovo Trasferimento</h2>
            <form id="add-transfer-form">
                <div class="form-group">
                    <label>Da Account</label>
                    <select name="from_account_id" id="from-account" required>
                        <option value="">Seleziona...</option>
                        ${sortedAccounts.map(a => `<option value="${a.id}" data-currency="${a.base_currency}" ${a.id == defaultFromAccountId ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>A Account</label>
                    <select name="to_account_id" id="to-account" required>
                        <option value="">Seleziona...</option>
                        ${sortedAccounts.map(a => `<option value="${a.id}" data-currency="${a.base_currency}">${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Importo</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="amount" id="transfer-amount" required>
                </div>
                <div class="form-group" id="exchange-rate-group" style="display: none;">
                    <label>Tasso di Cambio</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="exchange_rate" id="exchange-rate">
                    <button type="button" class="btn btn-secondary" id="fetch-rate-btn">Carica Tasso Attuale</button>
                </div>
                <div class="form-group">
                    <label>Data Partenza</label>
                    <input type="date" name="transfer_date" id="transfer-date" value="${getTodayDate()}" required>
                </div>
                <div class="form-group">
                    <label>Data Arrivo</label>
                    <input type="date" name="arrival_date" id="arrival-date" value="${getNextBusinessDay()}" required>
                </div>
                <div class="form-group">
                    <label>Note</label>
                    <textarea name="notes"></textarea>
                </div>
                <div class="form-group">
                    <label>Allegati</label>
                    <div id="transfer-attachments-list" style="margin-bottom: 10px;"></div>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="btn btn-secondary" onclick="addTransferAttachmentFile()">Aggiungi Foto/Documento</button>
                        <button type="button" class="btn btn-secondary" onclick="addTransferAttachmentLink()">Aggiungi Link</button>
                    </div>
                    <input type="file" id="transfer-file-input" style="display: none;" accept="image/*,.pdf,.doc,.docx" multiple>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        // Initialize attachments array for this form
        window.currentTransferAttachments = [];
        updateTransferAttachmentsList();
        
        document.getElementById('transfer-file-input').addEventListener('change', (e) => {
            Array.from(e.target.files).forEach(file => {
                window.currentTransferAttachments.push({
                    type: file.type.startsWith('image/') ? 'PHOTO' : 'DOCUMENT',
                    file: file,
                    tempId: 'temp_' + Date.now() + '_' + Math.random()
                });
            });
            updateTransferAttachmentsList();
            e.target.value = ''; // Reset input
        });
        
        const fromAccount = document.getElementById('from-account');
        const toAccount = document.getElementById('to-account');
        const exchangeRateGroup = document.getElementById('exchange-rate-group');
        
        function checkCurrencyDifference() {
            const fromCurrency = fromAccount.options[fromAccount.selectedIndex]?.dataset.currency;
            const toCurrency = toAccount.options[toAccount.selectedIndex]?.dataset.currency;
            
            if (fromCurrency && toCurrency && fromCurrency !== toCurrency) {
                exchangeRateGroup.style.display = 'block';
            } else {
                exchangeRateGroup.style.display = 'none';
            }
        }
        
        fromAccount.addEventListener('change', checkCurrencyDifference);
        toAccount.addEventListener('change', checkCurrencyDifference);
        
        document.getElementById('fetch-rate-btn').addEventListener('click', async () => {
            const fromCurrency = fromAccount.options[fromAccount.selectedIndex]?.dataset.currency;
            const toCurrency = toAccount.options[toAccount.selectedIndex]?.dataset.currency;
            
            if (fromCurrency && toCurrency) {
                try {
                    const rate = await API.getExchangeRate(fromCurrency, toCurrency);
                    document.getElementById('exchange-rate').value = rate.rate;
                } catch (error) {
                    alert('Errore nel caricamento del tasso: ' + error.message);
                }
            }
        });
        
        // Setup decimal separator normalization for amount and exchange_rate fields
        const transferAmountInput = document.getElementById('transfer-amount');
        const exchangeRateInput = document.getElementById('exchange-rate');
        if (transferAmountInput) setupDecimalNormalization(transferAmountInput);
        if (exchangeRateInput) setupDecimalNormalization(exchangeRateInput);
        
        document.getElementById('add-transfer-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const data = {
                from_account_id: parseInt(formData.get('from_account_id')),
                to_account_id: parseInt(formData.get('to_account_id')),
                amount: parseFloat(normalizeDecimalSeparator(formData.get('amount'))),
                transfer_date: formData.get('transfer_date'),
                arrival_date: formData.get('arrival_date') || null,
                notes: formData.get('notes') || null,
                exchange_rate: formData.get('exchange_rate') ? parseFloat(normalizeDecimalSeparator(formData.get('exchange_rate'))) : null
            };
            
            try {
                const result = await API.createTransfer(data);
                const transferId = result.id;
                
                // Upload attachments
                if (window.currentTransferAttachments && window.currentTransferAttachments.length > 0) {
                    for (const att of window.currentTransferAttachments) {
                        if (att.type === 'LINK') {
                            await API.addTransferAttachment(transferId, 'LINK', null, att.linkUrl, att.linkText);
                        } else if (att.file) {
                            await API.addTransferAttachment(transferId, att.type, att.file);
                        }
                    }
                }
                
                closeModal();
                loadAccounts();
                loadDashboard();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing add transfer modal:', error);
    }
}

// Make function globally accessible
window.showAddTransferModal = showAddTransferModal;

window.showEditTransferModal = async function(transferId) {
    try {
        const [transfer, accounts] = await Promise.all([
            API.getTransfer(transferId),
            API.getAccounts()
        ]);
        
        if (!transfer) {
            alert('Trasferimento non trovato');
            return;
        }
        
        // Sort accounts alphabetically
        const sortedAccounts = [...accounts].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Modifica Trasferimento</h2>
            <form id="edit-transfer-form">
                <input type="hidden" name="id" value="${transfer.id}">
                <div class="form-group">
                    <label>Da Account</label>
                    <select name="from_account_id" id="edit-from-account" required>
                        <option value="">Seleziona...</option>
                        ${sortedAccounts.map(a => `<option value="${a.id}" data-currency="${a.base_currency}" ${a.id == transfer.from_account_id ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>A Account</label>
                    <select name="to_account_id" id="edit-to-account" required>
                        <option value="">Seleziona...</option>
                        ${sortedAccounts.map(a => `<option value="${a.id}" data-currency="${a.base_currency}" ${a.id == transfer.to_account_id ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Importo</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="amount" id="edit-transfer-amount" value="${transfer.amount}" required>
                </div>
                <div class="form-group" id="edit-exchange-rate-group" style="display: ${transfer.from_currency !== transfer.to_currency ? 'block' : 'none'};">
                    <label>Tasso di Cambio</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="exchange_rate" id="edit-exchange-rate" value="${transfer.exchange_rate || ''}">
                    <button type="button" class="btn btn-secondary" id="edit-fetch-rate-btn">Carica Tasso Attuale</button>
                </div>
                <div class="form-group">
                    <label>Data Partenza</label>
                    <input type="date" name="transfer_date" id="edit-transfer-date" value="${transfer.transfer_date}" required>
                </div>
                <div class="form-group">
                    <label>Data Arrivo</label>
                    <input type="date" name="arrival_date" id="edit-arrival-date" value="${transfer.arrival_date || ''}">
                </div>
                <div class="form-group">
                    <label>Note</label>
                    <textarea name="notes">${escapeHtml(transfer.notes || '')}</textarea>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        const fromAccount = document.getElementById('edit-from-account');
        const toAccount = document.getElementById('edit-to-account');
        const exchangeRateGroup = document.getElementById('edit-exchange-rate-group');
        
        function checkCurrencyDifference() {
            const fromCurrency = fromAccount.options[fromAccount.selectedIndex]?.dataset.currency;
            const toCurrency = toAccount.options[toAccount.selectedIndex]?.dataset.currency;
            
            if (fromCurrency && toCurrency && fromCurrency !== toCurrency) {
                exchangeRateGroup.style.display = 'block';
            } else {
                exchangeRateGroup.style.display = 'none';
            }
        }
        
        fromAccount.addEventListener('change', checkCurrencyDifference);
        toAccount.addEventListener('change', checkCurrencyDifference);
        
        document.getElementById('edit-fetch-rate-btn').addEventListener('click', async () => {
            const fromCurrency = fromAccount.options[fromAccount.selectedIndex]?.dataset.currency;
            const toCurrency = toAccount.options[toAccount.selectedIndex]?.dataset.currency;
            
            if (fromCurrency && toCurrency) {
                try {
                    const rate = await API.getExchangeRate(fromCurrency, toCurrency);
                    document.getElementById('edit-exchange-rate').value = rate.rate;
                } catch (error) {
                    alert('Errore nel caricamento del tasso: ' + error.message);
                }
            }
        });
        
        // Setup decimal separator normalization for amount and exchange_rate fields
        const editTransferAmountInput = document.getElementById('edit-transfer-amount');
        const editExchangeRateInput = document.getElementById('edit-exchange-rate');
        setupDecimalNormalization(editTransferAmountInput);
        setupDecimalNormalization(editExchangeRateInput);
        
        document.getElementById('edit-transfer-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const data = {
                id: parseInt(formData.get('id')),
                from_account_id: parseInt(formData.get('from_account_id')),
                to_account_id: parseInt(formData.get('to_account_id')),
                amount: parseFloat(normalizeDecimalSeparator(formData.get('amount'))),
                transfer_date: formData.get('transfer_date'),
                arrival_date: formData.get('arrival_date') || null,
                notes: formData.get('notes') || null,
                exchange_rate: formData.get('exchange_rate') ? parseFloat(normalizeDecimalSeparator(formData.get('exchange_rate'))) : null
            };
            
            try {
                await API.updateTransfer(data);
                closeModal();
                loadExpenses();
                loadAccounts();
                loadDashboard();
            } catch (error) {
                // Check if error requires contacting the other user
                let errorMessage = error.message || 'Errore sconosciuto';
                if (error.requires_contact || (errorMessage && errorMessage.includes('permessi'))) {
                    alert('Errore: ' + errorMessage + '\n\nContatta l\'utente che ha creato il trasferimento per richiedere la modifica o l\'eliminazione.');
                } else {
                    alert('Errore: ' + errorMessage);
                }
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing edit transfer modal:', error);
        alert('Errore: ' + error.message);
    }
};

window.deleteTransfer = async function(transferId) {
    if (!confirm('Sei sicuro di voler eliminare questo trasferimento?')) {
        return;
    }
    
    try {
        await API.deleteTransfer(transferId);
        loadExpenses();
        loadAccounts();
        loadDashboard();
    } catch (error) {
        // Check if error requires contacting the other user
        let errorMessage = error.message || 'Errore sconosciuto';
        if (error.requires_contact || (errorMessage && errorMessage.includes('permessi'))) {
            alert('Errore: ' + errorMessage + '\n\nContatta l\'utente che ha creato il trasferimento per richiedere la modifica o l\'eliminazione.');
        } else {
            alert('Errore: ' + errorMessage);
        }
    }
};

window.goToInitialTransferMovement = async function(transferId) {
    try {
        // Switch to expenses view and scroll to the outgoing transfer
        switchView('expenses');
        
        // Wait for view switch to complete
        await new Promise(resolve => setTimeout(resolve, 50));
        
        // Load expenses
        await loadExpenses();
        
        // Wait for DOM to be fully rendered - use requestAnimationFrame for better timing
        await new Promise(resolve => {
            requestAnimationFrame(() => {
                requestAnimationFrame(() => {
                    resolve();
                });
            });
        });
        
        // Find the outgoing transfer item by its unique ID
        const itemId = `expense-item-transfer-${transferId}-OUTGOING`;
        let item = document.getElementById(itemId);
        
        // If not found, try again after a short delay (sometimes DOM needs more time)
        if (!item) {
            await new Promise(resolve => setTimeout(resolve, 200));
            item = document.getElementById(itemId);
        }
        
        if (item) {
            // Scroll to the transfer item
            item.scrollIntoView({ behavior: 'smooth', block: 'center' });
            
            // Wait a bit for scroll to start, then highlight
            setTimeout(() => {
                // Highlight it briefly with a more visible color
                const originalBg = item.style.backgroundColor || '';
                const originalTransition = item.style.transition || '';
                item.style.transition = 'background-color 0.3s ease, border 0.3s ease';
                item.style.backgroundColor = '#fff3cd';
                
                // Add a border to make it more visible
                const originalBorder = item.style.border || '';
                item.style.border = '2px solid #ffc107';
                item.style.borderRadius = '4px';
                
                setTimeout(() => {
                    item.style.backgroundColor = originalBg;
                    item.style.border = originalBorder;
                    item.style.transition = originalTransition;
                }, 3000);
            }, 300);
        } else {
            // Fallback: try to find by searching through items
            const container = document.getElementById('expenses-list');
            if (container) {
                const items = container.querySelectorAll('.list-item');
                let found = false;
                items.forEach((item) => {
                    const itemIdAttr = item.getAttribute('id');
                    if (itemIdAttr && itemIdAttr.includes(`transfer-${transferId}-OUTGOING`)) {
                        item.scrollIntoView({ behavior: 'smooth', block: 'center' });
                        setTimeout(() => {
                            const originalBg = item.style.backgroundColor || '';
                            const originalTransition = item.style.transition || '';
                            item.style.transition = 'background-color 0.3s ease, border 0.3s ease';
                            item.style.backgroundColor = '#fff3cd';
                            const originalBorder = item.style.border || '';
                            item.style.border = '2px solid #ffc107';
                            item.style.borderRadius = '4px';
                            setTimeout(() => {
                                item.style.backgroundColor = originalBg;
                                item.style.border = originalBorder;
                                item.style.transition = originalTransition;
                            }, 3000);
                        }, 300);
                        found = true;
                    }
                });
                if (!found) {
                    console.warn('Could not find outgoing transfer item with ID:', itemId);
                    alert('Movimento iniziale non trovato nella lista delle spese.');
                }
            }
        }
    } catch (error) {
        console.error('Error navigating to initial transfer movement:', error);
        alert('Errore: ' + error.message);
    }
};

// Securities
async function loadSecurities() {
    try {
        // Use filter if set (from dashboard click)
        const filters = window.currentSecuritiesFilter || {};
        
        const [summary, securities] = await Promise.all([
            API.getSecuritiesSummary(filters.account_id || null),
            API.getSecurities(filters.account_id || null)
        ]);
        
        // Clear filter after loading
        window.currentSecuritiesFilter = null;
        
        const summaryContainer = document.getElementById('securities-summary');
        summaryContainer.innerHTML = `
            <h3>Riepilogo Titoli</h3>
            <table class="summary-table">
                <thead>
                    <tr>
                        <th>Simbolo</th>
                        <th>Quantità</th>
                        <th>Prezzo Medio</th>
                        <th>Valore Totale</th>
                        <th>Valuta</th>
                    </tr>
                </thead>
                <tbody>
                    ${summary.map(s => `
                        <tr>
                            <td>${escapeHtml(s.symbol)}</td>
                            <td>${Math.round(parseFloat(s.total_quantity))}</td>
                            <td>${parseFloat(s.mean_purchase_price || 0).toFixed(4)} ${s.currency}</td>
                            <td>${formatCurrency(s.total_value, s.currency)}</td>
                            <td>${s.currency}</td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;
        
        const container = document.getElementById('securities-list');
        
        // Group securities by symbol to calculate mean purchase price for profit/loss
        const securitiesBySymbol = {};
        securities.forEach(sec => {
            if (!securitiesBySymbol[sec.symbol]) {
                securitiesBySymbol[sec.symbol] = [];
            }
            securitiesBySymbol[sec.symbol].push(sec);
        });
        
        // Calculate mean purchase price for each symbol
        const meanPrices = {};
        Object.keys(securitiesBySymbol).forEach(symbol => {
            const purchases = securitiesBySymbol[symbol].filter(s => s.transaction_type === 'PURCHASE' || !s.transaction_type);
            if (purchases.length > 0) {
                const totalPrice = purchases.reduce((sum, s) => sum + parseFloat(s.purchase_price), 0);
                const totalQuantity = purchases.reduce((sum, s) => sum + parseFloat(s.quantity), 0);
                meanPrices[symbol] = totalPrice / totalQuantity;
            }
        });
        
         container.innerHTML = securities.map(security => {
             const isSale = security.transaction_type === 'SALE';
             const profitLoss = isSale && meanPrices[security.symbol] 
                 ? (security.purchase_price / security.quantity) - meanPrices[security.symbol]
                 : null;
             const profitLossTotal = profitLoss !== null ? profitLoss * security.quantity : null;
             const profitLossClass = profitLossTotal !== null ? (profitLossTotal >= 0 ? 'profit' : 'loss') : '';
             
             const attachmentsHtml = security.attachments && security.attachments.length > 0
                 ? `<div style="margin-top: 10px; display: flex; gap: 10px; flex-wrap: wrap;">
                     ${security.attachments.map(att => {
                         if (att.type === 'PHOTO') {
                             return `<a href="/uploads/${att.file_path}" target="_blank"><img src="/uploads/${att.file_path}" style="max-width: 60px; max-height: 60px; border-radius: 5px; border: 1px solid #ddd;"></a>`;
                         } else if (att.type === 'DOCUMENT') {
                             return `<a href="/uploads/${att.file_path}" target="_blank" style="display: inline-block; padding: 5px 10px; background: #f0f0f0; border-radius: 5px; text-decoration: none; color: #333;">📄 ${att.file_path.split('/').pop()}</a>`;
                         } else if (att.type === 'LINK') {
                             return `<a href="${escapeHtml(att.link_url)}" target="_blank" style="display: inline-block; padding: 5px 10px; background: #e3f2fd; border-radius: 5px; text-decoration: none; color: #1976d2;">🔗 ${escapeHtml(att.link_text || att.link_url)}</a>`;
                         }
                     }).join('')}
                    </div>`
                 : '';
             
             return `
             <div class="list-item">
                 <div class="list-item-content">
                     <div class="list-item-title">
                         ${escapeHtml(security.symbol)} 
                         ${isSale ? '<span style="color: #d32f2f; font-weight: normal;">(Vendita)</span>' : ''}
                     </div>
                     <div class="list-item-meta">
                         Quantità: ${parseFloat(security.quantity).toFixed(4)} • 
                         Prezzo totale: ${formatCurrency(security.purchase_price, security.currency)} • 
                         Prezzo unitario: ${formatCurrency(security.purchase_price / security.quantity, security.currency)}
                         ${profitLossTotal !== null ? ` • <span class="${profitLossClass}" style="font-weight: bold;">${profitLossTotal >= 0 ? '+' : ''}${formatCurrency(profitLossTotal, security.currency)}</span>` : ''}
                         • Data: ${formatDate(security.purchase_date)}
                         ${security.account_name ? ` • ${escapeHtml(security.account_name)}` : ''}
                         ${security.notes ? `<br>Note: ${escapeHtml(security.notes)}` : ''}
                         ${attachmentsHtml}
                     </div>
                 </div>
                 <div class="list-item-actions">
                     <button class="btn btn-secondary" onclick="window.showEditSecurityModal(${security.id})">Modifica</button>
                     <button class="btn btn-danger" onclick="window.deleteSecurity(${security.id})">Elimina</button>
                 </div>
             </div>
             `;
         }).join('');
    } catch (error) {
        console.error('Error loading securities:', error);
    }
}

async function showAddSecurityModal() {
    try {
        const [accounts, profile] = await Promise.all([
            API.getAccounts(),
            API.getProfile()
        ]);
        
        const securitiesAccounts = accounts.filter(a => a.type_code === 'SECURITIES')
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        if (securitiesAccounts.length === 0) {
            alert('Devi prima creare un account di tipo Securities');
            return;
        }
        
        // Determine which account to select
        let selectedAccountId = null;
        if (securitiesAccounts.length === 1) {
            // If there's only one securities account, auto-select it
            selectedAccountId = securitiesAccounts[0].id;
        } else {
            // Otherwise, check for default account from profile
            const defaultAccountId = profile.default_account_id || null;
            // Only use default account if it's a securities account
            selectedAccountId = securitiesAccounts.find(a => a.id == defaultAccountId) ? defaultAccountId : null;
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Acquista Titolo</h2>
            <form id="add-security-form">
                <div class="form-group">
                    <label>Account</label>
                    <select name="account_id" required>
                        ${securitiesAccounts.length === 1 ? '' : '<option value="">Seleziona...</option>'}
                        ${securitiesAccounts.map(a => `<option value="${a.id}" ${a.id == selectedAccountId ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Simbolo</label>
                    <input type="text" name="symbol" required>
                </div>
                <div class="form-group">
                    <label>Quantità</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="quantity" required>
                </div>
                <div class="form-group">
                    <label>Prezzo Totale di Acquisto</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="purchase_price" required>
                    <small style="color: #666; font-size: 0.9em;">Importo totale pagato per questa transazione</small>
                </div>
                <div class="form-group">
                    <label>Data Acquisto</label>
                    <input type="date" name="purchase_date" value="${getTodayDate()}" required>
                </div>
                <div class="form-group">
                    <label>Valuta</label>
                    <input type="text" name="currency" value="EUR" maxlength="3" required>
                </div>
                <div class="form-group">
                    <label>Note</label>
                    <textarea name="notes"></textarea>
                </div>
                <div class="form-group">
                    <label>Allegati</label>
                    <div id="add-security-attachments-list" style="margin-bottom: 10px;"></div>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="btn btn-secondary" onclick="addSecurityAttachmentFile('add-security')">Aggiungi Foto/Documento</button>
                        <button type="button" class="btn btn-secondary" onclick="addSecurityAttachmentLink('add-security')">Aggiungi Link</button>
                    </div>
                    <input type="file" id="add-security-file-input" style="display: none;" accept="image/*,.pdf,.doc,.docx" multiple>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        // Initialize attachments array for this form
        window.currentSecurityAttachments = [];
        
        document.getElementById('add-security-file-input').addEventListener('change', (e) => {
            Array.from(e.target.files).forEach(file => {
                window.currentSecurityAttachments.push({
                    type: file.type.startsWith('image/') ? 'PHOTO' : 'DOCUMENT',
                    file: file,
                    tempId: 'temp_' + Date.now() + '_' + Math.random()
                });
            });
            updateSecurityAttachmentsList('add-security');
            e.target.value = ''; // Reset input
        });
        
        // Setup decimal separator normalization for quantity and purchase_price fields
        const quantityInput = document.querySelector('#add-security-form input[name="quantity"]');
        const purchasePriceInput = document.querySelector('#add-security-form input[name="purchase_price"]');
        if (quantityInput) setupDecimalNormalization(quantityInput);
        if (purchasePriceInput) setupDecimalNormalization(purchasePriceInput);
        
        document.getElementById('add-security-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const data = {
                account_id: parseInt(formData.get('account_id')),
                symbol: formData.get('symbol'),
                quantity: parseFloat(normalizeDecimalSeparator(formData.get('quantity'))),
                purchase_price: parseFloat(normalizeDecimalSeparator(formData.get('purchase_price'))),
                purchase_date: formData.get('purchase_date'),
                currency: formData.get('currency'),
                notes: formData.get('notes') || null
            };
            
            try {
                const result = await API.createSecurity(data);
                const securityId = result.id;
                
                // Upload attachments
                if (window.currentSecurityAttachments && window.currentSecurityAttachments.length > 0) {
                    for (const attachment of window.currentSecurityAttachments) {
                        if (attachment.type === 'PHOTO' || attachment.type === 'DOCUMENT') {
                            await API.addSecurityAttachment(securityId, attachment.type, attachment.file);
                        } else if (attachment.type === 'LINK') {
                            await API.addSecurityAttachment(securityId, 'LINK', null, attachment.link_url, attachment.link_text);
                        }
                    }
                }
                
                window.currentSecurityAttachments = [];
                closeModal();
                loadSecurities();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing add security modal:', error);
    }
}

window.showEditSecurityModal = async function(securityId) {
    try {
        const [accounts, securities] = await Promise.all([
            API.getAccounts(),
            API.getSecurities()
        ]);
        
        const security = securities.find(s => s.id === securityId);
        if (!security) {
            alert('Titolo non trovato');
            return;
        }
        
        const securitiesAccounts = accounts.filter(a => a.type_code === 'SECURITIES')
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        if (securitiesAccounts.length === 0) {
            alert('Errore: Nessun account di tipo Securities trovato');
            return;
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Modifica Titolo</h2>
            <form id="edit-security-form">
                <input type="hidden" name="id" value="${security.id}">
                <div class="form-group">
                    <label>Account</label>
                    <select name="account_id" required>
                        <option value="">Seleziona...</option>
                        ${securitiesAccounts.map(a => `<option value="${a.id}" ${a.id == security.account_id ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Simbolo</label>
                    <input type="text" name="symbol" value="${escapeHtml(security.symbol)}" required>
                </div>
                <div class="form-group">
                    <label>Quantità</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="quantity" value="${security.quantity}" required>
                </div>
                <div class="form-group">
                    <label>Prezzo Totale di Acquisto</label>
                    <input type="text" inputmode="decimal" pattern="[0-9]+([,\.][0-9]+)?" name="purchase_price" value="${security.purchase_price}" required>
                    <small style="color: #666; font-size: 0.9em;">Importo totale pagato per questa transazione</small>
                </div>
                <div class="form-group">
                    <label>Data Acquisto</label>
                    <input type="date" name="purchase_date" value="${security.purchase_date}" required>
                </div>
                <div class="form-group">
                    <label>Valuta</label>
                    <input type="text" name="currency" value="${escapeHtml(security.currency)}" maxlength="3" required>
                </div>
                <div class="form-group">
                    <label>Note</label>
                    <textarea name="notes">${escapeHtml(security.notes || '')}</textarea>
                </div>
                <div class="form-group">
                    <label>Allegati</label>
                    <div id="edit-security-attachments-list" style="margin-bottom: 10px;">
                        ${security.attachments && security.attachments.length > 0 ? security.attachments.map(att => `
                            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                                ${att.type === 'PHOTO' ? `<img src="/uploads/${att.file_path}" style="max-width: 50px; max-height: 50px; border-radius: 3px;">` : ''}
                                ${att.type === 'DOCUMENT' ? `<span>📄 ${att.file_path.split('/').pop()}</span>` : ''}
                                ${att.type === 'LINK' ? `<span>🔗 ${escapeHtml(att.link_text || att.link_url)}</span>` : ''}
                                <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="deleteSecurityAttachment(${att.id}, 'edit-security')">Elimina</button>
                            </div>
                        `).join('') : '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>'}
                    </div>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="btn btn-secondary" onclick="addSecurityAttachmentFile('edit-security')">Aggiungi Foto/Documento</button>
                        <button type="button" class="btn btn-secondary" onclick="addSecurityAttachmentLink('edit-security')">Aggiungi Link</button>
                    </div>
                    <input type="file" id="edit-security-file-input" style="display: none;" accept="image/*,.pdf,.doc,.docx" multiple>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        // Initialize attachments array for this form
        window.currentSecurityAttachments = [];
        window.currentSecurityId = securityId;
        
        document.getElementById('edit-security-file-input').addEventListener('change', (e) => {
            Array.from(e.target.files).forEach(file => {
                window.currentSecurityAttachments.push({
                    type: file.type.startsWith('image/') ? 'PHOTO' : 'DOCUMENT',
                    file: file,
                    tempId: 'temp_' + Date.now() + '_' + Math.random()
                });
            });
            updateSecurityAttachmentsList('edit-security');
            e.target.value = ''; // Reset input
        });
        
        // Setup decimal separator normalization for quantity and purchase_price fields
        const editQuantityInput = document.querySelector('#edit-security-form input[name="quantity"]');
        const editPurchasePriceInput = document.querySelector('#edit-security-form input[name="purchase_price"]');
        setupDecimalNormalization(editQuantityInput);
        setupDecimalNormalization(editPurchasePriceInput);
        
        document.getElementById('edit-security-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const data = {
                id: parseInt(formData.get('id')),
                account_id: parseInt(formData.get('account_id')),
                symbol: formData.get('symbol'),
                quantity: parseFloat(normalizeDecimalSeparator(formData.get('quantity'))),
                purchase_price: parseFloat(normalizeDecimalSeparator(formData.get('purchase_price'))),
                purchase_date: formData.get('purchase_date'),
                currency: formData.get('currency'),
                notes: formData.get('notes') || null
            };
            
            try {
                await API.updateSecurity(data);
                
                // Upload new attachments
                if (window.currentSecurityAttachments && window.currentSecurityAttachments.length > 0) {
                    for (const attachment of window.currentSecurityAttachments) {
                        if (attachment.type === 'PHOTO' || attachment.type === 'DOCUMENT') {
                            await API.addSecurityAttachment(securityId, attachment.type, attachment.file);
                        } else if (attachment.type === 'LINK') {
                            await API.addSecurityAttachment(securityId, 'LINK', null, attachment.link_url, attachment.link_text);
                        }
                    }
                }
                
                window.currentSecurityAttachments = [];
                window.currentSecurityId = null;
                closeModal();
                loadSecurities();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing edit security modal:', error);
        alert('Errore nel caricamento del titolo: ' + error.message);
    }
}

window.deleteSecurity = async function(securityId) {
    if (!confirm('Sei sicuro di voler eliminare questo titolo?')) {
        return;
    }
    
    try {
        await API.deleteSecurity(securityId);
        loadSecurities();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function showSellSecurityModal() {
    try {
        const [accounts, securities, summary] = await Promise.all([
            API.getAccounts(),
            API.getSecurities(),
            API.getSecuritiesSummary()
        ]);
        
        const securitiesAccounts = accounts.filter(a => a.type_code === 'SECURITIES')
            .sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        if (securitiesAccounts.length === 0) {
            alert('Devi prima creare un account di tipo Securities');
            return;
        }
        
        // Get available securities (only purchases, grouped by symbol and account)
        const availableSecurities = {};
        securities.forEach(sec => {
            if (sec.transaction_type !== 'SALE' && (!sec.transaction_type || sec.transaction_type === 'PURCHASE')) {
                const key = `${sec.account_id}_${sec.symbol}`;
                if (!availableSecurities[key]) {
                    availableSecurities[key] = {
                        account_id: sec.account_id,
                        account_name: sec.account_name,
                        symbol: sec.symbol,
                        currency: sec.currency,
                        total_quantity: 0,
                        mean_price: 0,
                        total_price: 0
                    };
                }
                availableSecurities[key].total_quantity += parseFloat(sec.quantity);
                availableSecurities[key].total_price += parseFloat(sec.purchase_price);
            }
        });
        
        // Calculate mean prices
        Object.keys(availableSecurities).forEach(key => {
            const sec = availableSecurities[key];
            sec.mean_price = sec.total_price / sec.total_quantity;
        });
        
        const availableSecuritiesList = Object.values(availableSecurities).filter(sec => sec.total_quantity > 0);
        
        if (availableSecuritiesList.length === 0) {
            alert('Non ci sono titoli disponibili per la vendita');
            return;
        }
        
        // Sort securities accounts alphabetically
        const sortedSecuritiesAccounts = [...securitiesAccounts].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        
        // Determine which account to select
        let selectedAccountId = null;
        if (sortedSecuritiesAccounts.length === 1) {
            selectedAccountId = sortedSecuritiesAccounts[0].id;
        } else {
            const profile = await API.getProfile();
            const defaultAccountId = profile.default_account_id || null;
            selectedAccountId = sortedSecuritiesAccounts.find(a => a.id == defaultAccountId) ? defaultAccountId : null;
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Vendi Titolo</h2>
            <form id="sell-security-form">
                <div class="form-group">
                    <label>Account</label>
                    <select name="account_id" id="sell-account-select" required>
                        ${sortedSecuritiesAccounts.length === 1 ? '' : '<option value="">Seleziona...</option>'}
                        ${sortedSecuritiesAccounts.map(a => `<option value="${a.id}" ${a.id == selectedAccountId ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Titolo da Vendere</label>
                    <select name="symbol" id="sell-symbol-select" required>
                        <option value="">Seleziona...</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Quantità Disponibile</label>
                    <div id="available-quantity" style="padding: 8px; background: #f5f5f5; border-radius: 4px; color: #666;">
                        Seleziona un titolo
                    </div>
                </div>
                <div class="form-group">
                    <label>Prezzo Medio di Acquisto</label>
                    <div id="mean-purchase-price" style="padding: 8px; background: #f5f5f5; border-radius: 4px; color: #666;">
                        -
                    </div>
                </div>
                <div class="form-group">
                    <label>Quantità da Vendere</label>
                    <input type="number" step="0.0001" name="quantity" id="sell-quantity" required>
                </div>
                <div class="form-group">
                    <label>Prezzo Totale di Vendita</label>
                    <input type="number" step="0.01" name="purchase_price" id="sell-price" required>
                    <small style="color: #666; font-size: 0.9em;">Importo totale ricevuto dalla vendita</small>
                </div>
                <div class="form-group">
                    <label>Plus/Minus Valenza</label>
                    <div id="profit-loss" style="padding: 8px; background: #f5f5f5; border-radius: 4px; font-weight: bold;">
                        -
                    </div>
                </div>
                <div class="form-group">
                    <label>Data Vendita</label>
                    <input type="date" name="purchase_date" value="${getTodayDate()}" required>
                </div>
                <div class="form-group">
                    <label>Valuta</label>
                    <input type="text" name="currency" id="sell-currency" value="EUR" maxlength="3" required>
                </div>
                <div class="form-group">
                    <label>Note</label>
                    <textarea name="notes"></textarea>
                </div>
                <div class="form-group">
                    <label>Allegati</label>
                    <div id="sell-security-attachments-list" style="margin-bottom: 10px;"></div>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="btn btn-secondary" onclick="addSecurityAttachmentFile('sell-security')">Aggiungi Foto/Documento</button>
                        <button type="button" class="btn btn-secondary" onclick="addSecurityAttachmentLink('sell-security')">Aggiungi Link</button>
                    </div>
                    <input type="file" id="sell-security-file-input" style="display: none;" accept="image/*,.pdf,.doc,.docx" multiple>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Conferma Vendita</button>
                </div>
            </form>
        `;
        
        // Populate symbols based on selected account
        const accountSelect = document.getElementById('sell-account-select');
        const symbolSelect = document.getElementById('sell-symbol-select');
        const availableQuantityDiv = document.getElementById('available-quantity');
        const meanPriceDiv = document.getElementById('mean-purchase-price');
        const quantityInput = document.getElementById('sell-quantity');
        const priceInput = document.getElementById('sell-price');
        const currencyInput = document.getElementById('sell-currency');
        const profitLossDiv = document.getElementById('profit-loss');
        
        function updateSymbols() {
            const accountId = parseInt(accountSelect.value);
            const filteredSecurities = availableSecuritiesList.filter(sec => sec.account_id === accountId)
                .sort((a, b) => a.symbol.localeCompare(b.symbol, 'it', { sensitivity: 'base' }));
            
            symbolSelect.innerHTML = '<option value="">Seleziona...</option>';
            filteredSecurities.forEach(sec => {
                symbolSelect.innerHTML += `<option value="${escapeHtml(sec.symbol)}" data-quantity="${sec.total_quantity}" data-mean-price="${sec.mean_price}" data-currency="${sec.currency}">${escapeHtml(sec.symbol)}</option>`;
            });
            
            availableQuantityDiv.textContent = 'Seleziona un titolo';
            meanPriceDiv.textContent = '-';
            quantityInput.value = '';
            priceInput.value = '';
            profitLossDiv.textContent = '-';
        }
        
        function updateProfitLoss() {
            const selectedOption = symbolSelect.options[symbolSelect.selectedIndex];
            if (!selectedOption || !selectedOption.value) {
                availableQuantityDiv.textContent = 'Seleziona un titolo';
                meanPriceDiv.textContent = '-';
                profitLossDiv.textContent = '-';
                return;
            }
            
            const availableQuantity = parseFloat(selectedOption.dataset.quantity);
            const meanPrice = parseFloat(selectedOption.dataset.meanPrice);
            const currency = selectedOption.dataset.currency;
            
            availableQuantityDiv.textContent = `${availableQuantity.toFixed(4)}`;
            meanPriceDiv.textContent = formatCurrency(meanPrice, currency);
            currencyInput.value = currency;
            
            const quantity = parseFloat(quantityInput.value) || 0;
            const price = parseFloat(priceInput.value) || 0;
            
            if (quantity > 0 && price > 0) {
                const unitPrice = price / quantity;
                const profitLoss = unitPrice - meanPrice;
                const profitLossTotal = profitLoss * quantity;
                const profitLossClass = profitLossTotal >= 0 ? 'profit' : 'loss';
                profitLossDiv.innerHTML = `<span class="${profitLossClass}" style="color: ${profitLossTotal >= 0 ? '#2e7d32' : '#d32f2f'};">
                    ${profitLossTotal >= 0 ? '+' : ''}${formatCurrency(profitLossTotal, currency)}
                </span>`;
            } else {
                profitLossDiv.textContent = '-';
            }
        }
        
        // Initialize attachments array for this form
        window.currentSecurityAttachments = [];
        
        document.getElementById('sell-security-file-input').addEventListener('change', (e) => {
            Array.from(e.target.files).forEach(file => {
                window.currentSecurityAttachments.push({
                    type: file.type.startsWith('image/') ? 'PHOTO' : 'DOCUMENT',
                    file: file,
                    tempId: 'temp_' + Date.now() + '_' + Math.random()
                });
            });
            updateSecurityAttachmentsList('sell-security');
            e.target.value = ''; // Reset input
        });
        
        accountSelect.addEventListener('change', updateSymbols);
        symbolSelect.addEventListener('change', () => {
            updateProfitLoss();
            const selectedOption = symbolSelect.options[symbolSelect.selectedIndex];
            if (selectedOption && selectedOption.value) {
                const availableQuantity = parseFloat(selectedOption.dataset.quantity);
                quantityInput.max = availableQuantity;
            }
        });
        quantityInput.addEventListener('input', updateProfitLoss);
        priceInput.addEventListener('input', updateProfitLoss);
        
        // Initialize
        updateSymbols();
        
        // Setup decimal separator normalization for quantity and purchase_price fields
        const sellQuantityInput = document.getElementById('sell-quantity');
        const sellPriceInput = document.getElementById('sell-price');
        if (sellQuantityInput) setupDecimalNormalization(sellQuantityInput);
        if (sellPriceInput) setupDecimalNormalization(sellPriceInput);
        
        document.getElementById('sell-security-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const quantity = parseFloat(normalizeDecimalSeparator(formData.get('quantity')));
            const accountId = parseInt(formData.get('account_id'));
            const symbol = formData.get('symbol');
            
            // Check available quantity
            const selectedSec = availableSecuritiesList.find(s => s.account_id === accountId && s.symbol === symbol);
            if (!selectedSec || quantity > selectedSec.total_quantity) {
                alert(`Quantità non disponibile. Disponibile: ${selectedSec ? selectedSec.total_quantity.toFixed(4) : 0}`);
                return;
            }
            
            const data = {
                account_id: accountId,
                symbol: symbol,
                quantity: quantity,
                purchase_price: parseFloat(normalizeDecimalSeparator(formData.get('purchase_price'))),
                purchase_date: formData.get('purchase_date'),
                currency: formData.get('currency'),
                transaction_type: 'SALE',
                notes: formData.get('notes') || null
            };
            
            try {
                const result = await API.createSecurity(data);
                const securityId = result.id;
                
                // Upload attachments
                if (window.currentSecurityAttachments && window.currentSecurityAttachments.length > 0) {
                    for (const attachment of window.currentSecurityAttachments) {
                        if (attachment.type === 'PHOTO' || attachment.type === 'DOCUMENT') {
                            await API.addSecurityAttachment(securityId, attachment.type, attachment.file);
                        } else if (attachment.type === 'LINK') {
                            await API.addSecurityAttachment(securityId, 'LINK', null, attachment.link_url, attachment.link_text);
                        }
                    }
                }
                
                window.currentSecurityAttachments = [];
                closeModal();
                loadSecurities();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        console.error('Error showing sell security modal:', error);
        alert('Errore nel caricamento: ' + error.message);
    }
}

// Prepaid Cards
async function loadPrepaidCards() {
    try {
        // Reset button text if showing archived cards
        const showArchivedCardsBtn = document.getElementById('show-archived-cards-btn');
        if (showArchivedCardsBtn) {
            showArchivedCardsBtn.textContent = 'Carte Archiviate';
        }
        
        const cards = await API.getPrepaidCards();
        const container = document.getElementById('prepaid-cards-list');
        
        container.innerHTML = cards.map(card => {
            const isShared = card.card_status === 'shared';
            const isOwned = card.card_status === 'owned';
            const ownerInfo = card.owner_name || card.owner_email ? `Condiviso da ${card.owner_name || card.owner_email}` : '';
            const hasShares = card.has_shares === 1 || card.has_shares === true;
            
            const nominalAmountDisplay = card.nominal_amount !== null && card.nominal_amount !== undefined 
                ? `Importo: ${formatCurrency(card.nominal_amount, card.base_currency)}` 
                : '';
            const remainingBalanceDisplay = card.remaining_balance !== null && card.remaining_balance !== undefined
                ? `Credito residuo: ${formatCurrency(card.remaining_balance, card.base_currency)}`
                : '';
            const amountLine = nominalAmountDisplay && remainingBalanceDisplay
                ? `${nominalAmountDisplay} • ${remainingBalanceDisplay}<br>`
                : nominalAmountDisplay || remainingBalanceDisplay
                    ? `${nominalAmountDisplay || remainingBalanceDisplay}<br>`
                    : '';
            
            return `
            <div class="card" style="position: relative; cursor: pointer;" onclick="window.showPrepaidCardFullscreen(${card.id})">
                <div class="card-header">
                    <div class="card-title">
                        ${escapeHtml(card.name)}
                        ${isShared ? `<span style="color: #666; font-size: 0.9em; margin-left: 0.5rem;">(${ownerInfo})</span>` : ''}
                    </div>
                </div>
                <div class="card-meta">
                    Numero: ${card.card_number || 'N/A'}<br>
                    Valuta: ${card.base_currency}<br>
                    ${amountLine}
                    Aperta: ${formatDate(card.opening_date)}
                    ${card.expire_date ? `<br>Scade: ${formatDate(card.expire_date)}` : ''}
                </div>
                ${card.photo_path ? `<img src="/uploads/${card.photo_path}" style="max-width: 100%; margin-top: 1rem;">` : ''}
                <div style="position: absolute; bottom: 10px; left: 10px; right: 10px; display: flex; justify-content: space-between; align-items: flex-end;">
                    <div style="display: flex; gap: 10px;">
                        <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s;" 
                              onclick="event.stopPropagation(); window.filterByPrepaidCard(${card.id})" 
                              onmouseover="this.style.color='#007bff'" 
                              onmouseout="this.style.color='#666'"
                              title="Vedi movimenti">📋</span>
                    </div>
                    <div style="display: flex; gap: 10px; flex-direction: column; align-items: flex-end;">
                        <div style="display: flex; gap: 10px;">
                            ${card.card_status === 'owned' ? `
                                <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s;" 
                                      onclick="event.stopPropagation(); showPrepaidCardSharesModal(${card.id})" 
                                      onmouseover="this.style.color='#007bff'" 
                                      onmouseout="this.style.color='#666'"
                                      title="${hasShares ? 'Condivisioni attive - Gestisci condivisioni' : 'Gestisci condivisioni'}">${hasShares ? '🔗👤' : '🔗'}</span>
                            ` : ''}
                            <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s;" 
                                  onclick="event.stopPropagation(); showEditPrepaidCardModal(${card.id})" 
                                  onmouseover="this.style.color='#007bff'" 
                                  onmouseout="this.style.color='#666'"
                                  title="Modifica carta">✏️</span>
                            ${card.card_status === 'owned' ? `
                                <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s;" 
                                      onclick="event.stopPropagation(); deletePrepaidCard(${card.id})" 
                                      onmouseover="this.style.color='#dc3545'" 
                                      onmouseout="this.style.color='#666'"
                                      title="Elimina carta">🗑️</span>
                            ` : ''}
                        </div>
                        ${isShared ? `
                            <button class="btn btn-secondary" onclick="event.stopPropagation(); stopSharingPrepaidCard(${card.id})" style="padding: 0.25rem 0.5rem; font-size: 0.85rem; margin-top: 0.5rem;">Rimuovi Condivisione</button>
                        ` : ''}
                    </div>
                </div>
            </div>
            `;
        }).join('');
    } catch (error) {
        console.error('Error loading prepaid cards:', error);
    }
}

async function showAddPrepaidCardModal() {
    const modalBody = document.getElementById('modal-body');
    
    // Get today's date in YYYY-MM-DD format
    const today = new Date();
    const todayStr = today.toISOString().split('T')[0];
    
    modalBody.innerHTML = `
        <h2>Aggiungi Carta Prepagata</h2>
        <form id="add-card-form">
            <div class="form-group">
                <label>Nome</label>
                <input type="text" name="name" required>
            </div>
            <div class="form-group">
                <label>Numero Carta</label>
                <input type="text" name="card_number" placeholder="Alfanumerico (es. ABC123)">
            </div>
            <div class="form-group">
                <label>Valuta Base</label>
                <input type="text" name="base_currency" value="EUR" maxlength="3" required>
            </div>
            <div class="form-group">
                <label>Importo Nominale Caricato</label>
                <input type="number" name="nominal_amount" step="0.01" min="0" placeholder="0.00">
            </div>
            <div class="form-group">
                <label>Data Apertura</label>
                <input type="date" name="opening_date" value="${todayStr}" required>
            </div>
            <div class="form-group">
                <label>Data Scadenza</label>
                <input type="date" name="expire_date" required>
            </div>
            <div class="form-group">
                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                    <input type="checkbox" name="enabled" checked>
                    <span>Abilitata</span>
                </label>
            </div>
            <div class="form-group">
                <label>Foto</label>
                <input type="file" name="photo" accept="image/*">
            </div>
            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                <button type="submit" class="btn btn-primary">Salva</button>
            </div>
        </form>
    `;
    
    document.getElementById('add-card-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData(e.target);
        const cardNumber = formData.get('card_number');
        const data = {
            name: formData.get('name'),
            card_number: cardNumber && cardNumber.trim() !== '' ? cardNumber.trim() : '',
            base_currency: formData.get('base_currency'),
            nominal_amount: formData.get('nominal_amount') || null,
            opening_date: formData.get('opening_date'),
            expire_date: formData.get('expire_date') // Now required, so no default to null
        };
        const photo = formData.get('photo').size > 0 ? formData.get('photo') : null;
        
        try {
            await API.createPrepaidCard(data, photo);
            closeModal();
            loadPrepaidCards();
        } catch (error) {
            alert('Errore: ' + error.message);
        }
    });
    
    showModal();
}

async function showEditPrepaidCardModal(cardId) {
    try {
        const cards = await API.getPrepaidCards();
        const card = cards.find(c => c.id === cardId);
        
        if (!card) {
            alert('Carta prepagata non trovata');
            return;
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Modifica Carta Prepagata</h2>
            <form id="edit-card-form">
                <div class="form-group">
                    <label>Nome</label>
                    <input type="text" name="name" value="${escapeHtml(card.name)}" required>
                </div>
                <div class="form-group">
                    <label>Numero Carta</label>
                    <input type="text" name="card_number" value="${escapeHtml(card.card_number || '')}" placeholder="Alfanumerico (es. ABC123)">
                </div>
                <div class="form-group">
                    <label>Valuta Base</label>
                    <input type="text" name="base_currency" value="${escapeHtml(card.base_currency)}" maxlength="3" required>
                </div>
                <div class="form-group">
                    <label>Importo Nominale Caricato</label>
                    <input type="number" name="nominal_amount" step="0.01" min="0" value="${card.nominal_amount || ''}" placeholder="0.00">
                </div>
                <div class="form-group">
                    <label>Data Apertura</label>
                    <input type="date" name="opening_date" value="${card.opening_date}" required>
                </div>
                <div class="form-group">
                    <label>Data Scadenza</label>
                    <input type="date" name="expire_date" value="${card.expire_date}" required>
                </div>
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                        <input type="checkbox" name="enabled" ${card.enabled !== false ? 'checked' : ''}>
                        <span>Abilitata</span>
                    </label>
                </div>
                <div class="form-group">
                    <label>Foto</label>
                    <input type="file" name="photo" accept="image/*">
                    ${card.photo_path ? `
                        <div style="margin-top: 10px;">
                            <img src="/uploads/${card.photo_path}" style="max-width: 200px; max-height: 100px; border-radius: 5px;" alt="Current photo">
                            <div style="margin-top: 10px;">
                                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                                    <input type="checkbox" name="remove_photo" id="remove-photo-checkbox">
                                    <span>Rimuovi foto</span>
                                </label>
                            </div>
                        </div>
                    ` : ''}
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        document.getElementById('edit-card-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const cardNumber = formData.get('card_number');
            const data = {
            name: formData.get('name'),
            card_number: cardNumber && cardNumber.trim() !== '' ? cardNumber.trim() : '',
            base_currency: formData.get('base_currency'),
            nominal_amount: formData.get('nominal_amount') || null,
            opening_date: formData.get('opening_date'),
            expire_date: formData.get('expire_date'),
            enabled: formData.get('enabled') === 'on' ? true : false
        };
            
            if (formData.get('remove_photo') === 'on') {
                data.remove_photo = true;
            }
            
            const photo = formData.get('photo').size > 0 ? formData.get('photo') : null;
            
            try {
                await API.updatePrepaidCard(cardId, data, photo);
                closeModal();
                loadPrepaidCards();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        alert('Errore nel caricamento: ' + error.message);
    }
}

async function deletePrepaidCard(cardId) {
    if (!confirm('Sei sicuro di voler eliminare questa carta prepagata?')) {
        return;
    }
    
    try {
        const result = await API.deletePrepaidCard(cardId);
        if (result.archived) {
            alert('La carta è stata archiviata perché ha movimenti associati. Puoi visualizzarla nella sezione "Carte Archiviate".');
        }
        loadPrepaidCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function loadArchivedPrepaidCards() {
    try {
        // Update button text
        const showArchivedCardsBtn = document.getElementById('show-archived-cards-btn');
        if (showArchivedCardsBtn) {
            showArchivedCardsBtn.textContent = 'Torna a Carte Attive';
        }
        
        const cards = await API.getPrepaidCards({ archived_only: true });
        const container = document.getElementById('prepaid-cards-list');
        
        if (cards.length === 0) {
            container.innerHTML = '<p style="text-align: center; color: #666; padding: 2rem;">Nessuna carta archiviata.</p>';
            return;
        }
        
        container.innerHTML = cards.map(card => {
            const isShared = card.card_status === 'shared';
            const isOwned = card.card_status === 'owned';
            const ownerInfo = card.owner_name || card.owner_email ? `Condiviso da ${card.owner_name || card.owner_email}` : '';
            
            const nominalAmountDisplay = card.nominal_amount !== null && card.nominal_amount !== undefined 
                ? `Importo: ${formatCurrency(card.nominal_amount, card.base_currency)}` 
                : '';
            const remainingBalanceDisplay = card.remaining_balance !== null && card.remaining_balance !== undefined
                ? `Credito residuo: ${formatCurrency(card.remaining_balance, card.base_currency)}`
                : '';
            const amountLine = nominalAmountDisplay && remainingBalanceDisplay
                ? `${nominalAmountDisplay} • ${remainingBalanceDisplay}<br>`
                : nominalAmountDisplay || remainingBalanceDisplay
                    ? `${nominalAmountDisplay || remainingBalanceDisplay}<br>`
                    : '';
            
            return `
            <div class="card" style="position: relative; cursor: pointer; opacity: 0.7;" onclick="window.showPrepaidCardFullscreen(${card.id})">
                <div class="card-header">
                    <div class="card-title">
                        ${escapeHtml(card.name)}
                        <span style="color: #dc3545; font-size: 0.9em; margin-left: 0.5rem;">(Archiviata)</span>
                        ${isShared ? `<span style="color: #666; font-size: 0.9em; margin-left: 0.5rem;">(${ownerInfo})</span>` : ''}
                    </div>
                </div>
                <div class="card-meta">
                    Numero: ${card.card_number || 'N/A'}<br>
                    Valuta: ${card.base_currency}<br>
                    ${amountLine}
                    Aperta: ${formatDate(card.opening_date)}
                    ${card.expire_date ? `<br>Scade: ${formatDate(card.expire_date)}` : ''}
                </div>
                ${card.photo_path ? `<img src="/uploads/${card.photo_path}" style="max-width: 100%; margin-top: 1rem;">` : ''}
                <div style="position: absolute; bottom: 10px; left: 10px; right: 10px; display: flex; justify-content: space-between; align-items: flex-end;">
                    <div style="display: flex; gap: 10px;">
                        <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s;" 
                              onclick="event.stopPropagation(); window.filterByPrepaidCard(${card.id})" 
                              onmouseover="this.style.color='#007bff'" 
                              onmouseout="this.style.color='#666'"
                              title="Vedi movimenti">📋</span>
                    </div>
                    <div style="display: flex; gap: 10px; flex-direction: column; align-items: flex-end;">
                        <div style="display: flex; gap: 10px;">
                            ${card.card_status === 'owned' ? `
                                <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s;" 
                                      onclick="event.stopPropagation(); restorePrepaidCard(${card.id})" 
                                      onmouseover="this.style.color='#28a745'" 
                                      onmouseout="this.style.color='#666'"
                                      title="Ripristina carta">↩️</span>
                            ` : ''}
                        </div>
                    </div>
                </div>
            </div>
            `;
        }).join('');
    } catch (error) {
        console.error('Error loading archived prepaid cards:', error);
        alert('Errore: ' + error.message);
    }
}

async function restorePrepaidCard(cardId) {
    if (!confirm('Vuoi ripristinare questa carta archiviata?')) {
        return;
    }
    
    try {
        await API.archivePrepaidCard(cardId, false);
        loadArchivedPrepaidCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

// Users Management (Admin only)
async function loadUsers() {
    try {
        const users = await API.getAllUsers();
        const container = document.getElementById('users-list');
        
        if (!container) {
            console.error('Users list container not found');
            return;
        }
        
        container.innerHTML = users.map(user => {
            const adminBadge = user.is_admin ? '<span style="background: #667eea; color: white; padding: 2px 8px; border-radius: 3px; font-size: 0.8em; margin-left: 8px;">Admin</span>' : '';
            const blockedBadge = user.is_blocked ? '<span style="background: #dc3545; color: white; padding: 2px 8px; border-radius: 3px; font-size: 0.8em; margin-left: 8px;">Bloccato</span>' : '';
            
            return `
            <div class="list-item">
                <div class="list-item-content">
                    <div class="list-item-title">
                        ${escapeHtml(user.name || user.email)}${adminBadge}${blockedBadge}
                    </div>
                    <div class="list-item-meta">
                        ${escapeHtml(user.email)} • Registrato: ${formatDate(user.created_at)}
                    </div>
                </div>
                <div class="list-item-actions">
                    <button class="btn btn-secondary" onclick="toggleAdminStatus(${user.id}, ${user.is_admin})" ${user.is_admin && users.filter(u => u.is_admin).length === 1 ? 'disabled title="Deve esserci almeno un amministratore"' : ''}>
                        ${user.is_admin ? 'Rimuovi Admin' : 'Rendi Admin'}
                    </button>
                    <button class="btn ${user.is_blocked ? 'btn-secondary' : 'btn-danger'}" onclick="toggleBlockStatus(${user.id}, ${user.is_blocked})">
                        ${user.is_blocked ? 'Sblocca' : 'Blocca'}
                    </button>
                    <button class="btn btn-danger" onclick="showDeleteUserModal(${user.id}, '${escapeHtml(user.name || user.email)}')" ${user.is_admin && users.filter(u => u.is_admin).length === 1 ? 'disabled title="Non puoi eliminare l\'ultimo amministratore"' : ''}>
                        Elimina
                    </button>
                </div>
            </div>
            `;
        }).join('');
        
        if (users.length === 0) {
            container.innerHTML = '<p style="color: #666; text-align: center; padding: 2rem;">Nessun utente trovato</p>';
        }
    } catch (error) {
        console.error('Error loading users:', error);
        const container = document.getElementById('users-list');
        if (container) {
            container.innerHTML = `<p style="color: #d32f2f; text-align: center; padding: 2rem;">Errore nel caricamento: ${escapeHtml(error.message)}</p>`;
        }
    }
}

async function toggleAdminStatus(userId, currentStatus) {
    if (!confirm(`Sei sicuro di voler ${currentStatus ? 'rimuovere' : 'assegnare'} il ruolo di amministratore a questo utente?`)) {
        return;
    }
    
    try {
        await API.updateUser(userId, { is_admin: !currentStatus });
        loadUsers();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function toggleBlockStatus(userId, currentStatus) {
    if (!confirm(`Sei sicuro di voler ${currentStatus ? 'sbloccare' : 'bloccare'} questo utente?`)) {
        return;
    }
    
    try {
        await API.updateUser(userId, { is_blocked: !currentStatus });
        loadUsers();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function showDeleteUserModal(userId, userName) {
    try {
        // Get data counts for the user
        const counts = await API.getUserDataCounts(userId);
        
        // Calculate total data count
        const totalCount = Object.values(counts).reduce((sum, count) => sum + count, 0);
        
        // Build the modal content
        const modalBody = document.getElementById('modal-body');
        
        if (totalCount === 0) {
            // No data, show simple confirmation
            modalBody.innerHTML = `
                <h2 style="margin-top: 0; color: #d32f2f;">Elimina Utente</h2>
                <p style="margin-bottom: 1.5rem;">
                    Stai per eliminare l'utente <strong>${escapeHtml(userName)}</strong>.
                    Questo utente non ha dati associati.
                </p>
                <div class="form-actions">
                    <button class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button class="btn btn-danger" onclick="confirmDeleteUser(${userId}, true)">Elimina Utente</button>
                </div>
            `;
        } else {
            modalBody.innerHTML = `
                <h2 style="margin-top: 0; color: #d32f2f;">Elimina Utente</h2>
                <p style="margin-bottom: 1.5rem;">
                    Stai per eliminare l'utente <strong>${escapeHtml(userName)}</strong>.
                    Di seguito sono elencati i dati associati a questo utente. 
                    Seleziona quali tipologie di dati vuoi eliminare prima di procedere.
                </p>
                
                <div style="margin-bottom: 1.5rem;">
                    <h3 style="font-size: 1.1rem; margin-bottom: 1rem; color: #333;">Dati disponibili:</h3>
                    <div style="display: flex; flex-direction: column; gap: 0.75rem;">
                    ${counts.accounts > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-accounts" checked>
                            <span><strong>Account</strong> (${counts.accounts})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.prepaid_cards > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-prepaid-cards" checked>
                            <span><strong>Carte Prepagate</strong> (${counts.prepaid_cards})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.fidelity_cards > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-fidelity-cards" checked>
                            <span><strong>Carte Fedeltà</strong> (${counts.fidelity_cards})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.expenses > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-expenses" checked>
                            <span><strong>Movimenti</strong> (${counts.expenses})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.securities > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-securities" checked>
                            <span><strong>Titoli</strong> (${counts.securities})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.fund_transfers > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-fund-transfers" checked>
                            <span><strong>Trasferimenti</strong> (${counts.fund_transfers})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.expense_attachments > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-expense-attachments" checked>
                            <span><strong>Allegati Movimenti</strong> (${counts.expense_attachments})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.securities_attachments > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-securities-attachments" checked>
                            <span><strong>Allegati Titoli</strong> (${counts.securities_attachments})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.transfer_attachments > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-transfer-attachments" checked>
                            <span><strong>Allegati Trasferimenti</strong> (${counts.transfer_attachments})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.account_shares > 0 || counts.account_share_invitations > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-account-shares" checked>
                            <span><strong>Condivisioni Account</strong> (${counts.account_shares + counts.account_share_invitations})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.prepaid_card_shares > 0 || counts.prepaid_card_share_invitations > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-prepaid-card-shares" checked>
                            <span><strong>Condivisioni Carte Prepagate</strong> (${counts.prepaid_card_shares + counts.prepaid_card_share_invitations})</span>
                        </label>
                    ` : ''}
                    
                    ${counts.fidelity_card_shares > 0 || counts.fidelity_card_share_invitations > 0 ? `
                        <label class="checkbox-group" style="padding: 0.75rem; background: #f8f9fa; border-radius: 5px;">
                            <input type="checkbox" id="delete-fidelity-card-shares" checked>
                            <span><strong>Condivisioni Carte Fedeltà</strong> (${counts.fidelity_card_shares + counts.fidelity_card_share_invitations})</span>
                        </label>
                    ` : ''}
                    </div>
                </div>
                
                <div class="form-actions">
                    <button class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button class="btn btn-danger" onclick="confirmDeleteUser(${userId}, false)">Elimina Utente</button>
                </div>
            `;
        }
        
        showModal();
    } catch (error) {
        alert('Errore nel caricamento dei dati: ' + error.message);
    }
}

async function confirmDeleteUser(userId, noData) {
    let deleteFlags = {};
    
    if (!noData) {
        // Collect all checked flags
        deleteFlags = {
            accounts: document.getElementById('delete-accounts')?.checked || false,
            prepaid_cards: document.getElementById('delete-prepaid-cards')?.checked || false,
            fidelity_cards: document.getElementById('delete-fidelity-cards')?.checked || false,
            expenses: document.getElementById('delete-expenses')?.checked || false,
            securities: document.getElementById('delete-securities')?.checked || false,
            fund_transfers: document.getElementById('delete-fund-transfers')?.checked || false,
            expense_attachments: document.getElementById('delete-expense-attachments')?.checked || false,
            securities_attachments: document.getElementById('delete-securities-attachments')?.checked || false,
            transfer_attachments: document.getElementById('delete-transfer-attachments')?.checked || false,
            account_shares: document.getElementById('delete-account-shares')?.checked || false,
            prepaid_card_shares: document.getElementById('delete-prepaid-card-shares')?.checked || false,
            fidelity_card_shares: document.getElementById('delete-fidelity-card-shares')?.checked || false
        };
        
        // Check if at least one flag is set
        const hasAnyFlag = Object.values(deleteFlags).some(flag => flag === true);
        
        if (!hasAnyFlag) {
            alert('Devi selezionare almeno una tipologia di dati da eliminare.');
            return;
        }
    } else {
        // No data, set a dummy flag to satisfy backend validation
        deleteFlags = { confirm: true };
    }
    
    if (!confirm('Sei sicuro di voler eliminare questo utente? Questa operazione non può essere annullata.')) {
        return;
    }
    
    try {
        await API.deleteUser(userId, deleteFlags);
        closeModal();
        loadUsers();
        alert('Utente eliminato con successo.');
    } catch (error) {
        alert('Errore durante l\'eliminazione: ' + error.message);
    }
}

// Make functions globally accessible
window.toggleAdminStatus = toggleAdminStatus;
window.toggleBlockStatus = toggleBlockStatus;
window.showDeleteUserModal = showDeleteUserModal;
window.confirmDeleteUser = confirmDeleteUser;

// Fidelity Card Brands Management (Admin only)
async function loadBrands() {
    try {
        const brands = await API.getFidelityCardBrands();
        const container = document.getElementById('brands-list');
        
        if (!container) {
            console.error('Brands list container not found');
            return;
        }
        
        if (brands.length === 0) {
            container.innerHTML = '<p style="text-align: center; color: #666; padding: 2rem;">Nessuna insegna configurata. Clicca su "Aggiungi Insegna" per crearne una.</p>';
            return;
        }
        
        container.innerHTML = brands.map(brand => {
            return `
            <div class="list-item">
                <div class="list-item-content">
                    <div class="list-item-title">
                        ${escapeHtml(brand.name)}
                    </div>
                    <div class="list-item-meta">
                        ${brand.logo_path ? `<img src="/uploads/${brand.logo_path}" style="max-width: 100px; max-height: 50px; margin-top: 0.5rem; border-radius: 4px;">` : 'Nessun logo'}
                    </div>
                </div>
                <div class="list-item-actions">
                    <button class="btn btn-secondary" onclick="showEditBrandModal(${brand.id})">Modifica</button>
                    <button class="btn btn-danger" onclick="deleteBrand(${brand.id}, '${escapeHtml(brand.name)}')">Elimina</button>
                </div>
            </div>
            `;
        }).join('');
    } catch (error) {
        console.error('Error loading brands:', error);
        alert('Errore: ' + error.message);
    }
}

async function showAddBrandModal() {
    const modalBody = document.getElementById('modal-body');
    
    modalBody.innerHTML = `
        <h2>Aggiungi Insegna</h2>
        <form id="add-brand-form">
            <div class="form-group">
                <label>Nome Insegna</label>
                <input type="text" name="name" required placeholder="Es. Coop, Esselunga, etc.">
            </div>
            <div class="form-group">
                <label>Logo</label>
                <input type="file" name="logo" accept="image/*" required>
                <small style="color: #666; display: block; margin-top: 0.25rem;">Carica il logo dell'insegna</small>
            </div>
            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                <button type="submit" class="btn btn-primary">Salva</button>
            </div>
        </form>
    `;
    
    document.getElementById('add-brand-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData(e.target);
        
        const data = {
            name: formData.get('name')
        };
        
        const logo = formData.get('logo') && formData.get('logo').size > 0 ? formData.get('logo') : null;
        
        if (!logo) {
            alert('Il logo è obbligatorio');
            return;
        }
        
        try {
            await API.createFidelityCardBrand(data, logo);
            closeModal();
            loadBrands();
        } catch (error) {
            alert('Errore: ' + error.message);
        }
    });
    
    showModal();
}

async function showEditBrandModal(brandId) {
    try {
        const brand = await API.getFidelityCardBrand(brandId);
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Modifica Insegna</h2>
            <form id="edit-brand-form">
                <div class="form-group">
                    <label>Nome Insegna</label>
                    <input type="text" name="name" value="${escapeHtml(brand.name)}" required>
                </div>
                <div class="form-group">
                    <label>Logo Attuale</label>
                    ${brand.logo_path ? `<img src="/uploads/${brand.logo_path}" style="max-width: 200px; max-height: 100px; display: block; margin-bottom: 0.5rem; border-radius: 4px;">` : 'Nessun logo'}
                </div>
                <div class="form-group">
                    <label>Nuovo Logo (opzionale)</label>
                    <input type="file" name="logo" accept="image/*">
                    <small style="color: #666; display: block; margin-top: 0.25rem;">Lascia vuoto per mantenere il logo attuale</small>
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" name="remove_logo"> Rimuovi logo
                    </label>
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        document.getElementById('edit-brand-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            const data = {
                name: formData.get('name'),
                remove_logo: formData.get('remove_logo') === 'on'
            };
            
            const logo = formData.get('logo') && formData.get('logo').size > 0 ? formData.get('logo') : null;
            
            try {
                await API.updateFidelityCardBrand(brandId, data, logo);
                closeModal();
                loadBrands();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function deleteBrand(brandId, brandName) {
    if (!confirm(`Sei sicuro di voler eliminare l'insegna "${brandName}"?`)) {
        return;
    }
    
    try {
        await API.deleteFidelityCardBrand(brandId);
        loadBrands();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

// Make functions globally accessible
window.showAddBrandModal = showAddBrandModal;
window.showEditBrandModal = showEditBrandModal;
window.deleteBrand = deleteBrand;

// Profile
async function loadProfile() {
    try {
        const profile = await API.getProfile();
        const accounts = await API.getAccounts();
        // Sort accounts alphabetically
        const sortedAccounts = [...accounts].sort((a, b) => a.name.localeCompare(b.name, 'it', { sensitivity: 'base' }));
        const container = document.getElementById('profile-form');
        
        container.innerHTML = `
            <form id="profile-form">
                <div class="form-group">
                    <label>Email</label>
                    <input type="email" value="${escapeHtml(profile.email)}" disabled>
                </div>
                <div class="form-group">
                    <label>Valuta Predefinita</label>
                    <input type="text" name="default_currency" value="${escapeHtml(profile.default_currency || 'EUR')}" maxlength="3">
                </div>
                <div class="form-group">
                    <label>Account Predefinito</label>
                    <select name="default_account_id">
                        <option value="">Nessuno</option>
                        ${sortedAccounts.map(a => `<option value="${a.id}" ${a.id == profile.default_account_id ? 'selected' : ''}>${escapeHtml(a.name)}</option>`).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>Vista Predefinita</label>
                    <select name="default_view">
                        <option value="dashboard" ${(profile.default_view || 'dashboard') === 'dashboard' ? 'selected' : ''}>Dashboard</option>
                        <option value="expenses" ${(profile.default_view || 'dashboard') === 'expenses' ? 'selected' : ''}>Spese</option>
                        <option value="securities" ${(profile.default_view || 'dashboard') === 'securities' ? 'selected' : ''}>Titoli</option>
                        <option value="prepaid-cards" ${(profile.default_view || 'dashboard') === 'prepaid-cards' ? 'selected' : ''}>Carte Prepagate</option>
                        <option value="fidelity-cards" ${(profile.default_view || 'dashboard') === 'fidelity-cards' ? 'selected' : ''}>Carte Fedeltà</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Giorni Anticipo per Segnalazione Scadenze</label>
                    <input type="number" name="expiration_notification_days" value="${profile.expiration_notification_days !== null && profile.expiration_notification_days !== undefined ? profile.expiration_notification_days : 7}" min="0" max="365">
                    <small style="color: #666; display: block; margin-top: 0.25rem;">Imposta a 0 per disabilitare le notifiche di scadenza. Default: 7 giorni.</small>
                </div>
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
            <div style="margin-top: 2rem; padding-top: 2rem; border-top: 1px solid #ddd;">
                <h3 style="margin-bottom: 1rem;">Esporta/Importa Dati</h3>
                <div style="display: flex; gap: 1rem; flex-wrap: wrap;">
                    <button type="button" class="btn btn-secondary" onclick="exportUserData()">Esporta Dati</button>
                    <button type="button" class="btn btn-secondary" onclick="document.getElementById('import-file-input').click()">Importa Dati</button>
                    <input type="file" id="import-file-input" style="display: none;" accept=".zip" onchange="importUserData(event)">
                </div>
                <p style="margin-top: 1rem; color: #666; font-size: 0.9rem;">
                    Esporta tutti i tuoi dati (account, spese, allegati, carte, ecc.) in un file ZIP che contiene il JSON con i dati e tutti i file allegati. Il file può essere importato in una nuova istanza dell'applicazione.
                </p>
            </div>
        `;
        
        // Remove any existing event listeners by cloning and replacing the form
        const form = document.getElementById('profile-form');
        const newForm = form.cloneNode(true);
        form.parentNode.replaceChild(newForm, form);
        
        document.getElementById('profile-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const data = {
                default_currency: formData.get('default_currency'),
                default_account_id: formData.get('default_account_id') ? parseInt(formData.get('default_account_id')) : null,
                default_view: formData.get('default_view') || 'dashboard',
                expiration_notification_days: parseInt(formData.get('expiration_notification_days')) || 0
            };
            
            try {
                await API.updateProfile(data);
                alert('Profilo aggiornato');
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
    } catch (error) {
        console.error('Error loading profile:', error);
    }
}

async function exportUserData() {
    try {
        await API.exportData();
        alert('Esportazione completata! Il file è stato scaricato.');
    } catch (error) {
        alert('Errore durante l\'esportazione: ' + error.message);
    }
}

async function importUserData(event) {
    const file = event.target.files[0];
    if (!file) {
        return;
    }
    
    if (!confirm('Sei sicuro di voler importare i dati? Questo aggiungerà i dati importati ai dati esistenti.')) {
        event.target.value = ''; // Reset file input
        return;
    }
    
    try {
        const result = await API.importData(file);
        let message = `Importazione completata!\n` +
              `Account: ${result.imported.accounts}\n` +
              `Spese: ${result.imported.expenses}\n` +
              `Trasferimenti: ${result.imported.transfers}\n` +
              `Titoli: ${result.imported.securities}\n` +
              `Carte prepagate: ${result.imported.prepaid_cards}`;
        
        if (result.skipped && result.skipped.accounts > 0) {
            message += `\n\nAccount saltati (nome già esistente): ${result.skipped.accounts}`;
        }
        
        alert(message);
        
        // Reload all data
        loadAccounts();
        loadExpenses();
        loadDashboard();
        loadProfile();
    } catch (error) {
        alert('Errore durante l\'importazione: ' + error.message);
    } finally {
        event.target.value = ''; // Reset file input
    }
}

// App Title
async function setupAppTitleClick() {
    const appTitle = document.getElementById('app-title');
    if (appTitle) {
        appTitle.addEventListener('click', async () => {
            try {
                // Get user profile to find default view
                const profile = await API.getProfile();
                const defaultView = profile.default_view || 'dashboard';
                switchView(defaultView);
            } catch (error) {
                console.error('Error loading default view:', error);
                // Fallback to dashboard if error
                switchView('dashboard');
            }
        });
    }
}

// User Menu
let userMenuOpen = false;
let notificationsDropdownOpen = false;

function setupUserMenuListeners() {
    const userName = document.getElementById('user-name');
    const userMenuDropdown = document.getElementById('user-menu-dropdown');
    const userMenuProfile = document.getElementById('user-menu-profile');
    const userMenuNotifications = document.getElementById('user-menu-notifications');
    const userMenuAdmin = document.getElementById('user-menu-admin');
    const adminSubmenu = document.getElementById('admin-submenu');
    const userMenuUsers = document.getElementById('user-menu-users');
    
    if (userName) {
        userName.addEventListener('click', (e) => {
            e.stopPropagation();
            userMenuOpen = !userMenuOpen;
            userMenuDropdown.style.display = userMenuOpen ? 'block' : 'none';
            // Close submenus when closing main menu
            if (!userMenuOpen) {
                if (adminSubmenu) adminSubmenu.style.display = 'none';
                const notificationsDropdown = document.getElementById('notifications-dropdown');
                if (notificationsDropdown) notificationsDropdown.style.display = 'none';
            }
        });
    }
    
    if (userMenuProfile) {
        userMenuProfile.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            userMenuOpen = false;
            userMenuDropdown.style.display = 'none';
            switchView('profile');
        });
    }
    
    if (userMenuNotifications) {
        // Note: mouseenter handler is also set up in setupNotificationsListeners()
        // This click handler navigates to notifications page
        userMenuNotifications.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            userMenuOpen = false;
            userMenuDropdown.style.display = 'none';
            // Close all submenus
            if (adminSubmenu) adminSubmenu.style.display = 'none';
            const notificationsDropdown = document.getElementById('notifications-dropdown');
            if (notificationsDropdown) notificationsDropdown.style.display = 'none';
            switchView('notifications');
        });
    }
    
    if (userMenuAdmin && adminSubmenu) {
        let adminSubmenuOpen = false;
        
        // Function to position submenu based on available space
        const positionAdminSubmenu = () => {
            const menuRect = userMenuDropdown.getBoundingClientRect();
            const submenuWidth = 180; // min-width from CSS
            const spaceOnLeft = menuRect.left;
            const spaceOnRight = window.innerWidth - menuRect.right;
            
            // If there's more space on the right or we're on mobile (small screen), open to the right
            if (spaceOnRight >= submenuWidth || window.innerWidth <= 768) {
                adminSubmenu.style.right = 'auto';
                adminSubmenu.style.left = '100%';
                adminSubmenu.style.marginLeft = '0.5rem';
                adminSubmenu.style.marginRight = '0';
            } else {
                // Otherwise open to the left
                adminSubmenu.style.left = 'auto';
                adminSubmenu.style.right = '100%';
                adminSubmenu.style.marginRight = '0.5rem';
                adminSubmenu.style.marginLeft = '0';
            }
        };
        
        // Toggle submenu on click (for mobile and desktop)
        userMenuAdmin.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            adminSubmenuOpen = !adminSubmenuOpen;
            if (adminSubmenuOpen) {
                positionAdminSubmenu();
            }
            const newDisplay = adminSubmenuOpen ? 'block' : 'none';
            adminSubmenu.style.display = newDisplay;
            // Close notifications dropdown if open
            const notificationsDropdown = document.getElementById('notifications-dropdown');
            if (notificationsDropdown) notificationsDropdown.style.display = 'none';
        });
        
        // Show submenu on mouseenter (for desktop)
        userMenuAdmin.addEventListener('mouseenter', (e) => {
            e.stopPropagation();
            positionAdminSubmenu();
            adminSubmenu.style.display = 'block';
            adminSubmenuOpen = true;
            // Close notifications dropdown if open
            const notificationsDropdown = document.getElementById('notifications-dropdown');
            if (notificationsDropdown) notificationsDropdown.style.display = 'none';
        });
        
        // Keep submenu open when hovering over it
        adminSubmenu.addEventListener('mouseenter', () => {
            adminSubmenu.style.display = 'block';
            adminSubmenuOpen = true;
        });
        
        // Close submenu when mouse leaves both elements
        const closeAdminSubmenu = () => {
            setTimeout(() => {
                if (!userMenuAdmin.matches(':hover') && !adminSubmenu.matches(':hover')) {
                    adminSubmenu.style.display = 'none';
                    adminSubmenuOpen = false;
                }
            }, 200);
        };
        
        userMenuAdmin.addEventListener('mouseleave', closeAdminSubmenu);
        adminSubmenu.addEventListener('mouseleave', closeAdminSubmenu);
        
        // Reposition on window resize
        window.addEventListener('resize', () => {
            if (adminSubmenuOpen && adminSubmenu.style.display === 'block') {
                positionAdminSubmenu();
            }
        });
    }
    
    if (userMenuUsers) {
        userMenuUsers.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            userMenuOpen = false;
            userMenuDropdown.style.display = 'none';
            if (adminSubmenu) adminSubmenu.style.display = 'none';
            switchView('users');
        });
    }
    
    const userMenuBrands = document.getElementById('user-menu-brands');
    if (userMenuBrands) {
        userMenuBrands.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            userMenuOpen = false;
            userMenuDropdown.style.display = 'none';
            if (adminSubmenu) adminSubmenu.style.display = 'none';
            switchView('brands');
        });
    }
    
    // Logout
    const userMenuLogout = document.getElementById('user-menu-logout');
    if (userMenuLogout) {
        userMenuLogout.addEventListener('click', async (e) => {
            e.preventDefault();
            e.stopPropagation();
            userMenuOpen = false;
            userMenuDropdown.style.display = 'none';
            
            logSessionDebug('logout_called', {
                user_id: currentUser?.user_id,
                email: currentUser?.email
            });
            try {
                await API.logout();
                logSessionDebug('logout_success', {
                    user_id: currentUser?.user_id
                });
                // Clear saved view on logout
                localStorage.removeItem('lastView');
                showLoginScreen();
            } catch (error) {
                logSessionDebug('logout_error', {
                    error: error.message,
                    error_stack: error.stack,
                    user_id: currentUser?.user_id
                });
                console.error('Logout error:', error);
                // Clear saved view even if logout API fails
                localStorage.removeItem('lastView');
                // Still show login screen even if logout API fails
                showLoginScreen();
            }
        });
    }
    
    // Close user menu when clicking outside
    document.addEventListener('click', (e) => {
        if (userMenuOpen) {
            const clickedInMenu = userName.contains(e.target) || 
                                  userMenuDropdown.contains(e.target) ||
                                  (adminSubmenu && adminSubmenu.contains(e.target));
            
            if (!clickedInMenu) {
                userMenuOpen = false;
                userMenuDropdown.style.display = 'none';
                // Also close notifications dropdown if open
                const notificationsDropdown = document.getElementById('notifications-dropdown');
                if (notificationsDropdown) {
                    notificationsDropdown.style.display = 'none';
                    notificationsDropdownOpen = false;
                }
                // Also close admin submenu if open
                if (adminSubmenu) {
                    adminSubmenu.style.display = 'none';
                }
            }
        }
    });
}

// Notifications
function setupNotificationsListeners() {
    const notificationsDropdown = document.getElementById('notifications-dropdown');
    const markAllReadBtn = document.getElementById('mark-all-read-btn');
    const viewAllLink = document.getElementById('view-all-notifications-link');
    const userMenuNotifications = document.getElementById('user-menu-notifications');
    
    // Keep notifications dropdown open when hovering over it
    if (notificationsDropdown && userMenuNotifications) {
        // Function to position notifications dropdown based on available space
        const positionNotificationsDropdown = () => {
            const menuRect = userMenuDropdown.getBoundingClientRect();
            const dropdownWidth = 300; // min-width from inline style
            const spaceOnLeft = menuRect.left;
            const spaceOnRight = window.innerWidth - menuRect.right;
            
            // If there's more space on the right or we're on mobile (small screen), open to the right
            if (spaceOnRight >= dropdownWidth || window.innerWidth <= 768) {
                notificationsDropdown.style.right = 'auto';
                notificationsDropdown.style.left = '100%';
                notificationsDropdown.style.marginLeft = '0.5rem';
                notificationsDropdown.style.marginRight = '0';
            } else {
                // Otherwise open to the left
                notificationsDropdown.style.left = 'auto';
                notificationsDropdown.style.right = '100%';
                notificationsDropdown.style.marginRight = '0.5rem';
                notificationsDropdown.style.marginLeft = '0';
            }
        };
        
        // Show dropdown when hovering over notifications menu item
        userMenuNotifications.addEventListener('mouseenter', () => {
            positionNotificationsDropdown();
            notificationsDropdown.style.display = 'block';
            notificationsDropdownOpen = true;
            loadNotificationsDropdown();
            // Close admin submenu if open
            const adminSubmenu = document.getElementById('admin-submenu');
            if (adminSubmenu) adminSubmenu.style.display = 'none';
        });
        
        // Keep dropdown open when hovering over it
        notificationsDropdown.addEventListener('mouseenter', () => {
            notificationsDropdown.style.display = 'block';
            notificationsDropdownOpen = true;
        });
        
        // Close when mouse leaves both elements
        const closeDropdown = () => {
            setTimeout(() => {
                if (!userMenuNotifications.matches(':hover') && !notificationsDropdown.matches(':hover')) {
                    notificationsDropdown.style.display = 'none';
                    notificationsDropdownOpen = false;
                }
            }, 200);
        };
        
        userMenuNotifications.addEventListener('mouseleave', closeDropdown);
        notificationsDropdown.addEventListener('mouseleave', closeDropdown);
        
        // Reposition on window resize
        window.addEventListener('resize', () => {
            if (notificationsDropdownOpen && notificationsDropdown.style.display === 'block') {
                positionNotificationsDropdown();
            }
        });
    }
    
    if (markAllReadBtn) {
        markAllReadBtn.addEventListener('click', async (e) => {
            e.stopPropagation();
            try {
                await API.markAllNotificationsRead();
                await loadNotifications();
                loadNotificationsDropdown();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
    }
    
    if (viewAllLink) {
        viewAllLink.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            notificationsDropdownOpen = false;
            if (notificationsDropdown) {
                notificationsDropdown.style.display = 'none';
            }
            const userMenuDropdown = document.getElementById('user-menu-dropdown');
            if (userMenuDropdown) {
                userMenuDropdown.style.display = 'none';
                userMenuOpen = false;
            }
            switchView('notifications');
        });
    }
    
    // Page buttons
    const markAllReadPageBtn = document.getElementById('mark-all-read-page-btn');
    if (markAllReadPageBtn) {
        markAllReadPageBtn.addEventListener('click', async () => {
            try {
                await API.markAllNotificationsRead();
                await loadNotifications();
                await loadNotificationsPage();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
    }
    
    const deleteAllReadBtn = document.getElementById('delete-all-read-btn');
    if (deleteAllReadBtn) {
        deleteAllReadBtn.addEventListener('click', async () => {
            if (!confirm('Vuoi eliminare tutte le notifiche lette?')) {
                return;
            }
            try {
                await API.deleteAllReadNotifications();
                await loadNotifications();
                await loadNotificationsPage();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
    }
    
}

async function loadNotifications() {
    try {
        const result = await API.getNotifications({ unread_only: false, limit: 5 });
        const unreadCount = result.unread_count || 0;
        
        // Badge nel menu (con numero)
        const badge = document.getElementById('notifications-badge-menu');
        if (badge) {
            if (unreadCount > 0) {
                badge.textContent = unreadCount > 99 ? '99+' : unreadCount;
                badge.style.display = 'flex';
            } else {
                badge.style.display = 'none';
            }
        }
        
        // Badge sul nome utente (senza numero, solo pallino rosso)
        const badgeUserName = document.getElementById('notifications-badge-user-name');
        if (badgeUserName) {
            if (unreadCount > 0) {
                badgeUserName.style.display = 'block';
            } else {
                badgeUserName.style.display = 'none';
            }
        }
        
        // Update dropdown if it's open
        const notificationsDropdown = document.getElementById('notifications-dropdown');
        if (notificationsDropdown && notificationsDropdown.style.display === 'block') {
            loadNotificationsDropdown();
        }
    } catch (error) {
        console.error('Error loading notifications:', error);
    }
}

async function loadNotificationsDropdown() {
    try {
        const result = await API.getNotifications({ unread_only: false, limit: 10 });
        const notifications = result.notifications || [];
        const list = document.getElementById('notifications-list');
        
        if (!list) return;
        
        if (notifications.length === 0) {
            list.innerHTML = '<div style="padding: 2rem; text-align: center; color: #666;">Nessuna notifica</div>';
            return;
        }
        
        list.innerHTML = notifications.map(notif => {
            const isRead = notif.is_read;
            const date = new Date(notif.created_at);
            const dateStr = date.toLocaleDateString('it-IT', { 
                day: '2-digit', 
                month: '2-digit', 
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
            
            return `
                <div class="notification-item" style="padding: 0.75rem 1rem; border-bottom: 1px solid #f0f0f0; ${!isRead ? 'background: #f8f9fa;' : ''} cursor: pointer;" 
                     onclick="handleNotificationClick(${notif.id}, '${notif.type}', '${notif.related_entity_type}', ${notif.related_entity_id || 'null'})">
                    <div style="display: flex; justify-content: space-between; align-items: start;">
                        <div style="flex: 1;">
                            <div style="font-weight: ${isRead ? 'normal' : 'bold'}; margin-bottom: 0.25rem;">${escapeHtml(notif.title)}</div>
                            <div style="font-size: 0.9em; color: #666;">${escapeHtml(notif.message)}</div>
                            <div style="font-size: 0.75em; color: #999; margin-top: 0.25rem;">${dateStr}</div>
                        </div>
                        ${!isRead ? '<span style="width: 8px; height: 8px; background: #007bff; border-radius: 50%; margin-left: 0.5rem; flex-shrink: 0; margin-top: 0.25rem;"></span>' : ''}
                    </div>
                </div>
            `;
        }).join('');
    } catch (error) {
        console.error('Error loading notifications dropdown:', error);
    }
}

async function handleNotificationClick(notificationId, type, relatedEntityType, relatedEntityId) {
    try {
        // Mark as read
        const notificationItem = document.querySelector(`.notification-item[onclick*="${notificationId}"]`);
        if (notificationItem && !notificationItem.style.background.includes('#f8f9fa')) {
            await API.markNotificationRead(notificationId);
            await loadNotifications();
            loadNotificationsDropdown();
        }
        
        // Navigate based on type
        if (relatedEntityType === 'prepaid_card' && relatedEntityId) {
            switchView('prepaid-cards');
        } else if (relatedEntityType === 'account' && relatedEntityId) {
            switchView('accounts');
        } else if (relatedEntityType === 'fidelity_card' && relatedEntityId) {
            switchView('fidelity-cards');
        } else if (type === 'share_created' || type === 'share_removed' || type === 'share_updated') {
            // Default navigation based on share type
            if (relatedEntityType === 'prepaid_card') {
                switchView('prepaid-cards');
            } else if (relatedEntityType === 'account') {
                switchView('accounts');
            } else if (relatedEntityType === 'fidelity_card') {
                switchView('fidelity-cards');
            }
        }
        
        // Close dropdowns
        notificationsDropdownOpen = false;
        const notificationsDropdown = document.getElementById('notifications-dropdown');
        if (notificationsDropdown) {
            notificationsDropdown.style.display = 'none';
        }
        const userMenuDropdown = document.getElementById('user-menu-dropdown');
        if (userMenuDropdown) {
            userMenuDropdown.style.display = 'none';
            userMenuOpen = false;
        }
    } catch (error) {
        console.error('Error handling notification click:', error);
    }
}

async function loadNotificationsPage() {
    try {
        const result = await API.getNotifications({ unread_only: false });
        const notifications = result.notifications || [];
        const container = document.getElementById('notifications-page-list');
        
        if (!container) return;
        
        if (notifications.length === 0) {
            container.innerHTML = '<p style="text-align: center; color: #666; padding: 2rem;">Nessuna notifica</p>';
            return;
        }
        
        container.innerHTML = notifications.map(notif => {
            const isRead = notif.is_read;
            const date = new Date(notif.created_at);
            const dateStr = date.toLocaleDateString('it-IT', { 
                day: '2-digit', 
                month: '2-digit', 
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
            
            return `
                <div class="card" style="margin-bottom: 1rem; ${!isRead ? 'border-left: 4px solid #007bff;' : ''}">
                    <div style="display: flex; justify-content: space-between; align-items: start;">
                        <div style="flex: 1;">
                            <div style="font-weight: ${isRead ? 'normal' : 'bold'}; margin-bottom: 0.5rem; font-size: 1.1em;">
                                ${escapeHtml(notif.title)}
                            </div>
                            <div style="color: #666; margin-bottom: 0.5rem;">${escapeHtml(notif.message)}</div>
                            <div style="font-size: 0.85em; color: #999;">${dateStr}</div>
                        </div>
                        <div style="display: flex; gap: 0.5rem; margin-left: 1rem;">
                            ${!isRead ? `
                                <button class="btn btn-sm" onclick="markNotificationReadPage(${notif.id})" style="padding: 0.25rem 0.5rem; font-size: 0.85em;">
                                    Segna letta
                                </button>
                            ` : ''}
                            <button class="btn btn-sm" onclick="deleteNotificationPage(${notif.id})" style="padding: 0.25rem 0.5rem; font-size: 0.85em; background: #dc3545; color: white;">
                                Elimina
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
    } catch (error) {
        console.error('Error loading notifications page:', error);
        alert('Errore: ' + error.message);
    }
}

async function markNotificationReadPage(notificationId) {
    try {
        await API.markNotificationRead(notificationId);
        await loadNotifications();
        await loadNotificationsPage();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function deleteNotificationPage(notificationId) {
    if (!confirm('Vuoi eliminare questa notifica?')) {
        return;
    }
    
    try {
        await API.deleteNotification(notificationId);
        await loadNotifications();
        await loadNotificationsPage();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

// Make functions globally accessible
window.handleNotificationClick = handleNotificationClick;
window.markNotificationReadPage = markNotificationReadPage;
window.deleteNotificationPage = deleteNotificationPage;

// Transfer attachment helper functions
function addTransferAttachmentFile() {
    document.getElementById('transfer-file-input').click();
}

function addTransferAttachmentLink() {
    const linkUrl = prompt('Inserisci URL del link:');
    if (!linkUrl) return;
    
    const linkText = prompt('Inserisci testo del link (opzionale):') || linkUrl;
    
    if (!window.currentTransferAttachments) {
        window.currentTransferAttachments = [];
    }
    
    window.currentTransferAttachments.push({
        type: 'LINK',
        linkUrl: linkUrl,
        linkText: linkText,
        tempId: 'temp_' + Date.now() + '_' + Math.random()
    });
    
    updateTransferAttachmentsList();
}

function updateTransferAttachmentsList() {
    const container = document.getElementById('transfer-attachments-list');
    if (!container) return;
    
    if (!window.currentTransferAttachments || window.currentTransferAttachments.length === 0) {
        container.innerHTML = '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>';
        return;
    }
    
    container.innerHTML = window.currentTransferAttachments.map(att => {
        if (att.type === 'PHOTO' || att.type === 'DOCUMENT') {
            const preview = att.type === 'PHOTO' && att.file
                ? `<img src="${URL.createObjectURL(att.file)}" style="max-width: 50px; max-height: 50px; border-radius: 3px; margin-right: 10px;">`
                : `<span style="margin-right: 10px;">📄 ${att.file.name}</span>`;
            return `
                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                    ${preview}
                    <span style="flex: 1;">${att.file.name}</span>
                    <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="removeTransferAttachment('${att.tempId}')">Rimuovi</button>
                </div>
            `;
        } else if (att.type === 'LINK') {
            return `
                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                    <span style="margin-right: 10px;">🔗</span>
                    <span style="flex: 1;"><a href="${escapeHtml(att.linkUrl)}" target="_blank">${escapeHtml(att.linkText)}</a></span>
                    <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="removeTransferAttachment('${att.tempId}')">Rimuovi</button>
                </div>
            `;
        }
    }).join('');
}

function removeTransferAttachment(tempId) {
    if (!window.currentTransferAttachments) return;
    window.currentTransferAttachments = window.currentTransferAttachments.filter(att => att.tempId !== tempId);
    updateTransferAttachmentsList();
}

window.addTransferAttachmentFile = addTransferAttachmentFile;
window.addTransferAttachmentLink = addTransferAttachmentLink;
window.removeTransferAttachment = removeTransferAttachment;

// Attachment helper functions
window.addAttachmentFile = function(formType) {
    const inputId = formType === 'add-expense' ? 'add-expense-file-input' : 'edit-expense-file-input';
    document.getElementById(inputId).click();
}

window.addAttachmentLink = function(formType) {
    const linkUrl = prompt('Inserisci URL del link:');
    if (!linkUrl) return;
    
    const linkText = prompt('Inserisci testo del link (opzionale):') || linkUrl;
    
    if (!window.currentAttachments) {
        window.currentAttachments = [];
    }
    
    window.currentAttachments.push({
        type: 'LINK',
        link_url: linkUrl,
        link_text: linkText,
        tempId: 'temp_' + Date.now() + '_' + Math.random()
    });
    
    updateAttachmentsList(formType);
}

window.removeAttachment = function(tempId, formType) {
    window.currentAttachments = window.currentAttachments.filter(att => att.tempId !== tempId);
    updateAttachmentsList(formType);
}

window.deleteExpenseAttachment = async function(attachmentId, formType) {
    if (!confirm('Sei sicuro di voler eliminare questo allegato?')) {
        return;
    }
    
    try {
        await API.deleteExpenseAttachment(attachmentId);
        // Reload expenses to refresh the list
        const expenses = await API.getExpenses();
        const expense = expenses.find(e => e.id === window.currentExpenseId);
        if (expense) {
            // Update the attachments list in the form
            const container = document.getElementById('edit-attachments-list');
            container.innerHTML = expense.attachments && expense.attachments.length > 0 
                ? expense.attachments.map(att => `
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                        ${att.type === 'PHOTO' ? `<img src="/uploads/${att.file_path}" style="max-width: 50px; max-height: 50px; border-radius: 3px;">` : ''}
                        ${att.type === 'DOCUMENT' ? `<span>📄 ${att.file_path.split('/').pop()}</span>` : ''}
                        ${att.type === 'LINK' ? `<span>🔗 ${escapeHtml(att.link_text || att.link_url)}</span>` : ''}
                        <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="deleteExpenseAttachment(${att.id}, 'edit-expense')">Elimina</button>
                    </div>
                `).join('')
                : '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>';
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

// Securities attachment functions
window.addSecurityAttachmentFile = function(formType) {
    const inputId = formType === 'add-security' ? 'add-security-file-input' : 'edit-security-file-input';
    document.getElementById(inputId).click();
}

window.addSecurityAttachmentLink = function(formType) {
    const linkUrl = prompt('Inserisci URL del link:');
    if (!linkUrl) return;
    
    const linkText = prompt('Inserisci testo del link (opzionale):') || linkUrl;
    
    if (!window.currentSecurityAttachments) {
        window.currentSecurityAttachments = [];
    }
    
    window.currentSecurityAttachments.push({
        type: 'LINK',
        link_url: linkUrl,
        link_text: linkText,
        tempId: 'temp_' + Date.now() + '_' + Math.random()
    });
    
    updateSecurityAttachmentsList(formType);
}

window.removeSecurityAttachment = function(tempId, formType) {
    window.currentSecurityAttachments = window.currentSecurityAttachments.filter(att => att.tempId !== tempId);
    updateSecurityAttachmentsList(formType);
}

window.deleteSecurityAttachment = async function(attachmentId, formType) {
    if (!confirm('Sei sicuro di voler eliminare questo allegato?')) {
        return;
    }
    
    try {
        await API.deleteSecurityAttachment(attachmentId);
        // Reload securities to refresh the list
        const securities = await API.getSecurities();
        const security = securities.find(s => s.id === window.currentSecurityId);
        if (security) {
            // Update the attachments list in the form
            const container = document.getElementById('edit-security-attachments-list');
            container.innerHTML = security.attachments && security.attachments.length > 0 
                ? security.attachments.map(att => `
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                        ${att.type === 'PHOTO' ? `<img src="/uploads/${att.file_path}" style="max-width: 50px; max-height: 50px; border-radius: 3px;">` : ''}
                        ${att.type === 'DOCUMENT' ? `<span>📄 ${att.file_path.split('/').pop()}</span>` : ''}
                        ${att.type === 'LINK' ? `<span>🔗 ${escapeHtml(att.link_text || att.link_url)}</span>` : ''}
                        <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="deleteSecurityAttachment(${att.id}, 'edit-security')">Elimina</button>
                    </div>
                `).join('')
                : '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>';
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

function updateSecurityAttachmentsList(formType) {
    let containerId;
    if (formType === 'add-security') {
        containerId = 'add-security-attachments-list';
    } else if (formType === 'edit-security') {
        containerId = 'edit-security-attachments-list';
    } else if (formType === 'sell-security') {
        containerId = 'sell-security-attachments-list';
    } else {
        return;
    }
    const container = document.getElementById(containerId);
    
    if (!window.currentSecurityAttachments || window.currentSecurityAttachments.length === 0) {
        if (formType === 'add-security') {
            container.innerHTML = '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>';
        }
        return;
    }
    
    container.innerHTML = window.currentSecurityAttachments.map(att => {
        if (att.type === 'PHOTO' || att.type === 'DOCUMENT') {
            const preview = att.type === 'PHOTO' && att.file
                ? `<img src="${URL.createObjectURL(att.file)}" style="max-width: 50px; max-height: 50px; border-radius: 3px; margin-right: 10px;">`
                : `<span style="margin-right: 10px;">📄 ${att.file.name}</span>`;
            return `
                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                    ${preview}
                    <span style="flex: 1;">${att.file.name}</span>
                    <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="removeSecurityAttachment('${att.tempId}', '${formType}')">Rimuovi</button>
                </div>
            `;
        } else if (att.type === 'LINK') {
            return `
                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                    <span>🔗 ${escapeHtml(att.link_text || att.link_url)}</span>
                    <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem; margin-left: auto;" onclick="removeSecurityAttachment('${att.tempId}', '${formType}')">Rimuovi</button>
                </div>
            `;
        }
    }).join('');
}

function updateAttachmentsList(formType) {
    const containerId = formType === 'add-expense' ? 'attachments-list' : 'edit-attachments-list';
    const container = document.getElementById(containerId);
    
    if (!window.currentAttachments || window.currentAttachments.length === 0) {
        container.innerHTML = '<p style="color: #666; font-size: 0.9rem;">Nessun allegato</p>';
        return;
    }
    
    container.innerHTML = window.currentAttachments.map(att => {
        if (att.type === 'PHOTO' || att.type === 'DOCUMENT') {
            const preview = att.type === 'PHOTO' && att.file
                ? `<img src="${URL.createObjectURL(att.file)}" style="max-width: 50px; max-height: 50px; border-radius: 3px; margin-right: 10px;">`
                : `<span style="margin-right: 10px;">📄 ${att.file.name}</span>`;
            return `
                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                    ${preview}
                    <span style="flex: 1;">${att.file.name}</span>
                    <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem;" onclick="removeAttachment('${att.tempId}', '${formType}')">Rimuovi</button>
                </div>
            `;
        } else if (att.type === 'LINK') {
            return `
                <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 5px; padding: 5px; background: #f5f5f5; border-radius: 5px;">
                    <span>🔗 ${escapeHtml(att.link_text || att.link_url)}</span>
                    <button type="button" class="btn btn-danger" style="padding: 2px 8px; font-size: 0.8rem; margin-left: auto;" onclick="removeAttachment('${att.tempId}', '${formType}')">Rimuovi</button>
                </div>
            `;
        }
    }).join('');
}

// Modal functions
function showModal() {
    const modalOverlay = document.getElementById('modal-overlay');
    const modalContent = document.querySelector('.modal-content');
    modalOverlay.classList.remove('hidden');
    // Check if this is a fidelity card fullscreen view
    setTimeout(() => {
        if (document.getElementById('fidelity-card-container')) {
            if (modalContent) {
                modalContent.classList.add('fullscreen');
            }
        } else {
            if (modalContent) {
                modalContent.classList.remove('fullscreen');
            }
        }
    }, 10);
}

function closeModal() {
    // Remove resize handler if it exists
    if (window.fidelityCardResizeHandler) {
        window.removeEventListener('resize', window.fidelityCardResizeHandler);
        window.fidelityCardResizeHandler = null;
    }
    
    // Remove touch scroll prevention
    if (window.fidelityCardPreventScroll) {
        document.removeEventListener('touchmove', window.fidelityCardPreventScroll);
        document.removeEventListener('touchstart', window.fidelityCardPreventScroll);
        window.fidelityCardPreventScroll = null;
    }
    
    // Handle prepaid card scroll prevention
    if (window.prepaidCardPreventScroll) {
        document.removeEventListener('touchmove', window.prepaidCardPreventScroll);
        document.removeEventListener('touchstart', window.prepaidCardPreventScroll);
        window.prepaidCardPreventScroll = null;
    }
    
    // Remove body scroll lock
    document.body.classList.remove('modal-fullscreen-open');
    
    // Reset modal open flags
    isFidelityCardModalOpen = false;
    isPrepaidCardModalOpen = false;
    
    // Remove history state if it was added (but don't call back() if we're already handling popstate)
    // When user presses back button, popstate event handles the closing, so we don't need to call back() again
    if (window.fidelityCardHistoryState && !window.fidelityCardClosingFromBackButton) {
        // Only remove the history state if we're closing manually (not from back button)
        // Check if current state is our modal state before going back
        if (window.history && window.history.state && window.history.state.modalOpen) {
            // Remove our modal state from history by going back
            window.history.back();
        }
    }
    window.fidelityCardHistoryState = false;
    window.fidelityCardClosingFromBackButton = false;
    
    // Handle prepaid card history state
    if (window.prepaidCardHistoryState && !window.prepaidCardClosingFromBackButton) {
        if (window.history && window.history.state && window.history.state.modalOpen) {
            window.history.back();
        }
    }
    window.prepaidCardHistoryState = false;
    window.prepaidCardClosingFromBackButton = false;
    
    // Reset modal overlay padding
    const modalOverlay = document.getElementById('modal-overlay');
    if (modalOverlay) {
        modalOverlay.style.padding = '';
    }
    
    // Remove fullscreen class
    const modalContent = document.querySelector('.modal-content');
    if (modalContent) {
        modalContent.classList.remove('fullscreen');
    }
    
    document.getElementById('modal-overlay').classList.add('hidden');
    document.getElementById('modal-body').innerHTML = '';
}

// Utility functions
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatCurrency(amount, currency) {
    return new Intl.NumberFormat('it-IT', {
        style: 'currency',
        currency: currency || 'EUR'
    }).format(amount);
}

function formatDate(dateString) {
    if (!dateString) return '';
    const date = new Date(dateString);
    return date.toLocaleDateString('it-IT');
}

function getTodayDate() {
    const today = new Date();
    const year = today.getFullYear();
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const day = String(today.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
}

function getNextBusinessDay(date = null) {
    const d = date ? new Date(date) : new Date();
    const now = new Date();
    const currentHour = now.getHours();
    
    // If current time is after 5PM, start from tomorrow
    if (!date && currentHour >= 17) {
        d.setDate(d.getDate() + 1);
    } else if (!date) {
        // If before 5PM, use today as departure date, tomorrow as arrival
        d.setDate(d.getDate() + 1);
    }
    
    // Skip weekends (Saturday = 6, Sunday = 0)
    while (d.getDay() === 0 || d.getDay() === 6) {
        d.setDate(d.getDate() + 1);
    }
    
    const year = d.getFullYear();
    const month = String(d.getMonth() + 1).padStart(2, '0');
    const day = String(d.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
}

// Global functions for onclick handlers
window.viewAccountBalance = async function(accountId) {
    try {
        const balance = await API.getAccountBalance(accountId);
        
        if (balance.type_code === 'SECURITIES' && balance.securities_by_currency) {
            // For securities accounts, convert currencies to base currency (same logic as dashboard)
            let totalBalance = 0;
            const exchangeRates = {};
            
            const conversionPromises = balance.securities_by_currency.map(async (sec) => {
                const value = parseFloat(sec.total_value) || 0;
                if (isNaN(value)) {
                    return { converted: 0, currency: sec.currency };
                }
                
                if (sec.currency === balance.currency) {
                    return { converted: value, currency: sec.currency };
                } else {
                    try {
                        const rateData = await API.getExchangeRate(sec.currency, balance.currency);
                        const rate = parseFloat(rateData.rate) || 1;
                        const converted = value * rate;
                        exchangeRates[sec.currency] = rate;
                        return { converted: converted, currency: sec.currency };
                    } catch (error) {
                        console.error(`Error fetching exchange rate ${sec.currency} to ${balance.currency}:`, error);
                        return { converted: 0, currency: sec.currency };
                    }
                }
            });
            
            const conversionResults = await Promise.all(conversionPromises);
            totalBalance = conversionResults.reduce((sum, result) => {
                const numVal = parseFloat(result.converted) || 0;
                if (isNaN(numVal)) {
                    return sum;
                }
                return sum + numVal;
            }, 0);
            
            // Build display message with breakdown
            let message = `Bilancio: ${formatCurrency(totalBalance, balance.currency)}\n\n`;
            message += `Dettaglio per valuta:\n`;
            balance.securities_by_currency.forEach(sec => {
                const value = parseFloat(sec.total_value) || 0;
                message += `  ${formatCurrency(value, sec.currency)}`;
                if (sec.currency !== balance.currency && exchangeRates[sec.currency]) {
                    message += ` (tasso: ${parseFloat(exchangeRates[sec.currency]).toFixed(4)})`;
                }
                message += `\n`;
            });
            
            alert(message);
        } else {
            // For regular accounts, show standard balance
            alert(`Bilancio: ${formatCurrency(balance.balance, balance.currency)}`);
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
};

// Account Sharing Functions
let currentManageSharesAccountId = null;

async function showAccountSharesModal(accountId) {
    try {
        currentManageSharesAccountId = accountId;
        const [accounts, sharesData, previouslySharedUsers] = await Promise.all([
            API.getAccounts(),
            API.getAccountShares(accountId),
            API.getPreviouslySharedUsers()
        ]);
        
        const account = accounts.find(a => a.id === accountId);
        
        if (!account) {
            alert('Account non trovato');
            return;
        }
        
        // Prevent sharing "No Account"
        if (account.name === 'No Account') {
            alert('Non è possibile condividere "No Account"');
            return;
        }
        
        const modalBody = document.getElementById('modal-body');
        modalBody.innerHTML = `
            <h2>Gestisci Condivisioni: ${escapeHtml(account.name)}</h2>
            
            <div id="shares-list">
                <h3>Condivisioni attive:</h3>
                ${sharesData.shares.length === 0 
                    ? '<p style="color: #666;">Nessuna condivisione attiva</p>'
                    : sharesData.shares.map(share => `
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; margin-bottom: 0.5rem; background: #f5f5f5; border-radius: 4px;">
                            <div>
                                <div style="font-weight: 500;">${escapeHtml(share.name || share.email)}</div>
                                <div style="font-size: 0.9em; color: #666;">${escapeHtml(share.email)}</div>
                            </div>
                            <div style="display: flex; gap: 0.5rem; align-items: center;">
                                <select onchange="updateShareMode(${share.id}, this.value)" 
                                        style="padding: 0.25rem; border: 1px solid #ddd; border-radius: 4px;">
                                    <option value="read-only" ${share.sharing_mode === 'read-only' ? 'selected' : ''}>Sola lettura</option>
                                    <option value="write" ${share.sharing_mode === 'write' ? 'selected' : ''}>Scrittura</option>
                                </select>
                                <button class="btn btn-danger" onclick="removeShare(${share.id})" style="padding: 0.25rem 0.5rem;">Rimuovi</button>
                            </div>
                        </div>
                    `).join('')
                }
                
                <h3 style="margin-top: 1.5rem;">Inviti in attesa:</h3>
                ${sharesData.invitations.length === 0 
                    ? '<p style="color: #666;">Nessun invito in attesa</p>'
                    : sharesData.invitations.map(inv => `
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; margin-bottom: 0.5rem; background: #fff3cd; border-radius: 4px;">
                            <div>
                                <div style="font-weight: 500;">${escapeHtml(inv.recipient_email)}</div>
                                <div style="font-size: 0.85em; color: #888; margin-top: 0.25rem;">In attesa di registrazione</div>
                            </div>
                            <div style="display: flex; gap: 0.5rem; align-items: center;">
                                <select onchange="updateInvitationMode(${inv.id}, this.value)" 
                                        style="padding: 0.25rem; border: 1px solid #ddd; border-radius: 4px;">
                                    <option value="read-only" ${inv.sharing_mode === 'read-only' ? 'selected' : ''}>Sola lettura</option>
                                    <option value="write" ${inv.sharing_mode === 'write' ? 'selected' : ''}>Scrittura</option>
                                </select>
                                <button class="btn btn-danger" onclick="removeInvitation(${inv.id})" style="padding: 0.25rem 0.5rem;">Rimuovi</button>
                            </div>
                        </div>
                    `).join('')
                }
            </div>
            
            <div style="margin-top: 2rem; padding-top: 1.5rem; border-top: 1px solid #ddd;">
                <h3>Aggiungi nuova condivisione:</h3>
                <form id="share-account-form">
                    <div class="form-group">
                        <label for="recipient-email">Email utente:</label>
                        <input type="email" id="recipient-email" list="previously-shared-users" required 
                               placeholder="Inserisci email utente" style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;">
                        <datalist id="previously-shared-users">
                            ${previouslySharedUsers.map(user => `<option value="${escapeHtml(user.email)}">${escapeHtml(user.name || user.email)}</option>`).join('')}
                        </datalist>
                    </div>
                    <div class="form-group">
                        <label for="sharing-mode">Modalità di condivisione:</label>
                        <select id="sharing-mode" required style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;">
                            <option value="read-only">Sola lettura</option>
                            <option value="write">Scrittura</option>
                        </select>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeModal()">Chiudi</button>
                        <button type="submit" class="btn btn-primary">Condividi</button>
                    </div>
                </form>
            </div>
        `;
        
        document.getElementById('share-account-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const recipientEmail = document.getElementById('recipient-email').value.trim();
            const sharingMode = document.getElementById('sharing-mode').value;
            
            try {
                await API.shareAccount(accountId, recipientEmail, sharingMode);
                // Reload the modal to show the new share/invitation
                await showAccountSharesModal(accountId);
                loadAccounts();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function updateShareMode(shareId, sharingMode) {
    try {
        await API.updateShareMode(shareId, sharingMode);
        if (currentManageSharesAccountId) {
            await showAccountSharesModal(currentManageSharesAccountId);
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function updateInvitationMode(invitationId, sharingMode) {
    try {
        await API.updateInvitationMode(invitationId, sharingMode);
        if (currentManageSharesAccountId) {
            await showAccountSharesModal(currentManageSharesAccountId);
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function removeShare(shareId) {
    if (!confirm('Sei sicuro di voler rimuovere questa condivisione?')) {
        return;
    }
    
    try {
        await API.removeShare(shareId);
        loadAccounts();
        if (currentManageSharesAccountId) {
            await showAccountSharesModal(currentManageSharesAccountId);
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function removeInvitation(invitationId) {
    if (!confirm('Sei sicuro di voler rimuovere questo invito?')) {
        return;
    }
    
    try {
        await API.removeInvitation(invitationId);
        alert('Invito rimosso con successo');
        if (currentManageSharesAccountId) {
            await showAccountSharesModal(currentManageSharesAccountId);
        }
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function stopSharingAccount(accountId) {
    if (!confirm('Sei sicuro di voler rimuovere questa condivisione?')) {
        return;
    }
    
    try {
        // Get all accounts to find the share ID for this account
        const accounts = await API.getAccounts();
        const account = accounts.find(a => a.id === accountId && a.account_status === 'shared');
        
        if (!account || !account.share_id) {
            alert('Condivisione non trovata');
            return;
        }
        
        await API.removeShare(account.share_id);
        loadAccounts();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

let currentManageSharesPrepaidCardId = null;

async function showPrepaidCardSharesModal(prepaidCardId) {
    try {
        currentManageSharesPrepaidCardId = prepaidCardId;
        const [prepaidCards, sharesData, previouslySharedUsers] = await Promise.all([
            API.getPrepaidCards(),
            API.getPrepaidCardShares(prepaidCardId),
            API.getPreviouslySharedUsersForPrepaidCards()
        ]);
        
        const prepaidCard = prepaidCards.find(c => c.id === prepaidCardId);
        
        if (!prepaidCard) {
            alert('Carta prepagata non trovata');
            return;
        }
        
        const modalBody = document.getElementById('modal-body');
        modalBody.innerHTML = `
            <h2>Gestisci Condivisioni: ${escapeHtml(prepaidCard.name)}</h2>
            
            <div id="shares-list">
                <h3>Condivisioni attive:</h3>
                ${sharesData.shares.length === 0 
                    ? '<p style="color: #666;">Nessuna condivisione attiva</p>'
                    : sharesData.shares.map(share => `
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; margin-bottom: 0.5rem; background: #f5f5f5; border-radius: 4px;">
                            <div>
                                <div style="font-weight: 500;">${escapeHtml(share.name || share.email)}</div>
                                <div style="font-size: 0.9em; color: #666;">${escapeHtml(share.email)}</div>
                                <div style="font-size: 0.85em; color: #888; margin-top: 0.25rem;">Scrittura</div>
                            </div>
                            <button class="btn btn-danger" onclick="removePrepaidCardShare(${share.id})" style="padding: 0.25rem 0.5rem;">Rimuovi</button>
                        </div>
                    `).join('')
                }
                
                <h3 style="margin-top: 1.5rem;">Inviti in attesa:</h3>
                ${sharesData.invitations.length === 0 
                    ? '<p style="color: #666;">Nessun invito in attesa</p>'
                    : sharesData.invitations.map(invitation => `
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; margin-bottom: 0.5rem; background: #fff3cd; border-radius: 4px;">
                            <div>
                                <div style="font-weight: 500;">${escapeHtml(invitation.recipient_email)}</div>
                                <div style="font-size: 0.85em; color: #888; margin-top: 0.25rem;">In attesa di registrazione</div>
                            </div>
                            <button class="btn btn-danger" onclick="removePrepaidCardInvitation(${invitation.id})" style="padding: 0.25rem 0.5rem;">Rimuovi</button>
                        </div>
                    `).join('')
                }
            </div>
            
            <div style="margin-top: 2rem; padding-top: 1.5rem; border-top: 1px solid #ddd;">
                <h3>Aggiungi nuova condivisione:</h3>
                <form id="share-prepaid-card-form">
                    <div class="form-group">
                        <label for="recipient-email">Email utente:</label>
                        <input type="email" id="recipient-email" list="previously-shared-users" required 
                               placeholder="Inserisci email utente" style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;">
                        <datalist id="previously-shared-users">
                            ${previouslySharedUsers.map(user => `<option value="${escapeHtml(user.email)}">${escapeHtml(user.name || user.email)}</option>`).join('')}
                        </datalist>
                        <small style="color: #666; display: block; margin-top: 4px;">Le carte prepagate possono essere condivise solo in modalità scrittura</small>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeModal()">Chiudi</button>
                        <button type="submit" class="btn btn-primary">Condividi</button>
                    </div>
                </form>
            </div>
        `;
        
        document.getElementById('share-prepaid-card-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const recipientEmail = document.getElementById('recipient-email').value.trim();
            
            try {
                await API.sharePrepaidCard(prepaidCardId, recipientEmail);
                // Reload the modal to show the new share/invitation
                await showPrepaidCardSharesModal(prepaidCardId);
                loadPrepaidCards();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function removePrepaidCardShare(shareId) {
    if (!confirm('Sei sicuro di voler rimuovere questa condivisione?')) {
        return;
    }
    
    try {
        await API.removePrepaidCardShare(shareId);
        if (currentManageSharesPrepaidCardId) {
            await showPrepaidCardSharesModal(currentManageSharesPrepaidCardId);
        }
        loadPrepaidCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function removePrepaidCardInvitation(invitationId) {
    if (!confirm('Sei sicuro di voler rimuovere questo invito?')) {
        return;
    }
    
    try {
        await API.removePrepaidCardInvitation(invitationId);
        alert('Invito rimosso con successo');
        if (currentManageSharesPrepaidCardId) {
            await showPrepaidCardSharesModal(currentManageSharesPrepaidCardId);
        }
        loadPrepaidCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function stopSharingPrepaidCard(prepaidCardId) {
    if (!confirm('Sei sicuro di voler rimuovere questa condivisione?')) {
        return;
    }
    
    try {
        // Get all prepaid cards to find the share ID for this card
        const cards = await API.getPrepaidCards();
        const card = cards.find(c => c.id === prepaidCardId && c.card_status === 'shared');
        
        if (!card || !card.share_id) {
            alert('Condivisione non trovata');
            return;
        }
        
        await API.removePrepaidCardShare(card.share_id);
        loadPrepaidCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

window.showPrepaidCardFullscreen = async function(cardId) {
    try {
        // Get card data
        const cards = await API.getPrepaidCards();
        const card = cards.find(c => c.id === cardId);
        
        if (!card) {
            alert('Carta non trovata');
            return;
        }
        
        const hasCardNumber = card.card_number && card.card_number.trim() !== '';
        
        const modalBody = document.getElementById('modal-body');
        
        // Get viewport dimensions
        const viewportWidth = window.innerWidth;
        const viewportHeight = window.innerHeight;
        
        // Card aspect ratio: 85/54 = 1.574
        const cardAspectRatio = 85 / 54;
        
        // Calculate card dimensions to fill viewport
        let cardWidth, cardHeight;
        cardWidth = viewportWidth;
        cardHeight = cardWidth / cardAspectRatio;
        
        if (cardHeight > viewportHeight) {
            cardHeight = viewportHeight;
            cardWidth = cardHeight * cardAspectRatio;
        }
        
        // Create canvas
        const canvas = document.createElement('canvas');
        canvas.width = cardWidth;
        canvas.height = cardHeight;
        const ctx = canvas.getContext('2d');
        
        // Draw gradient background
        const drawGradientBackground = () => {
            const gradient = ctx.createLinearGradient(0, 0, canvas.width, canvas.height);
            gradient.addColorStop(0, '#667eea');
            gradient.addColorStop(1, '#764ba2');
            ctx.fillStyle = gradient;
            ctx.fillRect(0, 0, canvas.width, canvas.height);
        };
        
        // Draw card photo if available
        const drawCardPhoto = () => {
            return new Promise((resolve) => {
                if (card.photo_path) {
                    const photoImg = new Image();
                    photoImg.onload = () => {
                        try {
                            // Draw photo covering entire canvas
                            ctx.drawImage(photoImg, 0, 0, canvas.width, canvas.height);
                            resolve();
                        } catch (error) {
                            console.error('Error drawing photo:', error);
                            drawGradientBackground();
                            resolve();
                        }
                    };
                    photoImg.onerror = () => {
                        drawGradientBackground();
                        resolve();
                    };
                    photoImg.src = `/uploads/${card.photo_path}`;
                } else {
                    drawGradientBackground();
                    resolve();
                }
            });
        };
        
        // Draw barcode and text
        const drawBarcodeAndText = async () => {
            try {
                // Draw card name at top
                ctx.fillStyle = '#ffffff';
                ctx.font = `bold ${Math.max(24, canvas.width * 0.06)}px Arial`;
                ctx.textAlign = 'center';
                ctx.textBaseline = 'top';
                ctx.shadowColor = 'rgba(0, 0, 0, 0.7)';
                ctx.shadowBlur = 4;
                ctx.shadowOffsetX = 2;
                ctx.shadowOffsetY = 2;
                ctx.fillText(card.name, canvas.width / 2, canvas.height * 0.1);
                
                // Only generate barcode if card number exists
                if (!hasCardNumber) {
                    // No card number, just show card name
                    displayCanvas();
                    return;
                }
                
                // Barcode height: 0.23 of total height
                const barcodeHeight = canvas.height * 0.23;
                // Text is 0.05 from bottom, so text Y = canvas.height * 0.95
                const textY = canvas.height * 0.95;
                const estimatedTextHeight = canvas.width * 0.05;
                const barcodeY = textY - barcodeHeight - estimatedTextHeight - (canvas.height * 0.02);
                
                // Create temporary SVG for barcode
                const tempSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
                tempSvg.style.position = 'absolute';
                tempSvg.style.left = '-9999px';
                tempSvg.setAttribute('width', canvas.width.toString());
                tempSvg.setAttribute('height', barcodeHeight.toString());
                document.body.appendChild(tempSvg);
                
                if (typeof JsBarcode !== 'undefined') {
                    try {
                        // Generate barcode SVG
                        JsBarcode(tempSvg, card.card_number, {
                            format: "CODE128",
                            width: 2,
                            height: barcodeHeight * 0.8,
                            displayValue: true,
                            fontSize: Math.max(16, canvas.width * 0.04),
                            textMargin: 5,
                            background: "transparent",
                            lineColor: "#000000"
                        });
                        
                        // Convert SVG to image
                        const svgData = new XMLSerializer().serializeToString(tempSvg);
                        const svgUrl = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
                        
                        const barcodeImg = new Image();
                        barcodeImg.onload = () => {
                            try {
                                // Calculate barcode width and position (centered)
                                const barcodeWidth = Math.min(canvas.width * 0.7, barcodeImg.width);
                                const barcodeX = (canvas.width - barcodeWidth) / 2;
                                
                                // Draw white background for barcode
                                const padding = canvas.width * 0.02;
                                ctx.fillStyle = 'rgba(255, 255, 255, 0.9)';
                                ctx.fillRect(
                                    barcodeX - padding,
                                    barcodeY - padding,
                                    barcodeWidth + (padding * 2),
                                    barcodeHeight + (padding * 2)
                                );
                                
                                // Draw barcode
                                ctx.drawImage(
                                    barcodeImg,
                                    barcodeX,
                                    barcodeY,
                                    barcodeWidth,
                                    barcodeHeight
                                );
                                
                                // Draw card number below barcode
                                ctx.font = `${Math.max(18, canvas.width * 0.04)}px Arial`;
                                ctx.shadowBlur = 2;
                                ctx.shadowOffsetX = 1;
                                ctx.shadowOffsetY = 1;
                                ctx.fillText(card.card_number, canvas.width / 2, textY);
                                
                                // Clean up
                                document.body.removeChild(tempSvg);
                                displayCanvas();
                            } catch (drawError) {
                                console.error('Error drawing barcode:', drawError);
                                drawTextOnly();
                                displayCanvas();
                            }
                        };
                        barcodeImg.onerror = () => {
                            console.error('Error loading barcode image');
                            drawTextOnly();
                            displayCanvas();
                        };
                        barcodeImg.src = svgUrl;
                    } catch (barcodeError) {
                        console.error('Error generating barcode:', barcodeError);
                        drawTextOnly();
                        displayCanvas();
                    }
                } else {
                    console.warn('JsBarcode library not loaded');
                    drawTextOnly();
                    displayCanvas();
                }
            } catch (error) {
                console.error('Error in drawBarcodeAndText:', error);
                drawTextOnly();
                displayCanvas();
            }
        };
        
        // Draw text only (fallback)
        const drawTextOnly = () => {
            ctx.fillStyle = '#ffffff';
            ctx.font = `bold ${Math.max(24, canvas.width * 0.06)}px Arial`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.shadowColor = 'rgba(0, 0, 0, 0.7)';
            ctx.shadowBlur = 4;
            ctx.shadowOffsetX = 2;
            ctx.shadowOffsetY = 2;
            ctx.fillText(card.name, canvas.width / 2, canvas.height * 0.4);
            if (hasCardNumber) {
                ctx.font = `${Math.max(18, canvas.width * 0.04)}px Arial`;
                ctx.fillText(card.card_number, canvas.width / 2, canvas.height * 0.6);
            }
        };
        
        // Display canvas fullscreen
        const displayCanvas = () => {
            const imageDataUrl = canvas.toDataURL('image/png');
            
            modalBody.innerHTML = `
                <div id="prepaid-card-fullscreen-container" style="
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100vw;
                    height: 100vh;
                    background: #000;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    z-index: 10000;
                    overflow: hidden;
                ">
                    <img src="${imageDataUrl}" 
                         style="max-width: 100vw; max-height: 100vh; object-fit: contain;" 
                         alt="Prepaid Card">
                    <button class="modal-close" onclick="closeModal()" style="
                        position: absolute;
                        top: 20px;
                        right: 20px;
                        background: rgba(255, 255, 255, 0.9);
                        border: none;
                        border-radius: 50%;
                        width: 40px;
                        height: 40px;
                        font-size: 24px;
                        cursor: pointer;
                        z-index: 10001;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">×</button>
                </div>
            `;
            
            // Prevent body scrolling
            document.body.classList.add('modal-fullscreen-open');
            
            // Prevent touch scrolling
            const preventScroll = (e) => {
                if (e.touches.length > 1) {
                    return; // Allow pinch zoom
                }
                const target = e.target;
                if (target && (target.classList.contains('modal-close') || target.closest('.modal-close'))) {
                    return; // Allow click on close button
                }
                e.preventDefault();
            };
            document.addEventListener('touchmove', preventScroll, { passive: false });
            document.addEventListener('touchstart', preventScroll, { passive: false });
            window.prepaidCardPreventScroll = preventScroll;
            
            // Add history state for back button support
            if (window.history && window.history.pushState) {
                window.history.pushState({ modalOpen: true }, '');
                window.prepaidCardHistoryState = true;
            }
            
            isPrepaidCardModalOpen = true;
            showModal();
        };
        
        // Start drawing
        await drawCardPhoto();
        await drawBarcodeAndText();
    } catch (error) {
        console.error('Error showing prepaid card fullscreen:', error);
        alert('Errore: ' + error.message);
    }
}

// Fidelity Cards
let draggedFidelityCardId = null;
let draggedFidelityCardElement = null;
let isSavingFidelityCardOrder = false;

async function loadFidelityCards() {
    try {
        console.log('Loading fidelity cards...');
        const cards = await API.getFidelityCards();
        console.log('Fidelity cards loaded:', cards);
        const container = document.getElementById('fidelity-cards-list');
        
        if (!container) {
            console.error('Fidelity cards container not found');
            return;
        }
        
        if (!cards || cards.length === 0) {
            console.log('No fidelity cards found');
            container.innerHTML = '<p style="text-align: center; color: #666; padding: 2rem;">Nessuna carta fedeltà trovata. Clicca su "Aggiungi Carta Fedeltà" per crearne una.</p>';
            return;
        }
        
        container.innerHTML = cards.map(card => {
            const isShared = card.card_status === 'shared';
            const isOwned = card.card_status === 'owned';
            const ownerInfo = card.owner_name || card.owner_email ? `Condiviso da ${card.owner_name || card.owner_email}` : '';
            const hasShares = card.has_shares === 1 || card.has_shares === true;
            
            const logoStyle = card.logo_path 
                ? `background-image: url('/uploads/${card.logo_path}'); background-size: cover; background-position: center; background-repeat: no-repeat;`
                : 'background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);';
            
            return `
            <div class="card" 
                 data-fidelity-card-id="${card.id}" 
                 style="position: relative; cursor: pointer; min-height: 200px; ${logoStyle}; display: flex; flex-direction: column;"
                 draggable="true"
                 onclick="showFidelityCardFullscreen(${card.id})">
                <div style="padding: 1rem; margin-top: auto;">
                    <div class="card-header">
                        <div class="card-title" style="color: white; text-shadow: 2px 2px 4px rgba(0,0,0,0.7);">
                            ${escapeHtml(card.name)}
                            ${isShared ? `<span style="color: rgba(255,255,255,0.9); font-size: 0.9em; margin-left: 0.5rem;">(${ownerInfo})</span>` : ''}
                        </div>
                    </div>
                    <div class="card-meta" style="color: white; text-shadow: 2px 2px 4px rgba(0,0,0,0.7);">
                        ${escapeHtml(card.store_name)}<br>
                        Codice: ${escapeHtml(card.card_code)}
                    </div>
                </div>
                <div style="position: absolute; bottom: 10px; right: 10px; display: flex; gap: 10px; flex-direction: column; align-items: flex-end;">
                    <div style="display: flex; gap: 10px;">
                        ${card.card_status === 'owned' ? `
                            <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s; background: rgba(255, 255, 255, 0.9); padding: 0.25rem; border-radius: 4px;" 
                                  onclick="event.stopPropagation(); showFidelityCardSharesModal(${card.id})" 
                                  onmouseover="this.style.color='#007bff'" 
                                  onmouseout="this.style.color='#666'"
                                  title="${hasShares ? 'Condivisioni attive - Gestisci condivisioni' : 'Gestisci condivisioni'}">${hasShares ? '🔗👤' : '🔗'}</span>
                        ` : ''}
                        <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s; background: rgba(255, 255, 255, 0.9); padding: 0.25rem; border-radius: 4px;" 
                              onclick="event.stopPropagation(); showEditFidelityCardModal(${card.id})" 
                              onmouseover="this.style.color='#007bff'" 
                              onmouseout="this.style.color='#666'"
                              title="Modifica carta">✏️</span>
                        ${card.card_status === 'owned' ? `
                            <span style="cursor: pointer; font-size: 1.2em; color: #666; transition: color 0.2s; background: rgba(255, 255, 255, 0.9); padding: 0.25rem; border-radius: 4px;" 
                                  onclick="event.stopPropagation(); deleteFidelityCard(${card.id})" 
                                  onmouseover="this.style.color='#dc3545'" 
                                  onmouseout="this.style.color='#666'"
                                  title="Elimina carta">🗑️</span>
                        ` : ''}
                    </div>
                    ${isShared ? `
                        <button class="btn btn-secondary" onclick="event.stopPropagation(); stopSharingFidelityCard(${card.id})" style="padding: 0.25rem 0.5rem; font-size: 0.85rem; margin-top: 0.5rem;">Rimuovi Condivisione</button>
                    ` : ''}
                </div>
            </div>
            `;
        }).join('');
        
        // Initialize drag and drop
        initializeFidelityCardDragAndDrop();
    } catch (error) {
        console.error('Error loading fidelity cards:', error);
    }
}

function initializeFidelityCardDragAndDrop() {
    const container = document.getElementById('fidelity-cards-list');
    if (!container) {
        return;
    }
    
    const cards = container.querySelectorAll('.card[data-fidelity-card-id]');
    
    // Add drop handler to container
    container.addEventListener('dragover', (e) => {
        e.preventDefault();
        e.dataTransfer.dropEffect = 'move';
        
        const afterElement = getDragAfterElement(container, e.clientX);
        const dragging = container.querySelector('.dragging');
        if (dragging) {
            if (afterElement == null) {
                container.appendChild(dragging);
            } else {
                container.insertBefore(dragging, afterElement);
            }
        }
    });
    
    container.addEventListener('drop', async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        if (draggedFidelityCardId === null) {
            return;
        }
        
        await new Promise(resolve => setTimeout(resolve, 50));
        await saveFidelityCardOrder();
    });
    
    cards.forEach(card => {
        const cardId = parseInt(card.dataset.fidelityCardId);
        
        card.addEventListener('dragstart', (e) => {
            draggedFidelityCardId = cardId;
            draggedFidelityCardElement = card;
            card.style.opacity = '0.5';
            card.classList.add('dragging');
            e.dataTransfer.effectAllowed = 'move';
            e.dataTransfer.setData('text/html', '');
        });
        
        card.addEventListener('dragend', async (e) => {
            const wasDragging = draggedFidelityCardId !== null;
            card.style.opacity = '';
            card.classList.remove('dragging');
            
            draggedFidelityCardId = null;
            draggedFidelityCardElement = null;
            
            if (wasDragging) {
                setTimeout(async () => {
                    if (!isSavingFidelityCardOrder) {
                        await saveFidelityCardOrder();
                    }
                }, 150);
            }
        });
        
        card.addEventListener('dragover', (e) => {
            e.preventDefault();
            e.stopPropagation();
            e.dataTransfer.dropEffect = 'move';
            
            const afterElement = getDragAfterElement(container, e.clientX);
            const dragging = container.querySelector('.dragging');
            if (dragging && dragging !== card) {
                if (afterElement == null) {
                    container.appendChild(dragging);
                } else {
                    container.insertBefore(dragging, afterElement);
                }
            }
        });
        
        card.addEventListener('drop', async (e) => {
            e.preventDefault();
            e.stopPropagation();
            
            if (draggedFidelityCardId === null || draggedFidelityCardId === cardId) {
                return;
            }
            
            await new Promise(resolve => setTimeout(resolve, 50));
            await saveFidelityCardOrder();
        });
    });
}

async function saveFidelityCardOrder() {
    if (isSavingFidelityCardOrder) {
        return;
    }
    
    isSavingFidelityCardOrder = true;
    
    try {
        const container = document.getElementById('fidelity-cards-list');
        if (!container) {
            return;
        }
        
        const cards = Array.from(container.querySelectorAll('.card[data-fidelity-card-id]'));
        if (cards.length === 0) {
            return;
        }
        
        const cardOrders = cards.map((card, index) => ({
            fidelity_card_id: parseInt(card.dataset.fidelityCardId),
            display_order: index + 1
        }));
        
        await API.saveFidelityCardOrder(cardOrders);
        
        // Calculate hash of top 2 cards to force manifest URL change
        // This makes Chrome treat it as a new manifest URL, forcing shortcut refresh
        const topTwoCards = cards.slice(0, 2).map(c => parseInt(c.dataset.fidelityCardId));
        const shortcutsHash = topTwoCards.length >= 2 
            ? btoa(topTwoCards[0] + ':' + topTwoCards[1]).replace(/[+/=]/g, '').substring(0, 8)
            : topTwoCards.length === 1 
                ? btoa(topTwoCards[0] + ':').replace(/[+/=]/g, '').substring(0, 8)
                : '';
        
        // Update manifest with new shortcuts order
        // Generate new token and add hash to force Chrome to reload manifest with updated shortcuts
        if (currentUser && currentUser.user_id) {
            await updateManifestLinkWithHash(currentUser.user_id, shortcutsHash);
        }
        await updateServiceWorkerForShortcuts();
        
        // Notify user that shortcuts will update on next app restart
        // Chrome doesn't update shortcuts immediately, they're refreshed when app restarts
        if (window.matchMedia('(display-mode: standalone)').matches) {
            // Show a subtle notification
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                bottom: 20px;
                left: 50%;
                transform: translateX(-50%);
                background: #667eea;
                color: white;
                padding: 12px 24px;
                border-radius: 8px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                z-index: 10000;
                font-size: 14px;
                max-width: 90%;
                text-align: center;
            `;
            notification.textContent = 'Gli shortcuts nel menù contestuale si aggiorneranno al prossimo riavvio dell\'app';
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.transition = 'opacity 0.3s';
                notification.style.opacity = '0';
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
    } catch (error) {
        console.error('Error saving fidelity card order:', error);
    } finally {
        isSavingFidelityCardOrder = false;
    }
}

/**
 * Generate secure token for manifest
 * Token format: base64(user_id:timestamp:hmac)
 * This prevents unauthorized access to other users' shortcuts
 */
async function generateManifestToken(userId) {
    try {
        // Get token from server (more secure than generating client-side)
        const response = await fetch('/api/manifest_token.php', {
            method: 'POST',
            credentials: 'include',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ user_id: userId })
        });
        
        if (response.ok) {
            const data = await response.json();
            console.log('Manifest token generated successfully');
            return data.token;
        } else {
            const errorData = await response.json().catch(() => ({ error: 'Unknown error' }));
            console.error('Failed to generate manifest token:', errorData);
            return null;
        }
    } catch (error) {
        console.error('Error generating manifest token:', error);
        return null;
    }
}

/**
 * Update manifest link to include secure token
 * This allows Chrome to read shortcuts even when cookies aren't sent
 * SECURITY: Uses a secure token instead of user_id directly
 */
async function updateManifestLink(userId) {
    await updateManifestLinkWithHash(userId, null);
}

/**
 * Update manifest link with hash to force Chrome to reload when shortcuts change
 * @param {number} userId - User ID
 * @param {string|null} shortcutsHash - Hash of top 2 card IDs (optional)
 */
async function updateManifestLinkWithHash(userId, shortcutsHash = null) {
    const manifestLink = document.querySelector('link[rel="manifest"]');
    if (manifestLink && userId) {
        // Generate secure token
        const token = await generateManifestToken(userId);
        if (!token) {
            console.error('Failed to generate manifest token, manifest link not updated');
            return;
        }
        
        // Build manifest URL with token and optional hash
        let manifestUrl = `/manifest.php?token=${encodeURIComponent(token)}`;
        if (shortcutsHash) {
            manifestUrl += `&v=${shortcutsHash}`;
        }
        
        // Remove old manifest link
        const oldHref = manifestLink.href;
        manifestLink.remove();
        
        // Create new manifest link with secure token and hash
        const newManifestLink = document.createElement('link');
        newManifestLink.rel = 'manifest';
        newManifestLink.href = manifestUrl;
        newManifestLink.id = 'app-manifest';
        document.head.appendChild(newManifestLink);
        
        console.log('Manifest link updated with secure token and hash');
        console.log('Old href:', oldHref);
        console.log('New href:', newManifestLink.href);
        
        // Force Chrome to reload the manifest by dispatching a custom event
        // This might help Chrome recognize the updated manifest
        window.dispatchEvent(new Event('manifestupdated'));
    }
}

/**
 * Update service worker to refresh manifest with shortcuts after authentication
 * This forces Chrome to reload the manifest with updated shortcuts
 */
async function updateServiceWorkerForShortcuts() {
    if ('serviceWorker' in navigator) {
        try {
            const registration = await navigator.serviceWorker.getRegistration();
            if (registration) {
                // More aggressive approach: unregister and re-register service worker
                // This forces Chrome to completely reload the manifest
                try {
                    // First, try to update normally
                    await registration.update();
                    console.log('Service worker updated');
                    
                    // If update doesn't trigger, try unregistering and re-registering
                    // This is more aggressive but ensures Chrome picks up manifest changes
                    const manifestLink = document.querySelector('link[rel="manifest"]');
                    if (manifestLink && manifestLink.href.includes('&v=')) {
                        // Manifest URL has changed (has version hash), force complete refresh
                        console.log('Manifest URL changed, forcing service worker refresh');
                        await registration.unregister();
                        await new Promise(resolve => setTimeout(resolve, 1000));
                        await navigator.serviceWorker.register('/sw.js');
                        console.log('Service worker re-registered to refresh manifest');
                    }
                } catch (error) {
                    // If unregister fails, just try normal update
                    console.log('Service worker update method failed, trying normal update:', error);
                    await registration.update();
                }
                
                // Also try to reload the manifest by fetching it
                // This helps Chrome recognize the updated shortcuts
                try {
                    // Get current user ID from session or from manifest link
                    const manifestLink = document.querySelector('link[rel="manifest"]');
                    const manifestUrl = manifestLink ? manifestLink.href : '/manifest.php';
                    
                    const manifestResponse = await fetch(manifestUrl, {
                        cache: 'no-store',
                        credentials: 'include' // Include cookies for session
                    });
                    const manifest = await manifestResponse.json();
                    console.log('Manifest reloaded with shortcuts:', manifest.shortcuts ? manifest.shortcuts.length : 0);
                    
                    // If we're in an installed PWA, try to trigger manifest refresh
                    if (window.matchMedia('(display-mode: standalone)').matches) {
                        // Force a page reload to pick up new manifest
                        // But only if shortcuts are present and different from before
                        const lastShortcutsHash = localStorage.getItem('lastShortcutsHash');
                        const currentShortcutsHash = manifest.shortcuts ? 
                            JSON.stringify(manifest.shortcuts.map(s => s.url)).substring(0, 50) : '';
                        
                        if (currentShortcutsHash && currentShortcutsHash !== lastShortcutsHash) {
                            localStorage.setItem('lastShortcutsHash', currentShortcutsHash);
                            console.log('Shortcuts changed, manifest should be refreshed on next app launch');
                        }
                    }
                } catch (error) {
                    console.log('Failed to reload manifest (non-critical):', error);
                }
            }
        } catch (error) {
            console.log('Service worker update failed (non-critical):', error);
        }
    }
}

/**
 * Handle URL hash for Android shortcuts (e.g., #fidelity-card-123)
 */
async function handleUrlHash() {
    const hash = window.location.hash;
    if (!hash) {
        return;
    }
    
    // Check if hash matches fidelity card pattern: #fidelity-card-{id}
    const match = hash.match(/^#fidelity-card-(\d+)$/);
    if (match) {
        const cardId = parseInt(match[1], 10);
        if (cardId) {
            // Switch to fidelity cards view first
            switchView('fidelity-cards', false);
            
            // Wait a bit for the view to load, then open the card
            setTimeout(async () => {
                try {
                    await showFidelityCardFullscreen(cardId);
                } catch (error) {
                    console.error('Error opening fidelity card from hash:', error);
                }
            }, 500);
        }
    }
}

async function showFidelityCardFullscreen(cardId) {
    try {
        // Track click
        await API.trackFidelityCardClick(cardId);
        
        // Get card data
        const cards = await API.getFidelityCards();
        const card = cards.find(c => c.id === cardId);
        
        if (!card) {
            alert('Carta non trovata');
            return;
        }
        
        const modalBody = document.getElementById('modal-body');
        
        // Get viewport dimensions
        const viewportWidth = window.innerWidth;
        const viewportHeight = window.innerHeight;
        
        // Card aspect ratio: 85/54 = 1.574
        const cardAspectRatio = 85 / 54;
        
        // Calculate card dimensions to fill viewport
        let cardWidth, cardHeight;
        cardWidth = viewportWidth;
        cardHeight = cardWidth / cardAspectRatio;
        
        if (cardHeight > viewportHeight) {
            cardHeight = viewportHeight;
            cardWidth = cardHeight * cardAspectRatio;
        }
        
        // Create canvas
        const canvas = document.createElement('canvas');
        canvas.width = cardWidth;
        canvas.height = cardHeight;
        const ctx = canvas.getContext('2d');
        
        // Function to draw the card image
        const drawCard = async () => {
            // Clear canvas
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            // Draw background (logo or gradient)
            if (card.logo_path) {
                return new Promise((resolve, reject) => {
                    const logoImg = new Image();
                    logoImg.onload = () => {
                        try {
                            // Draw logo covering entire canvas
                            ctx.drawImage(logoImg, 0, 0, canvas.width, canvas.height);
                            drawBarcodeAndText();
                            resolve();
                        } catch (error) {
                            console.error('Error drawing logo:', error);
                            // Fallback to gradient if drawing fails
                            drawGradientBackground();
                            drawBarcodeAndText();
                            resolve();
                        }
                    };
                    logoImg.onerror = (error) => {
                        console.error('Error loading logo:', error);
                        // Fallback to gradient if logo fails to load
                        drawGradientBackground();
                        drawBarcodeAndText();
                        resolve();
                    };
                    logoImg.src = `/uploads/${card.logo_path}`;
                });
            } else {
                drawGradientBackground();
                drawBarcodeAndText();
                return Promise.resolve();
            }
        };
        
        // Draw gradient background
        const drawGradientBackground = () => {
            const gradient = ctx.createLinearGradient(0, 0, canvas.width, canvas.height);
            gradient.addColorStop(0, '#667eea');
            gradient.addColorStop(1, '#764ba2');
            ctx.fillStyle = gradient;
            ctx.fillRect(0, 0, canvas.width, canvas.height);
        };
        
        // Draw barcode and text
        const drawBarcodeAndText = () => {
            try {
                if (!card.card_code) {
                    console.error('Card code is missing');
                    drawTextOnly();
                    displayCanvas();
                    return;
                }
                
                // Barcode height: 0.23 of total height
                const barcodeHeight = canvas.height * 0.23;
                // Text is 0.05 from bottom, so text Y = canvas.height * 0.95
                // Position barcode so it ends just before text starts
                const textY = canvas.height * 0.95;
                const estimatedTextHeight = canvas.width * 0.05; // Approximate text height
                const barcodeY = textY - barcodeHeight - estimatedTextHeight - (canvas.height * 0.02); // Small gap between barcode and text
                
                // Create temporary SVG for barcode
                const tempSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
                tempSvg.style.position = 'absolute';
                tempSvg.style.left = '-9999px';
                tempSvg.setAttribute('width', canvas.width.toString());
                tempSvg.setAttribute('height', barcodeHeight.toString());
                document.body.appendChild(tempSvg);
                
                if (typeof JsBarcode !== 'undefined') {
                    try {
                        // Generate barcode SVG
                        JsBarcode(tempSvg, card.card_code, {
                            format: "CODE128",
                            width: 2,
                            height: barcodeHeight * 0.8, // 80% of barcode area height
                            displayValue: false,
                            margin: 0
                        });
                        
                        // Get SVG as image
                        const svgData = new XMLSerializer().serializeToString(tempSvg);
                        const svgBlob = new Blob([svgData], { type: 'image/svg+xml;charset=utf-8' });
                        const svgUrl = URL.createObjectURL(svgBlob);
                        
                        const barcodeImg = new Image();
                        barcodeImg.onload = () => {
                            try {
                                // Calculate barcode width and position (centered)
                                const barcodeWidth = Math.min(canvas.width * 0.7, barcodeImg.width);
                                const barcodeX = (canvas.width - barcodeWidth) / 2;
                                
                                // Draw white background for barcode
                                const padding = canvas.width * 0.05;
                                ctx.fillStyle = 'white';
                                ctx.fillRect(
                                    barcodeX - padding,
                                    barcodeY - padding,
                                    barcodeWidth + (padding * 2),
                                    barcodeHeight + (padding * 2)
                                );
                                
                                // Draw barcode
                                ctx.drawImage(
                                    barcodeImg,
                                    barcodeX,
                                    barcodeY,
                                    barcodeWidth,
                                    barcodeHeight
                                );
                                
                                // Draw text at 0.05 from bottom
                                const textY = canvas.height * 0.95;
                                const fontSize = Math.max(20, canvas.width * 0.05);
                                ctx.fillStyle = 'white';
                                ctx.font = `bold ${fontSize}px Arial`;
                                ctx.textAlign = 'center';
                                ctx.textBaseline = 'bottom';
                                ctx.shadowColor = 'rgba(0, 0, 0, 0.7)';
                                ctx.shadowBlur = 4;
                                ctx.shadowOffsetX = 2;
                                ctx.shadowOffsetY = 2;
                                ctx.fillText(card.card_code, canvas.width / 2, textY);
                                
                                // Clean up
                                if (document.body.contains(tempSvg)) {
                                    document.body.removeChild(tempSvg);
                                }
                                URL.revokeObjectURL(svgUrl);
                                
                                // Display canvas
                                displayCanvas();
                            } catch (drawError) {
                                console.error('Error drawing barcode:', drawError);
                                if (document.body.contains(tempSvg)) {
                                    document.body.removeChild(tempSvg);
                                }
                                URL.revokeObjectURL(svgUrl);
                                drawTextOnly();
                                displayCanvas();
                            }
                        };
                        barcodeImg.onerror = () => {
                            if (document.body.contains(tempSvg)) {
                                document.body.removeChild(tempSvg);
                            }
                            URL.revokeObjectURL(svgUrl);
                            drawTextOnly();
                            displayCanvas();
                        };
                        barcodeImg.src = svgUrl;
                    } catch (barcodeError) {
                        console.error('Error generating barcode:', barcodeError);
                        if (document.body.contains(tempSvg)) {
                            document.body.removeChild(tempSvg);
                        }
                        drawTextOnly();
                        displayCanvas();
                    }
                } else {
                    console.warn('JsBarcode library not loaded');
                    if (document.body.contains(tempSvg)) {
                        document.body.removeChild(tempSvg);
                    }
                    drawTextOnly();
                    displayCanvas();
                }
            } catch (error) {
                console.error('Error in drawBarcodeAndText:', error);
                drawTextOnly();
                displayCanvas();
            }
        };
        
        // Draw text only (fallback)
        const drawTextOnly = () => {
            const textY = canvas.height * 0.95;
            const fontSize = Math.max(20, canvas.width * 0.05);
            ctx.fillStyle = 'white';
            ctx.font = `bold ${fontSize}px Arial`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'bottom';
            ctx.shadowColor = 'rgba(0, 0, 0, 0.7)';
            ctx.shadowBlur = 4;
            ctx.shadowOffsetX = 2;
            ctx.shadowOffsetY = 2;
            ctx.fillText(card.card_code, canvas.width / 2, textY);
        };
        
        // Display canvas fullscreen
        const displayCanvas = () => {
            const imageDataUrl = canvas.toDataURL('image/png');
            modalBody.innerHTML = `
                <div id="fidelity-card-fullscreen-container" style="
                    width: 100%;
                    height: 100%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    background: #000;
                    margin: 0;
                    padding: 0;
                    overflow: hidden;
                    touch-action: none;
                    -webkit-overflow-scrolling: none;
                ">
                    <img src="${imageDataUrl}" style="
                        max-width: 100%;
                        max-height: 100%;
                        width: auto;
                        height: auto;
                        object-fit: contain;
                        display: block;
                        touch-action: none;
                        -webkit-user-select: none;
                        user-select: none;
                    " alt="Fidelity Card">
                </div>
            `;
            // Add fullscreen class to modal content
            const modalContent = document.querySelector('.modal-content');
            if (modalContent) {
                modalContent.classList.add('fullscreen');
            }
            // Remove padding from modal overlay
            const modalOverlay = document.getElementById('modal-overlay');
            if (modalOverlay) {
                modalOverlay.style.padding = '0';
            }
            // Prevent body scrolling
            document.body.classList.add('modal-fullscreen-open');
            // Prevent touch scrolling (but allow clicks on close button)
            const preventScroll = (e) => {
                if (e.touches.length > 1) {
                    return; // Allow pinch zoom
                }
                // Allow clicks on modal close button
                const target = e.target;
                if (target && (target.classList.contains('modal-close') || target.closest('.modal-close'))) {
                    return; // Allow click on close button
                }
                e.preventDefault();
            };
            document.addEventListener('touchmove', preventScroll, { passive: false });
            document.addEventListener('touchstart', preventScroll, { passive: false });
            window.fidelityCardPreventScroll = preventScroll;
            
            // Add history state for back button support
            if (window.history && window.history.pushState) {
                window.history.pushState({ modalOpen: true }, '');
                window.fidelityCardHistoryState = true;
            }
            
            showModal();
        };
        
        // Start drawing
        await drawCard();
    } catch (error) {
        console.error('Error showing fidelity card fullscreen:', error);
        alert('Errore: ' + error.message);
    }
}

async function showAddFidelityCardModal() {
    console.log('showAddFidelityCardModal called');
    const modalBody = document.getElementById('modal-body');
    
    if (!modalBody) {
        console.error('Modal body not found');
        alert('Errore: modal non trovato');
        return;
    }
    
    // Load available brands
    let brands = [];
    try {
        brands = await API.getFidelityCardBrands();
    } catch (error) {
        console.log('Could not load brands (non-critical):', error);
    }
    
    modalBody.innerHTML = `
        <h2>Aggiungi Carta Fedeltà</h2>
        <form id="add-fidelity-card-form">
            <div class="form-group">
                <label>Nome</label>
                <input type="text" name="name" required placeholder="Es. Carta Coop">
            </div>
            <div class="form-group">
                <label>Negozio/Insegna</label>
                ${brands.length > 0 ? `
                    <select id="store-name-select" name="store_name_select" style="width: 100%; padding: 0.5rem; margin-bottom: 0.5rem;">
                        <option value="">Seleziona insegna esistente...</option>
                        ${brands.map(brand => `<option value="${brand.id}" data-logo-path="${brand.logo_path || ''}">${escapeHtml(brand.name)}</option>`).join('')}
                        <option value="__custom__">Altro (inserisci manualmente)</option>
                    </select>
                    <input type="text" id="store-name-input" name="store_name" required placeholder="Es. Coop" style="display: none; width: 100%; padding: 0.5rem;">
                ` : `
                    <input type="text" name="store_name" required placeholder="Es. Coop">
                `}
            </div>
            <div class="form-group">
                <label>Codice Carta</label>
                <input type="text" name="card_code" required placeholder="Codice alfanumerico">
            </div>
            <div class="form-group">
                <label>Descrizione/Note</label>
                <textarea name="description" rows="3" placeholder="Note opzionali"></textarea>
            </div>
            <div class="form-group">
                <label>Logo</label>
                <div id="selected-brand-logo" style="display: none; margin-bottom: 0.5rem;">
                    <img id="brand-logo-preview" src="" style="max-width: 200px; max-height: 100px; border-radius: 4px;">
                    <p style="color: #666; font-size: 0.9em; margin-top: 0.25rem;">Logo selezionato dall'insegna</p>
                </div>
                <input type="file" id="logo-file-input" name="logo" accept="image/*">
                <small style="color: #666; display: block; margin-top: 0.25rem;">Lascia vuoto se hai selezionato un'insegna esistente</small>
            </div>
            <div class="form-group">
                <label>Immagine Carta</label>
                <input type="file" name="card_image" accept="image/*">
            </div>
            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                <button type="submit" class="btn btn-primary">Salva</button>
            </div>
        </form>
    `;
    
    // Handle brand selection
    if (brands.length > 0) {
        const storeNameSelect = document.getElementById('store-name-select');
        const storeNameInput = document.getElementById('store-name-input');
        const logoFileInput = document.getElementById('logo-file-input');
        const selectedBrandLogo = document.getElementById('selected-brand-logo');
        const brandLogoPreview = document.getElementById('brand-logo-preview');
        
        storeNameSelect.addEventListener('change', (e) => {
            const selectedOption = e.target.options[e.target.selectedIndex];
            const brandId = e.target.value;
            
            if (brandId === '__custom__') {
                storeNameSelect.style.display = 'none';
                storeNameInput.style.display = 'block';
                storeNameInput.required = true;
                selectedBrandLogo.style.display = 'none';
                logoFileInput.disabled = false;
            } else if (brandId && brandId !== '') {
                const logoPath = selectedOption.dataset.logoPath;
                storeNameInput.value = selectedOption.text;
                storeNameInput.style.display = 'block';
                storeNameInput.required = true;
                
                if (logoPath) {
                    brandLogoPreview.src = '/uploads/' + logoPath;
                    selectedBrandLogo.style.display = 'block';
                    logoFileInput.disabled = true;
                } else {
                    selectedBrandLogo.style.display = 'none';
                    logoFileInput.disabled = false;
                }
            } else {
                storeNameInput.style.display = 'none';
                storeNameInput.required = false;
                selectedBrandLogo.style.display = 'none';
                logoFileInput.disabled = false;
            }
        });
    }
    
    document.getElementById('add-fidelity-card-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData(e.target);
        
        let storeName = formData.get('store_name') || formData.get('store_name_select');
        const selectedBrandId = formData.get('store_name_select');
        
        const data = {
            name: formData.get('name'),
            store_name: storeName,
            card_code: formData.get('card_code'),
            description: formData.get('description') || null
        };
        
        let logo = formData.get('logo') && formData.get('logo').size > 0 ? formData.get('logo') : null;
        
        // If a brand was selected, pass brand_id to backend for logo copy
        if (selectedBrandId && selectedBrandId !== '' && selectedBrandId !== '__custom__') {
            data.brand_id = selectedBrandId;
            logo = null; // Don't upload logo file, backend will copy it from brand
        }
        
        const cardImage = formData.get('card_image') && formData.get('card_image').size > 0 ? formData.get('card_image') : null;
        
        try {
            // Check if function exists
            if (typeof API.createFidelityCard !== 'function') {
                console.error('API.createFidelityCard is not a function. Available methods:', Object.getOwnPropertyNames(API).filter(name => name.includes('Fidelity')));
                alert('Errore: funzione API non disponibile. Ricarica la pagina.');
                return;
            }
            await API.createFidelityCard(data, logo, cardImage);
            closeModal();
            await loadFidelityCards();
            // Update manifest with new shortcuts
            await updateServiceWorkerForShortcuts();
        } catch (error) {
            console.error('Error creating fidelity card:', error);
            alert('Errore: ' + error.message);
        }
    });
    
    showModal();
}

async function showEditFidelityCardModal(cardId) {
    try {
        const cards = await API.getFidelityCards();
        const card = cards.find(c => c.id === cardId);
        
        if (!card) {
            alert('Carta non trovata');
            return;
        }
        
        // Load available brands
        let brands = [];
        try {
            brands = await API.getFidelityCardBrands();
        } catch (error) {
            console.log('Could not load brands (non-critical):', error);
        }
        
        const modalBody = document.getElementById('modal-body');
        
        modalBody.innerHTML = `
            <h2>Modifica Carta Fedeltà</h2>
            <form id="edit-fidelity-card-form">
                <div class="form-group">
                    <label>Nome</label>
                    <input type="text" name="name" value="${escapeHtml(card.name)}" required>
                </div>
                <div class="form-group">
                    <label>Negozio/Insegna</label>
                    ${brands.length > 0 ? `
                        <select id="edit-store-name-select" name="store_name_select" style="width: 100%; padding: 0.5rem; margin-bottom: 0.5rem;">
                            <option value="">Seleziona insegna esistente...</option>
                            ${brands.map(brand => `<option value="${brand.id}" data-logo-path="${brand.logo_path || ''}" ${brand.name === card.store_name ? 'selected' : ''}>${escapeHtml(brand.name)}</option>`).join('')}
                            <option value="__custom__" ${brands.find(b => b.name === card.store_name) ? '' : 'selected'}>Altro (inserisci manualmente)</option>
                        </select>
                        <input type="text" id="edit-store-name-input" name="store_name" value="${escapeHtml(card.store_name)}" required style="${brands.find(b => b.name === card.store_name) ? 'display: none;' : ''} width: 100%; padding: 0.5rem;">
                    ` : `
                        <input type="text" name="store_name" value="${escapeHtml(card.store_name)}" required>
                    `}
                </div>
                <div class="form-group">
                    <label>Codice Carta</label>
                    <input type="text" name="card_code" value="${escapeHtml(card.card_code)}" required>
                </div>
                <div class="form-group">
                    <label>Descrizione/Note</label>
                    <textarea name="description" rows="3">${escapeHtml(card.description || '')}</textarea>
                </div>
                <div class="form-group">
                    <label>Logo</label>
                    <div id="edit-selected-brand-logo" style="display: none; margin-bottom: 0.5rem;">
                        <img id="edit-brand-logo-preview" src="" style="max-width: 200px; max-height: 100px; border-radius: 4px;">
                        <p style="color: #666; font-size: 0.9em; margin-top: 0.25rem;">Logo selezionato dall'insegna</p>
                    </div>
                    ${card.logo_path ? `<div style="margin-bottom: 0.5rem;"><img src="/uploads/${card.logo_path}" style="max-width: 200px; border-radius: 4px;"></div>` : ''}
                    <input type="file" id="edit-logo-file-input" name="logo" accept="image/*">
                    <small style="color: #666; display: block; margin-top: 0.25rem;">Lascia vuoto se hai selezionato un'insegna esistente</small>
                    ${card.logo_path ? `<label style="display: flex; align-items: center; gap: 8px; margin-top: 0.5rem;"><input type="checkbox" name="remove_logo"> Rimuovi logo</label>` : ''}
                </div>
                <div class="form-group">
                    <label>Immagine Carta</label>
                    ${card.card_image_path ? `<div style="margin-bottom: 0.5rem;"><img src="/uploads/${card.card_image_path}" style="max-width: 200px; border-radius: 4px;"></div>` : ''}
                    <input type="file" name="card_image" accept="image/*">
                    ${card.card_image_path ? `<label style="display: flex; align-items: center; gap: 8px; margin-top: 0.5rem;"><input type="checkbox" name="remove_card_image"> Rimuovi immagine</label>` : ''}
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annulla</button>
                    <button type="submit" class="btn btn-primary">Salva</button>
                </div>
            </form>
        `;
        
        // Handle brand selection
        if (brands.length > 0) {
            const storeNameSelect = document.getElementById('edit-store-name-select');
            const storeNameInput = document.getElementById('edit-store-name-input');
            const logoFileInput = document.getElementById('edit-logo-file-input');
            const selectedBrandLogo = document.getElementById('edit-selected-brand-logo');
            const brandLogoPreview = document.getElementById('edit-brand-logo-preview');
            
            // Check if current store_name matches a brand
            const matchingBrand = brands.find(b => b.name === card.store_name);
            if (matchingBrand && matchingBrand.logo_path) {
                brandLogoPreview.src = '/uploads/' + matchingBrand.logo_path;
                selectedBrandLogo.style.display = 'block';
                logoFileInput.disabled = true;
            }
            
            storeNameSelect.addEventListener('change', (e) => {
                const selectedOption = e.target.options[e.target.selectedIndex];
                const brandId = e.target.value;
                
                if (brandId === '__custom__') {
                    storeNameSelect.style.display = 'none';
                    storeNameInput.style.display = 'block';
                    storeNameInput.required = true;
                    selectedBrandLogo.style.display = 'none';
                    logoFileInput.disabled = false;
                } else if (brandId && brandId !== '') {
                    const logoPath = selectedOption.dataset.logoPath;
                    storeNameInput.value = selectedOption.text;
                    storeNameInput.style.display = 'block';
                    storeNameInput.required = true;
                    
                    if (logoPath) {
                        brandLogoPreview.src = '/uploads/' + logoPath;
                        selectedBrandLogo.style.display = 'block';
                        logoFileInput.disabled = true;
                    } else {
                        selectedBrandLogo.style.display = 'none';
                        logoFileInput.disabled = false;
                    }
                } else {
                    storeNameInput.style.display = 'none';
                    storeNameInput.required = false;
                    selectedBrandLogo.style.display = 'none';
                    logoFileInput.disabled = false;
                }
            });
        }
        
        document.getElementById('edit-fidelity-card-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            let storeName = formData.get('store_name') || formData.get('store_name_select');
            const selectedBrandId = formData.get('store_name_select');
            
            const data = {
                id: cardId,
                name: formData.get('name'),
                store_name: storeName,
                card_code: formData.get('card_code'),
                description: formData.get('description') || null,
                remove_logo: formData.get('remove_logo') === 'on',
                remove_card_image: formData.get('remove_card_image') === 'on'
            };
            
            let logo = formData.get('logo') && formData.get('logo').size > 0 ? formData.get('logo') : null;
            
            // If a brand was selected, pass brand_id to backend for logo copy
            if (selectedBrandId && selectedBrandId !== '' && selectedBrandId !== '__custom__' && !data.remove_logo) {
                data.brand_id = selectedBrandId;
                logo = null; // Don't upload logo file, backend will copy it from brand
            }
            
            const cardImage = formData.get('card_image') && formData.get('card_image').size > 0 ? formData.get('card_image') : null;
            
            try {
                await API.updateFidelityCard(cardId, data, logo, cardImage);
                closeModal();
                await loadFidelityCards();
                // Update manifest with new shortcuts
                await updateServiceWorkerForShortcuts();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function deleteFidelityCard(cardId) {
    if (!confirm('Sei sicuro di voler eliminare questa carta fedeltà?')) {
        return;
    }
    
    try {
        await API.deleteFidelityCard(cardId);
        await loadFidelityCards();
        // Update manifest with new shortcuts
        await updateServiceWorkerForShortcuts();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

let currentManageSharesFidelityCardId = null;

async function showFidelityCardSharesModal(fidelityCardId) {
    try {
        currentManageSharesFidelityCardId = fidelityCardId;
        
        // Load shares data, card data, and previously shared users in parallel
        const [sharesData, cards, previouslySharedUsers] = await Promise.all([
            API.getFidelityCardShares(fidelityCardId),
            API.getFidelityCards(),
            API.getPreviouslySharedUsersForFidelityCards()
        ]);
        
        const card = cards.find(c => c.id === fidelityCardId);
        if (!card) {
            alert('Carta fedeltà non trovata');
            return;
        }
        
        // Determine which image to show (prefer card_image, fallback to logo, then placeholder)
        const cardImageUrl = card.card_image_path 
            ? `/uploads/${card.card_image_path}` 
            : (card.logo_path ? `/uploads/${card.logo_path}` : null);
        
        const modalBody = document.getElementById('modal-body');
        modalBody.innerHTML = `
            <h2>Gestisci Condivisioni</h2>
            <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1.5rem; padding: 1rem; background: #f8f9fa; border-radius: 8px;">
                ${cardImageUrl ? `
                    <img src="${escapeHtml(cardImageUrl)}" 
                         alt="${escapeHtml(card.name)}" 
                         style="width: 80px; height: 50px; object-fit: cover; border-radius: 4px; border: 1px solid #ddd;">
                ` : `
                    <div style="width: 80px; height: 50px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 4px; display: flex; align-items: center; justify-content: center; color: white; font-size: 0.7rem; text-align: center; padding: 0.25rem;">
                        ${escapeHtml(card.name.substring(0, 10))}
                    </div>
                `}
                <div>
                    <div style="font-weight: 600; font-size: 1.1rem; margin-bottom: 0.25rem;">${escapeHtml(card.name)}</div>
                    <div style="color: #666; font-size: 0.9rem;">${escapeHtml(card.store_name)}</div>
                </div>
            </div>
            <div style="margin-bottom: 2rem;">
                <h3>Condivisioni Attive</h3>
                ${sharesData.shares && sharesData.shares.length > 0 ? `
                    <ul style="list-style: none; padding: 0;">
                        ${sharesData.shares.map(share => `
                            <li style="padding: 0.5rem; background: #f5f5f5; margin-bottom: 0.5rem; border-radius: 4px; display: flex; justify-content: space-between; align-items: center;">
                                <span>${escapeHtml(share.name || share.email)}</span>
                                <button class="btn btn-secondary" onclick="removeFidelityCardShare(${share.id})" style="padding: 0.25rem 0.5rem; font-size: 0.85rem;">Rimuovi</button>
                            </li>
                        `).join('')}
                    </ul>
                ` : '<p style="color: #666;">Nessuna condivisione attiva</p>'}
            </div>
            
            <div style="margin-bottom: 2rem;">
                <h3>Inviti Pendenti</h3>
                ${sharesData.invitations && sharesData.invitations.length > 0 ? `
                    <ul style="list-style: none; padding: 0;">
                        ${sharesData.invitations.map(inv => `
                            <li style="padding: 0.5rem; background: #f5f5f5; margin-bottom: 0.5rem; border-radius: 4px; display: flex; justify-content: space-between; align-items: center;">
                                <span>${escapeHtml(inv.recipient_email)}</span>
                                <button class="btn btn-secondary" onclick="removeFidelityCardInvitation(${inv.id})" style="padding: 0.25rem 0.5rem; font-size: 0.85rem;">Rimuovi</button>
                            </li>
                        `).join('')}
                    </ul>
                ` : '<p style="color: #666;">Nessun invito pendente</p>'}
            </div>
            
            <div>
                <h3>Condividi con Altri Utenti</h3>
                <form id="share-fidelity-card-form">
                    <div class="form-group">
                        <label for="recipient-email">Email utente:</label>
                        <input type="email" id="recipient-email" list="previously-shared-users" required 
                               placeholder="Inserisci email utente" style="width: 100%; padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;">
                        <datalist id="previously-shared-users">
                            ${previouslySharedUsers.map(user => `<option value="${escapeHtml(user.email)}">${escapeHtml(user.name || user.email)}</option>`).join('')}
                        </datalist>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeModal()">Chiudi</button>
                        <button type="submit" class="btn btn-primary">Condividi</button>
                    </div>
                </form>
            </div>
        `;
        
        document.getElementById('share-fidelity-card-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const recipientEmail = document.getElementById('recipient-email').value.trim();
            
            try {
                await API.shareFidelityCard(fidelityCardId, recipientEmail);
                await showFidelityCardSharesModal(fidelityCardId);
                loadFidelityCards();
            } catch (error) {
                alert('Errore: ' + error.message);
            }
        });
        
        showModal();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function removeFidelityCardShare(shareId) {
    if (!confirm('Sei sicuro di voler rimuovere questa condivisione?')) {
        return;
    }
    
    try {
        await API.removeFidelityCardShare(shareId);
        if (currentManageSharesFidelityCardId) {
            await showFidelityCardSharesModal(currentManageSharesFidelityCardId);
        }
        loadFidelityCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function removeFidelityCardInvitation(invitationId) {
    if (!confirm('Sei sicuro di voler rimuovere questo invito?')) {
        return;
    }
    
    try {
        await API.removeFidelityCardInvitation(invitationId);
        alert('Invito rimosso con successo');
        if (currentManageSharesFidelityCardId) {
            await showFidelityCardSharesModal(currentManageSharesFidelityCardId);
        }
        loadFidelityCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}

async function stopSharingFidelityCard(fidelityCardId) {
    if (!confirm('Sei sicuro di voler rimuovere questa condivisione?')) {
        return;
    }
    
    try {
        const cards = await API.getFidelityCards();
        const card = cards.find(c => c.id === fidelityCardId && c.card_status === 'shared');
        
        if (!card || !card.share_id) {
            alert('Condivisione non trovata');
            return;
        }
        
        await API.removeFidelityCardShare(card.share_id);
        loadFidelityCards();
    } catch (error) {
        alert('Errore: ' + error.message);
    }
}
