<?php
/**
 * Users Management API (Admin only)
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'GET':
            requireAdmin();
            $db = new Database();
            $conn = $db->getConnection();
            
            // Check if requesting data counts for a specific user
            $user_id = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
            
            if ($user_id) {
                // Get data counts for a specific user
                $counts = [];
                
                // Accounts
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM accounts WHERE user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['accounts'] = (int)$stmt->fetch()['count'];
                
                // Prepaid cards
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM prepaid_cards WHERE user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['prepaid_cards'] = (int)$stmt->fetch()['count'];
                
                // Fidelity cards
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM fidelity_cards WHERE user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['fidelity_cards'] = (int)$stmt->fetch()['count'];
                
                // Expenses (movimenti)
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM expenses WHERE user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['expenses'] = (int)$stmt->fetch()['count'];
                
                // Securities (through accounts)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM securities s
                    JOIN accounts a ON s.account_id = a.id
                    WHERE a.user_id = :user_id
                ");
                $stmt->execute([':user_id' => $user_id]);
                $counts['securities'] = (int)$stmt->fetch()['count'];
                
                // Fund transfers
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM fund_transfers WHERE user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['fund_transfers'] = (int)$stmt->fetch()['count'];
                
                // Expense attachments (through expenses)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM expense_attachments ea
                    JOIN expenses e ON ea.expense_id = e.id
                    WHERE e.user_id = :user_id
                ");
                $stmt->execute([':user_id' => $user_id]);
                $counts['expense_attachments'] = (int)$stmt->fetch()['count'];
                
                // Securities attachments (through securities -> accounts)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM securities_attachments sa
                    JOIN securities s ON sa.security_id = s.id
                    JOIN accounts a ON s.account_id = a.id
                    WHERE a.user_id = :user_id
                ");
                $stmt->execute([':user_id' => $user_id]);
                $counts['securities_attachments'] = (int)$stmt->fetch()['count'];
                
                // Transfer attachments (through fund_transfers)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM transfer_attachments ta
                    JOIN fund_transfers ft ON ta.transfer_id = ft.id
                    WHERE ft.user_id = :user_id
                ");
                $stmt->execute([':user_id' => $user_id]);
                $counts['transfer_attachments'] = (int)$stmt->fetch()['count'];
                
                // Account shares (as owner or shared with)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM account_shares 
                    WHERE owner_user_id = :user_id OR shared_with_user_id = :user_id2
                ");
                $stmt->execute([':user_id' => $user_id, ':user_id2' => $user_id]);
                $counts['account_shares'] = (int)$stmt->fetch()['count'];
                
                // Account share invitations (as owner)
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM account_share_invitations WHERE owner_user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['account_share_invitations'] = (int)$stmt->fetch()['count'];
                
                // Prepaid card shares (as owner or shared with)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM prepaid_card_shares 
                    WHERE owner_user_id = :user_id OR shared_with_user_id = :user_id2
                ");
                $stmt->execute([':user_id' => $user_id, ':user_id2' => $user_id]);
                $counts['prepaid_card_shares'] = (int)$stmt->fetch()['count'];
                
                // Prepaid card share invitations (as owner)
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM prepaid_card_share_invitations WHERE owner_user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['prepaid_card_share_invitations'] = (int)$stmt->fetch()['count'];
                
                // Fidelity card shares (as owner or shared with)
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM fidelity_card_shares 
                    WHERE owner_user_id = :user_id OR shared_with_user_id = :user_id2
                ");
                $stmt->execute([':user_id' => $user_id, ':user_id2' => $user_id]);
                $counts['fidelity_card_shares'] = (int)$stmt->fetch()['count'];
                
                // Fidelity card share invitations (as owner)
                $stmt = $conn->prepare("SELECT COUNT(*) as count FROM fidelity_card_share_invitations WHERE owner_user_id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $counts['fidelity_card_share_invitations'] = (int)$stmt->fetch()['count'];
                
                echo json_encode($counts);
            } else {
                // Get all users (admin only)
                $stmt = $conn->prepare("
                    SELECT 
                        id,
                        google_id,
                        email,
                        name,
                        picture_url,
                        default_currency,
                        is_admin,
                        is_blocked,
                        created_at,
                        updated_at
                    FROM users
                    ORDER BY created_at DESC
                ");
                $stmt->execute();
                $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Remove sensitive data
                foreach ($users as &$user) {
                    unset($user['google_id']);
                    $user['is_admin'] = (bool)$user['is_admin'];
                    $user['is_blocked'] = (bool)$user['is_blocked'];
                }
                
                echo json_encode($users);
            }
            break;
            
        case 'PUT':
            // Update user (admin only)
            requireAdmin();
            $db = new Database();
            $conn = $db->getConnection();
            
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
            
            if (!isset($data['id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'User ID is required']);
                break;
            }
            
            $user_id = (int)$data['id'];
            $admin_id = $_SESSION['user_id'];
            
            // Prevent admin from modifying themselves
            if ($user_id == $admin_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Cannot modify your own account']);
                break;
            }
            
            // Check if user exists
            $stmt = $conn->prepare("SELECT id FROM users WHERE id = :id");
            $stmt->execute([':id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'User not found']);
                break;
            }
            
            // Build update query
            $updates = [];
            $params = [':id' => $user_id];
            
            if (isset($data['is_admin'])) {
                $updates[] = 'is_admin = :is_admin';
                $params[':is_admin'] = $data['is_admin'] ? 1 : 0;
            }
            
            if (isset($data['is_blocked'])) {
                $updates[] = 'is_blocked = :is_blocked';
                $params[':is_blocked'] = $data['is_blocked'] ? 1 : 0;
            }
            
            if (empty($updates)) {
                http_response_code(400);
                echo json_encode(['error' => 'No fields to update']);
                break;
            }
            
            $updates[] = 'updated_at = CURRENT_TIMESTAMP';
            $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = :id";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            echo json_encode(['success' => true, 'message' => 'User updated successfully']);
            break;
            
        case 'DELETE':
            // Delete user (admin only)
            requireAdmin();
            $db = new Database();
            $conn = $db->getConnection();
            
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
            
            if (!isset($data['id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'User ID is required']);
                break;
            }
            
            $user_id = (int)$data['id'];
            $admin_id = $_SESSION['user_id'];
            
            // Prevent admin from deleting themselves
            if ($user_id == $admin_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Cannot delete your own account']);
                break;
            }
            
            // Check if user exists
            $stmt = $conn->prepare("SELECT id FROM users WHERE id = :id");
            $stmt->execute([':id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'User not found']);
                break;
            }
            
            // Check if user is the last admin
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM users WHERE is_admin = 1");
            $stmt->execute();
            $admin_count = (int)$stmt->fetch()['count'];
            
            $stmt = $conn->prepare("SELECT is_admin FROM users WHERE id = :id");
            $stmt->execute([':id' => $user_id]);
            $user_is_admin = (bool)$stmt->fetch()['is_admin'];
            
            if ($user_is_admin && $admin_count <= 1) {
                http_response_code(400);
                echo json_encode(['error' => 'Cannot delete the last admin user']);
                break;
            }
            
            // Get deletion flags
            $delete_flags = $data['delete_flags'] ?? [];
            
            // Check if user has any data
            $has_data = false;
            $data_checks = [
                'accounts', 'prepaid_cards', 'fidelity_cards', 'expenses', 
                'securities', 'fund_transfers', 'expense_attachments', 
                'securities_attachments', 'transfer_attachments',
                'account_shares', 'account_share_invitations',
                'prepaid_card_shares', 'prepaid_card_share_invitations',
                'fidelity_card_shares', 'fidelity_card_share_invitations'
            ];
            
            foreach ($data_checks as $check) {
                $stmt = null;
                switch ($check) {
                    case 'accounts':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM accounts WHERE user_id = :user_id");
                        break;
                    case 'prepaid_cards':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM prepaid_cards WHERE user_id = :user_id");
                        break;
                    case 'fidelity_cards':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM fidelity_cards WHERE user_id = :user_id");
                        break;
                    case 'expenses':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM expenses WHERE user_id = :user_id");
                        break;
                    case 'securities':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM securities s
                            JOIN accounts a ON s.account_id = a.id
                            WHERE a.user_id = :user_id
                        ");
                        break;
                    case 'fund_transfers':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM fund_transfers WHERE user_id = :user_id");
                        break;
                    case 'expense_attachments':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM expense_attachments ea
                            JOIN expenses e ON ea.expense_id = e.id
                            WHERE e.user_id = :user_id
                        ");
                        break;
                    case 'securities_attachments':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM securities_attachments sa
                            JOIN securities s ON sa.security_id = s.id
                            JOIN accounts a ON s.account_id = a.id
                            WHERE a.user_id = :user_id
                        ");
                        break;
                    case 'transfer_attachments':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM transfer_attachments ta
                            JOIN fund_transfers ft ON ta.transfer_id = ft.id
                            WHERE ft.user_id = :user_id
                        ");
                        break;
                    case 'account_shares':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM account_shares 
                            WHERE owner_user_id = :user_id OR shared_with_user_id = :user_id2
                        ");
                        break;
                    case 'account_share_invitations':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM account_share_invitations WHERE owner_user_id = :user_id");
                        break;
                    case 'prepaid_card_shares':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM prepaid_card_shares 
                            WHERE owner_user_id = :user_id OR shared_with_user_id = :user_id2
                        ");
                        break;
                    case 'prepaid_card_share_invitations':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM prepaid_card_share_invitations WHERE owner_user_id = :user_id");
                        break;
                    case 'fidelity_card_shares':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) as count 
                            FROM fidelity_card_shares 
                            WHERE owner_user_id = :user_id OR shared_with_user_id = :user_id2
                        ");
                        break;
                    case 'fidelity_card_share_invitations':
                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM fidelity_card_share_invitations WHERE owner_user_id = :user_id");
                        break;
                }
                
                if ($stmt) {
                    // For queries with OR conditions, need to pass user_id twice
                    if (in_array($check, ['account_shares', 'prepaid_card_shares', 'fidelity_card_shares'])) {
                        $stmt->execute([':user_id' => $user_id, ':user_id2' => $user_id]);
                    } else {
                        $stmt->execute([':user_id' => $user_id]);
                    }
                    $count = (int)$stmt->fetch()['count'];
                    if ($count > 0) {
                        $has_data = true;
                        break;
                    }
                }
            }
            
            // If user has data, validate that at least one flag is set
            if ($has_data) {
                $has_any_flag = false;
                foreach ($delete_flags as $flag) {
                    if ($flag === true) {
                        $has_any_flag = true;
                        break;
                    }
                }
                
                if (!$has_any_flag) {
                    http_response_code(400);
                    echo json_encode(['error' => 'At least one deletion flag must be set']);
                    break;
                }
            }
            
            // Start transaction
            $conn->beginTransaction();
            
            try {
                // Delete files first (before database records)
                // Expense attachments files
                if (isset($delete_flags['expense_attachments']) && $delete_flags['expense_attachments']) {
                    $stmt = $conn->prepare("
                        SELECT ea.file_path 
                        FROM expense_attachments ea
                        JOIN expenses e ON ea.expense_id = e.id
                        WHERE e.user_id = :user_id AND ea.file_path IS NOT NULL
                    ");
                    $stmt->execute([':user_id' => $user_id]);
                    $attachments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($attachments as $attachment) {
                        if ($attachment['file_path'] && file_exists(UPLOAD_DIR . $attachment['file_path'])) {
                            unlink(UPLOAD_DIR . $attachment['file_path']);
                        }
                    }
                }
                
                // Securities attachments files
                if (isset($delete_flags['securities_attachments']) && $delete_flags['securities_attachments']) {
                    $stmt = $conn->prepare("
                        SELECT sa.file_path 
                        FROM securities_attachments sa
                        JOIN securities s ON sa.security_id = s.id
                        JOIN accounts a ON s.account_id = a.id
                        WHERE a.user_id = :user_id AND sa.file_path IS NOT NULL
                    ");
                    $stmt->execute([':user_id' => $user_id]);
                    $attachments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($attachments as $attachment) {
                        if ($attachment['file_path'] && file_exists(UPLOAD_DIR . $attachment['file_path'])) {
                            unlink(UPLOAD_DIR . $attachment['file_path']);
                        }
                    }
                }
                
                // Transfer attachments files
                if (isset($delete_flags['transfer_attachments']) && $delete_flags['transfer_attachments']) {
                    $stmt = $conn->prepare("
                        SELECT ta.file_path 
                        FROM transfer_attachments ta
                        JOIN fund_transfers ft ON ta.transfer_id = ft.id
                        WHERE ft.user_id = :user_id AND ta.file_path IS NOT NULL
                    ");
                    $stmt->execute([':user_id' => $user_id]);
                    $attachments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($attachments as $attachment) {
                        if ($attachment['file_path'] && file_exists(UPLOAD_DIR . $attachment['file_path'])) {
                            unlink(UPLOAD_DIR . $attachment['file_path']);
                        }
                    }
                }
                
                // Account logos
                if (isset($delete_flags['accounts']) && $delete_flags['accounts']) {
                    $stmt = $conn->prepare("SELECT logo_path FROM accounts WHERE user_id = :user_id AND logo_path IS NOT NULL");
                    $stmt->execute([':user_id' => $user_id]);
                    $logos = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($logos as $logo) {
                        if ($logo['logo_path'] && file_exists(UPLOAD_DIR . $logo['logo_path'])) {
                            unlink(UPLOAD_DIR . $logo['logo_path']);
                        }
                    }
                }
                
                // Prepaid card photos
                if (isset($delete_flags['prepaid_cards']) && $delete_flags['prepaid_cards']) {
                    $stmt = $conn->prepare("SELECT photo_path FROM prepaid_cards WHERE user_id = :user_id AND photo_path IS NOT NULL");
                    $stmt->execute([':user_id' => $user_id]);
                    $photos = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($photos as $photo) {
                        if ($photo['photo_path'] && file_exists(UPLOAD_DIR . $photo['photo_path'])) {
                            unlink(UPLOAD_DIR . $photo['photo_path']);
                        }
                    }
                }
                
                // Fidelity card logos and images
                if (isset($delete_flags['fidelity_cards']) && $delete_flags['fidelity_cards']) {
                    $stmt = $conn->prepare("SELECT logo_path, card_image_path FROM fidelity_cards WHERE user_id = :user_id");
                    $stmt->execute([':user_id' => $user_id]);
                    $cards = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    foreach ($cards as $card) {
                        if ($card['logo_path'] && file_exists(UPLOAD_DIR . $card['logo_path'])) {
                            unlink(UPLOAD_DIR . $card['logo_path']);
                        }
                        if ($card['card_image_path'] && file_exists(UPLOAD_DIR . $card['card_image_path'])) {
                            unlink(UPLOAD_DIR . $card['card_image_path']);
                        }
                    }
                }
                
                // Delete expenses first (they have RESTRICT on account_id, so must be deleted before accounts)
                if (isset($delete_flags['expenses']) && $delete_flags['expenses']) {
                    $stmt = $conn->prepare("DELETE FROM expenses WHERE user_id = :user_id");
                    $stmt->execute([':user_id' => $user_id]);
                }
                
                // Delete fund transfers first (they have RESTRICT on account_id, so must be deleted before accounts)
                if (isset($delete_flags['fund_transfers']) && $delete_flags['fund_transfers']) {
                    $stmt = $conn->prepare("DELETE FROM fund_transfers WHERE user_id = :user_id");
                    $stmt->execute([':user_id' => $user_id]);
                }
                
                // Delete the user (CASCADE will handle accounts, prepaid_cards, fidelity_cards, and all shares)
                // The foreign keys with CASCADE will automatically delete related records
                $stmt = $conn->prepare("DELETE FROM users WHERE id = :id");
                $stmt->execute([':id' => $user_id]);
                
                $conn->commit();
                echo json_encode(['success' => true, 'message' => 'User deleted successfully']);
            } catch (Exception $e) {
                $conn->rollBack();
                throw $e;
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error: ' . $e->getMessage()]);
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
}

