<?php
/**
 * Import User Data API - Extracts ZIP file and imports data and attachments
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

try {
    // Check if ZipArchive is available
    if (!class_exists('ZipArchive')) {
        throw new Exception('ZipArchive class not available. Please install php-zip extension.');
    }
    
    // Check if file was uploaded
    if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
        http_response_code(400);
        echo json_encode(['error' => 'No file uploaded']);
        exit;
    }
    
    $file = $_FILES['import_file'];
    
    // Validate file type
    $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if ($file_ext !== 'zip') {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid file type. Please upload a ZIP file.']);
        exit;
    }
    
    // Create temporary extraction directory
    $extract_dir = sys_get_temp_dir() . '/import_' . $user_id . '_' . time();
    if (!mkdir($extract_dir, 0755, true)) {
        throw new Exception('Cannot create extraction directory');
    }
    
    // Extract ZIP file
    $zip = new ZipArchive();
    if ($zip->open($file['tmp_name']) !== TRUE) {
        rmdir($extract_dir);
        throw new Exception('Cannot open ZIP file');
    }
    
    $zip->extractTo($extract_dir);
    $zip->close();
    
    // Read JSON data file
    $json_file = $extract_dir . '/data.json';
    if (!file_exists($json_file)) {
        // Clean up
        array_map('unlink', glob("$extract_dir/*"));
        rmdir($extract_dir);
        throw new Exception('data.json not found in ZIP file');
    }
    
    $json_content = file_get_contents($json_file);
    $import_data = json_decode($json_content, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        // Clean up
        array_map('unlink', glob("$extract_dir/*"));
        rmdir($extract_dir);
        throw new Exception('Invalid JSON file: ' . json_last_error_msg());
    }
    
    // Validate export data structure
    if (!isset($import_data['version']) || !isset($import_data['accounts']) || !isset($import_data['expenses'])) {
        // Clean up
        array_map('unlink', glob("$extract_dir/*"));
        rmdir($extract_dir);
        throw new Exception('Invalid export file format');
    }
    
    // Start transaction
    $conn->beginTransaction();
    
    try {
        // Get account types mapping (code => id)
        $stmt = $conn->prepare("SELECT id, code FROM account_types");
        $stmt->execute();
        $account_types_by_code = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $account_types_by_code[$row['code']] = $row['id'];
        }
        
        // Import accounts and create ID mapping
        $account_id_mapping = []; // old_id => new_id
        $accounts = $import_data['accounts'] ?? [];
        $skipped_accounts = 0;
        
        foreach ($accounts as $old_account) {
            // Check if account name already exists for this user
            $stmt = $conn->prepare("SELECT id FROM accounts WHERE user_id = :user_id AND name = :name");
            $stmt->execute([':user_id' => $user_id, ':name' => $old_account['name']]);
            if ($stmt->fetch()) {
                // Skip this account - name already exists
                $skipped_accounts++;
                continue;
            }
            // Map type_id using account_types_map
            $type_code = null;
            if (isset($import_data['account_types_map'])) {
                foreach ($import_data['account_types_map'] as $old_type_id => $code) {
                    if ($old_type_id == $old_account['type_id']) {
                        $type_code = $code;
                        break;
                    }
                }
            }
            
            if (!$type_code || !isset($account_types_by_code[$type_code])) {
                // Fallback: use first available type
                $type_id = reset($account_types_by_code);
            } else {
                $type_id = $account_types_by_code[$type_code];
            }
            
            $old_account_id = $old_account['id'];
            
            // Handle logo file
            $logo_path = null;
            if (!empty($old_account['logo_path_export'])) {
                $source_file = $extract_dir . '/' . $old_account['logo_path_export'];
                if (file_exists($source_file)) {
                    $upload_dir = UPLOAD_DIR . 'accounts/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $logo_filename = uniqid() . '_' . basename($old_account['logo_path_export']);
                    $logo_file_path = $upload_dir . $logo_filename;
                    
                    if (copy($source_file, $logo_file_path)) {
                        $logo_path = 'accounts/' . $logo_filename;
                    }
                }
            }
            
            // Insert account
            $stmt = $conn->prepare("
                INSERT INTO accounts (user_id, name, type_id, base_currency, opening_date, id_number, expire_date, closure_date, tax_type, logo_path)
                VALUES (:user_id, :name, :type_id, :base_currency, :opening_date, :id_number, :expire_date, :closure_date, :tax_type, :logo_path)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':name' => $old_account['name'],
                ':type_id' => $type_id,
                ':base_currency' => $old_account['base_currency'],
                ':opening_date' => $old_account['opening_date'],
                ':id_number' => $old_account['id_number'] ?? null,
                ':expire_date' => $old_account['expire_date'] ?? null,
                ':closure_date' => $old_account['closure_date'] ?? null,
                ':tax_type' => $old_account['tax_type'] ?? null,
                ':logo_path' => $logo_path
            ]);
            
            $new_account_id = $conn->lastInsertId();
            $account_id_mapping[$old_account_id] = $new_account_id;
        }
        
        // Import prepaid cards and create ID mapping
        $prepaid_card_id_mapping = []; // old_id => new_id
        $prepaid_cards = $import_data['prepaid_cards'] ?? [];
        
        foreach ($prepaid_cards as $old_card) {
            // Check if prepaid card name already exists for this user
            $stmt = $conn->prepare("SELECT id FROM prepaid_cards WHERE user_id = :user_id AND name = :name");
            $stmt->execute([':user_id' => $user_id, ':name' => $old_card['name']]);
            if ($stmt->fetch()) {
                // Skip this card - name already exists
                continue;
            }
            
            $old_card_id = $old_card['id'];
            
            // Handle card photo file
            $photo_path = null;
            if (!empty($old_card['photo_path_export'])) {
                $source_file = $extract_dir . '/' . $old_card['photo_path_export'];
                if (file_exists($source_file)) {
                    $upload_dir = UPLOAD_DIR . 'prepaid_cards/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $photo_filename = uniqid() . '_' . basename($old_card['photo_path_export']);
                    $photo_file_path = $upload_dir . $photo_filename;
                    
                    if (copy($source_file, $photo_file_path)) {
                        $photo_path = 'prepaid_cards/' . $photo_filename;
                    }
                }
            }
            
            // Insert prepaid card
            $stmt = $conn->prepare("
                INSERT INTO prepaid_cards (user_id, name, card_number, base_currency, nominal_amount, opening_date, expire_date, enabled, photo_path)
                VALUES (:user_id, :name, :card_number, :base_currency, :nominal_amount, :opening_date, :expire_date, :enabled, :photo_path)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':name' => $old_card['name'],
                ':card_number' => $old_card['card_number'] ?? null,
                ':base_currency' => $old_card['base_currency'],
                ':nominal_amount' => $old_card['nominal_amount'] ?? null,
                ':opening_date' => $old_card['opening_date'],
                ':expire_date' => $old_card['expire_date'] ?? null,
                ':enabled' => isset($old_card['enabled']) ? ($old_card['enabled'] ? 1 : 0) : 1,
                ':photo_path' => $photo_path
            ]);
            
            $new_card_id = $conn->lastInsertId();
            $prepaid_card_id_mapping[$old_card_id] = $new_card_id;
        }
        
        // Import expenses
        $expenses = $import_data['expenses'] ?? [];
        $expense_id_mapping = []; // old_id => new_id
        
        foreach ($expenses as $old_expense) {
            $old_account_id = $old_expense['account_id'] ?? null;
            $old_prepaid_card_id = $old_expense['prepaid_card_id'] ?? null;
            
            $new_account_id = null;
            $new_prepaid_card_id = null;
            
            if ($old_account_id) {
                $new_account_id = $account_id_mapping[$old_account_id] ?? null;
            }
            if ($old_prepaid_card_id) {
                $new_prepaid_card_id = $prepaid_card_id_mapping[$old_prepaid_card_id] ?? null;
            }
            
            // Skip if neither account nor prepaid card mapping found
            if (!$new_account_id && !$new_prepaid_card_id) {
                continue;
            }
            
            $stmt = $conn->prepare("
                INSERT INTO expenses (user_id, account_id, prepaid_card_id, date, description, amount, currency, is_tax_deductible, tax_category_id)
                VALUES (:user_id, :account_id, :prepaid_card_id, :date, :description, :amount, :currency, :is_tax_deductible, :tax_category_id)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':account_id' => $new_account_id,
                ':prepaid_card_id' => $new_prepaid_card_id,
                ':date' => $old_expense['date'],
                ':description' => $old_expense['description'],
                ':amount' => $old_expense['amount'],
                ':currency' => $old_expense['currency'],
                ':is_tax_deductible' => $old_expense['is_tax_deductible'] ?? false,
                ':tax_category_id' => $old_expense['tax_category_id'] ?? null
            ]);
            
            $old_expense_id = $old_expense['id'];
            $new_expense_id = $conn->lastInsertId();
            $expense_id_mapping[$old_expense_id] = $new_expense_id;
            
            // Import attachments for this expense
            $attachments = $old_expense['attachments'] ?? [];
            foreach ($attachments as $old_attachment) {
                $stmt = $conn->prepare("
                    INSERT INTO expense_attachments (expense_id, type, file_path, link_url, link_text)
                    VALUES (:expense_id, :type, :file_path, :link_url, :link_text)
                ");
                
                $file_path = null;
                if ($old_attachment['type'] !== 'LINK' && !empty($old_attachment['file_path_export'])) {
                    // Copy attachment file
                    $source_file = $extract_dir . '/' . $old_attachment['file_path_export'];
                    if (file_exists($source_file)) {
                        $upload_dir = UPLOAD_DIR . 'expenses/' . $new_expense_id . '/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_filename = uniqid() . '_' . basename($old_attachment['file_path_export']);
                        $file_file_path = $upload_dir . $file_filename;
                        
                        if (copy($source_file, $file_file_path)) {
                            $file_path = 'expenses/' . $new_expense_id . '/' . $file_filename;
                        }
                    }
                }
                
                $stmt->execute([
                    ':expense_id' => $new_expense_id,
                    ':type' => $old_attachment['type'],
                    ':file_path' => $file_path,
                    ':link_url' => $old_attachment['link_url'] ?? null,
                    ':link_text' => $old_attachment['link_text'] ?? null
                ]);
            }
        }
        
        // Import fund transfers
        $transfers = $import_data['transfers'] ?? [];
        $transfer_id_mapping = []; // old_id => new_id
        
        foreach ($transfers as $old_transfer) {
            $old_from_account_id = $old_transfer['from_account_id'];
            $old_to_account_id = $old_transfer['to_account_id'];
            
            $new_from_account_id = $account_id_mapping[$old_from_account_id] ?? null;
            $new_to_account_id = $account_id_mapping[$old_to_account_id] ?? null;
            
            if (!$new_from_account_id || !$new_to_account_id) {
                continue; // Skip if account mapping not found
            }
            
            $old_transfer_id = $old_transfer['id'];
            
            $stmt = $conn->prepare("
                INSERT INTO fund_transfers (user_id, from_account_id, to_account_id, amount, from_currency, to_currency, exchange_rate, converted_amount, transfer_date, arrival_date, notes)
                VALUES (:user_id, :from_account_id, :to_account_id, :amount, :from_currency, :to_currency, :exchange_rate, :converted_amount, :transfer_date, :arrival_date, :notes)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':from_account_id' => $new_from_account_id,
                ':to_account_id' => $new_to_account_id,
                ':amount' => $old_transfer['amount'],
                ':from_currency' => $old_transfer['from_currency'],
                ':to_currency' => $old_transfer['to_currency'],
                ':exchange_rate' => $old_transfer['exchange_rate'],
                ':converted_amount' => $old_transfer['converted_amount'],
                ':transfer_date' => $old_transfer['transfer_date'],
                ':arrival_date' => $old_transfer['arrival_date'] ?? null,
                ':notes' => $old_transfer['notes'] ?? null
            ]);
            
            $new_transfer_id = $conn->lastInsertId();
            $transfer_id_mapping[$old_transfer_id] = $new_transfer_id;
            
            // Import attachments for this transfer
            $attachments = $old_transfer['attachments'] ?? [];
            foreach ($attachments as $old_attachment) {
                $stmt = $conn->prepare("
                    INSERT INTO transfer_attachments (transfer_id, type, file_path, link_url, link_text)
                    VALUES (:transfer_id, :type, :file_path, :link_url, :link_text)
                ");
                
                $file_path = null;
                if ($old_attachment['type'] !== 'LINK' && !empty($old_attachment['file_path_export'])) {
                    // Copy attachment file
                    $source_file = $extract_dir . '/' . $old_attachment['file_path_export'];
                    if (file_exists($source_file)) {
                        $upload_dir = UPLOAD_DIR . 'transfers/' . $new_transfer_id . '/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_filename = uniqid() . '_' . basename($old_attachment['file_path_export']);
                        $file_file_path = $upload_dir . $file_filename;
                        
                        if (copy($source_file, $file_file_path)) {
                            $file_path = 'transfers/' . $new_transfer_id . '/' . $file_filename;
                        }
                    }
                }
                
                $stmt->execute([
                    ':transfer_id' => $new_transfer_id,
                    ':type' => $old_attachment['type'],
                    ':file_path' => $file_path,
                    ':link_url' => $old_attachment['link_url'] ?? null,
                    ':link_text' => $old_attachment['link_text'] ?? null
                ]);
            }
        }
        
        // Import securities
        $securities = $import_data['securities'] ?? [];
        $security_id_mapping = []; // old_id => new_id
        
        foreach ($securities as $old_security) {
            $old_account_id = $old_security['account_id'];
            $new_account_id = $account_id_mapping[$old_account_id] ?? null;
            
            if (!$new_account_id) {
                continue; // Skip if account mapping not found
            }
            
            $stmt = $conn->prepare("
                INSERT INTO securities (account_id, symbol, quantity, purchase_price, purchase_date, currency, notes, transaction_type)
                VALUES (:account_id, :symbol, :quantity, :purchase_price, :purchase_date, :currency, :notes, :transaction_type)
            ");
            
            $stmt->execute([
                ':account_id' => $new_account_id,
                ':symbol' => $old_security['symbol'],
                ':quantity' => $old_security['quantity'],
                ':purchase_price' => $old_security['purchase_price'],
                ':purchase_date' => $old_security['purchase_date'],
                ':currency' => $old_security['currency'],
                ':notes' => $old_security['notes'] ?? null,
                ':transaction_type' => $old_security['transaction_type'] ?? null
            ]);
            
            $old_security_id = $old_security['id'];
            $new_security_id = $conn->lastInsertId();
            $security_id_mapping[$old_security_id] = $new_security_id;
            
            // Import attachments for this security
            $attachments = $old_security['attachments'] ?? [];
            foreach ($attachments as $old_attachment) {
                $stmt = $conn->prepare("
                    INSERT INTO securities_attachments (security_id, type, file_path, link_url, link_text)
                    VALUES (:security_id, :type, :file_path, :link_url, :link_text)
                ");
                
                $file_path = null;
                if ($old_attachment['type'] !== 'LINK' && !empty($old_attachment['file_path_export'])) {
                    // Copy attachment file
                    $source_file = $extract_dir . '/' . $old_attachment['file_path_export'];
                    if (file_exists($source_file)) {
                        $upload_dir = UPLOAD_DIR . 'securities/' . $new_security_id . '/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_filename = uniqid() . '_' . basename($old_attachment['file_path_export']);
                        $file_file_path = $upload_dir . $file_filename;
                        
                        if (copy($source_file, $file_file_path)) {
                            $file_path = 'securities/' . $new_security_id . '/' . $file_filename;
                        }
                    }
                }
                
                $stmt->execute([
                    ':security_id' => $new_security_id,
                    ':type' => $old_attachment['type'],
                    ':file_path' => $file_path,
                    ':link_url' => $old_attachment['link_url'] ?? null,
                    ':link_text' => $old_attachment['link_text'] ?? null
                ]);
            }
        }
        
        // Import fidelity cards
        $fidelity_cards = $import_data['fidelity_cards'] ?? [];
        $fidelity_card_id_mapping = []; // old_id => new_id
        
        foreach ($fidelity_cards as $old_card) {
            // Check if fidelity card name already exists for this user
            $stmt = $conn->prepare("SELECT id FROM fidelity_cards WHERE user_id = :user_id AND name = :name");
            $stmt->execute([':user_id' => $user_id, ':name' => $old_card['name']]);
            if ($stmt->fetch()) {
                // Skip this card - name already exists
                continue;
            }
            
            $old_card_id = $old_card['id'];
            
            // Handle logo file
            $logo_path = null;
            if (!empty($old_card['logo_path_export'])) {
                $source_file = $extract_dir . '/' . $old_card['logo_path_export'];
                if (file_exists($source_file)) {
                    $upload_dir = UPLOAD_DIR . 'fidelity_cards/logos/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $logo_filename = uniqid() . '_' . basename($old_card['logo_path_export']);
                    $logo_file_path = $upload_dir . $logo_filename;
                    
                    if (copy($source_file, $logo_file_path)) {
                        $logo_path = 'fidelity_cards/logos/' . $logo_filename;
                    }
                }
            }
            
            // Handle card image file
            $card_image_path = null;
            if (!empty($old_card['card_image_path_export'])) {
                $source_file = $extract_dir . '/' . $old_card['card_image_path_export'];
                if (file_exists($source_file)) {
                    $upload_dir = UPLOAD_DIR . 'fidelity_cards/images/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $image_filename = uniqid() . '_' . basename($old_card['card_image_path_export']);
                    $image_file_path = $upload_dir . $image_filename;
                    
                    if (copy($source_file, $image_file_path)) {
                        $card_image_path = 'fidelity_cards/images/' . $image_filename;
                    }
                }
            }
            
            // Insert fidelity card
            $stmt = $conn->prepare("
                INSERT INTO fidelity_cards (user_id, name, store_name, card_code, description, logo_path, card_image_path)
                VALUES (:user_id, :name, :store_name, :card_code, :description, :logo_path, :card_image_path)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':name' => $old_card['name'],
                ':store_name' => $old_card['store_name'],
                ':card_code' => $old_card['card_code'],
                ':description' => $old_card['description'] ?? null,
                ':logo_path' => $logo_path,
                ':card_image_path' => $card_image_path
            ]);
            
            $new_card_id = $conn->lastInsertId();
            $fidelity_card_id_mapping[$old_card_id] = $new_card_id;
        }
        
        // Import account orders
        $account_orders = $import_data['account_orders'] ?? [];
        foreach ($account_orders as $order) {
            $old_account_id = $order['account_id'];
            $new_account_id = $account_id_mapping[$old_account_id] ?? null;
            
            if (!$new_account_id) {
                continue; // Skip if account mapping not found
            }
            
            // Check if order already exists
            $stmt = $conn->prepare("SELECT id FROM user_account_order WHERE user_id = :user_id AND account_id = :account_id");
            $stmt->execute([':user_id' => $user_id, ':account_id' => $new_account_id]);
            if ($stmt->fetch()) {
                // Update existing order
                $stmt = $conn->prepare("UPDATE user_account_order SET display_order = :display_order WHERE user_id = :user_id AND account_id = :account_id");
                $stmt->execute([
                    ':user_id' => $user_id,
                    ':account_id' => $new_account_id,
                    ':display_order' => $order['display_order']
                ]);
            } else {
                // Insert new order
                $stmt = $conn->prepare("
                    INSERT INTO user_account_order (user_id, account_id, display_order)
                    VALUES (:user_id, :account_id, :display_order)
                ");
                $stmt->execute([
                    ':user_id' => $user_id,
                    ':account_id' => $new_account_id,
                    ':display_order' => $order['display_order']
                ]);
            }
        }
        
        // Import fidelity card orders
        $fidelity_card_orders = $import_data['fidelity_card_orders'] ?? [];
        foreach ($fidelity_card_orders as $order) {
            $old_card_id = $order['fidelity_card_id'];
            $new_card_id = $fidelity_card_id_mapping[$old_card_id] ?? null;
            
            if (!$new_card_id) {
                continue; // Skip if card mapping not found
            }
            
            // Check if order already exists
            $stmt = $conn->prepare("SELECT id FROM user_fidelity_card_order WHERE user_id = :user_id AND fidelity_card_id = :fidelity_card_id");
            $stmt->execute([':user_id' => $user_id, ':fidelity_card_id' => $new_card_id]);
            if ($stmt->fetch()) {
                // Update existing order
                $stmt = $conn->prepare("UPDATE user_fidelity_card_order SET display_order = :display_order WHERE user_id = :user_id AND fidelity_card_id = :fidelity_card_id");
                $stmt->execute([
                    ':user_id' => $user_id,
                    ':fidelity_card_id' => $new_card_id,
                    ':display_order' => $order['display_order']
                ]);
            } else {
                // Insert new order
                $stmt = $conn->prepare("
                    INSERT INTO user_fidelity_card_order (user_id, fidelity_card_id, display_order)
                    VALUES (:user_id, :fidelity_card_id, :display_order)
                ");
                $stmt->execute([
                    ':user_id' => $user_id,
                    ':fidelity_card_id' => $new_card_id,
                    ':display_order' => $order['display_order']
                ]);
            }
        }
        
        // Import account shares
        $account_shares = $import_data['account_shares'] ?? [];
        foreach ($account_shares as $share) {
            $old_account_id = $share['account_id'];
            $new_account_id = $account_id_mapping[$old_account_id] ?? null;
            
            if (!$new_account_id) {
                continue; // Skip if account mapping not found
            }
            
            // Find user by email
            $shared_with_email = $share['shared_with_email'] ?? null;
            if (!$shared_with_email) {
                continue; // Skip if no email
            }
            
            $stmt = $conn->prepare("SELECT id FROM users WHERE email = :email");
            $stmt->execute([':email' => $shared_with_email]);
            $shared_user = $stmt->fetch();
            
            if ($shared_user) {
                $shared_with_user_id = $shared_user['id'];
                
                // Check if share already exists
                $stmt = $conn->prepare("SELECT id FROM account_shares WHERE account_id = :account_id AND shared_with_user_id = :shared_with_user_id");
                $stmt->execute([':account_id' => $new_account_id, ':shared_with_user_id' => $shared_with_user_id]);
                if (!$stmt->fetch()) {
                    // Insert new share
                    $stmt = $conn->prepare("
                        INSERT INTO account_shares (account_id, owner_user_id, shared_with_user_id, sharing_mode)
                        VALUES (:account_id, :owner_user_id, :shared_with_user_id, :sharing_mode)
                    ");
                    $stmt->execute([
                        ':account_id' => $new_account_id,
                        ':owner_user_id' => $user_id,
                        ':shared_with_user_id' => $shared_with_user_id,
                        ':sharing_mode' => $share['sharing_mode'] ?? 'read-only'
                    ]);
                }
            }
        }
        
        // Import account share invitations
        $account_share_invitations = $import_data['account_share_invitations'] ?? [];
        foreach ($account_share_invitations as $invitation) {
            $old_account_id = $invitation['account_id'];
            $new_account_id = $account_id_mapping[$old_account_id] ?? null;
            
            if (!$new_account_id) {
                continue; // Skip if account mapping not found
            }
            
            $recipient_email = $invitation['recipient_email'] ?? null;
            if (!$recipient_email) {
                continue;
            }
            
            // Check if invitation already exists
            $stmt = $conn->prepare("SELECT id FROM account_share_invitations WHERE account_id = :account_id AND recipient_email = :recipient_email AND owner_user_id = :owner_user_id");
            $stmt->execute([
                ':account_id' => $new_account_id,
                ':recipient_email' => $recipient_email,
                ':owner_user_id' => $user_id
            ]);
            if (!$stmt->fetch()) {
                // Insert new invitation
                $stmt = $conn->prepare("
                    INSERT INTO account_share_invitations (account_id, owner_user_id, recipient_email, sharing_mode)
                    VALUES (:account_id, :owner_user_id, :recipient_email, :sharing_mode)
                ");
                $stmt->execute([
                    ':account_id' => $new_account_id,
                    ':owner_user_id' => $user_id,
                    ':recipient_email' => $recipient_email,
                    ':sharing_mode' => $invitation['sharing_mode'] ?? 'read-only'
                ]);
            }
        }
        
        // Import prepaid card shares
        $prepaid_card_shares = $import_data['prepaid_card_shares'] ?? [];
        foreach ($prepaid_card_shares as $share) {
            $old_card_id = $share['prepaid_card_id'];
            $new_card_id = $prepaid_card_id_mapping[$old_card_id] ?? null;
            
            if (!$new_card_id) {
                continue; // Skip if card mapping not found
            }
            
            // Find user by email
            $shared_with_email = $share['shared_with_email'] ?? null;
            if (!$shared_with_email) {
                continue; // Skip if no email
            }
            
            $stmt = $conn->prepare("SELECT id FROM users WHERE email = :email");
            $stmt->execute([':email' => $shared_with_email]);
            $shared_user = $stmt->fetch();
            
            if ($shared_user) {
                $shared_with_user_id = $shared_user['id'];
                
                // Check if share already exists
                $stmt = $conn->prepare("SELECT id FROM prepaid_card_shares WHERE prepaid_card_id = :prepaid_card_id AND shared_with_user_id = :shared_with_user_id");
                $stmt->execute([':prepaid_card_id' => $new_card_id, ':shared_with_user_id' => $shared_with_user_id]);
                if (!$stmt->fetch()) {
                    // Insert new share
                    $stmt = $conn->prepare("
                        INSERT INTO prepaid_card_shares (prepaid_card_id, owner_user_id, shared_with_user_id)
                        VALUES (:prepaid_card_id, :owner_user_id, :shared_with_user_id)
                    ");
                    $stmt->execute([
                        ':prepaid_card_id' => $new_card_id,
                        ':owner_user_id' => $user_id,
                        ':shared_with_user_id' => $shared_with_user_id
                    ]);
                }
            }
        }
        
        // Import prepaid card share invitations
        $prepaid_card_share_invitations = $import_data['prepaid_card_share_invitations'] ?? [];
        foreach ($prepaid_card_share_invitations as $invitation) {
            $old_card_id = $invitation['prepaid_card_id'];
            $new_card_id = $prepaid_card_id_mapping[$old_card_id] ?? null;
            
            if (!$new_card_id) {
                continue; // Skip if card mapping not found
            }
            
            $recipient_email = $invitation['recipient_email'] ?? null;
            if (!$recipient_email) {
                continue;
            }
            
            // Check if invitation already exists
            $stmt = $conn->prepare("SELECT id FROM prepaid_card_share_invitations WHERE prepaid_card_id = :prepaid_card_id AND recipient_email = :recipient_email AND owner_user_id = :owner_user_id");
            $stmt->execute([
                ':prepaid_card_id' => $new_card_id,
                ':recipient_email' => $recipient_email,
                ':owner_user_id' => $user_id
            ]);
            if (!$stmt->fetch()) {
                // Insert new invitation
                $stmt = $conn->prepare("
                    INSERT INTO prepaid_card_share_invitations (prepaid_card_id, owner_user_id, recipient_email)
                    VALUES (:prepaid_card_id, :owner_user_id, :recipient_email)
                ");
                $stmt->execute([
                    ':prepaid_card_id' => $new_card_id,
                    ':owner_user_id' => $user_id,
                    ':recipient_email' => $recipient_email
                ]);
            }
        }
        
        // Import fidelity card shares
        $fidelity_card_shares = $import_data['fidelity_card_shares'] ?? [];
        foreach ($fidelity_card_shares as $share) {
            $old_card_id = $share['fidelity_card_id'];
            $new_card_id = $fidelity_card_id_mapping[$old_card_id] ?? null;
            
            if (!$new_card_id) {
                continue; // Skip if card mapping not found
            }
            
            // Find user by email
            $shared_with_email = $share['shared_with_email'] ?? null;
            if (!$shared_with_email) {
                continue; // Skip if no email
            }
            
            $stmt = $conn->prepare("SELECT id FROM users WHERE email = :email");
            $stmt->execute([':email' => $shared_with_email]);
            $shared_user = $stmt->fetch();
            
            if ($shared_user) {
                $shared_with_user_id = $shared_user['id'];
                
                // Check if share already exists
                $stmt = $conn->prepare("SELECT id FROM fidelity_card_shares WHERE fidelity_card_id = :fidelity_card_id AND shared_with_user_id = :shared_with_user_id");
                $stmt->execute([':fidelity_card_id' => $new_card_id, ':shared_with_user_id' => $shared_with_user_id]);
                if (!$stmt->fetch()) {
                    // Insert new share
                    $stmt = $conn->prepare("
                        INSERT INTO fidelity_card_shares (fidelity_card_id, owner_user_id, shared_with_user_id)
                        VALUES (:fidelity_card_id, :owner_user_id, :shared_with_user_id)
                    ");
                    $stmt->execute([
                        ':fidelity_card_id' => $new_card_id,
                        ':owner_user_id' => $user_id,
                        ':shared_with_user_id' => $shared_with_user_id
                    ]);
                }
            }
        }
        
        // Import fidelity card share invitations
        $fidelity_card_share_invitations = $import_data['fidelity_card_share_invitations'] ?? [];
        foreach ($fidelity_card_share_invitations as $invitation) {
            $old_card_id = $invitation['fidelity_card_id'];
            $new_card_id = $fidelity_card_id_mapping[$old_card_id] ?? null;
            
            if (!$new_card_id) {
                continue; // Skip if card mapping not found
            }
            
            $recipient_email = $invitation['recipient_email'] ?? null;
            if (!$recipient_email) {
                continue;
            }
            
            // Check if invitation already exists
            $stmt = $conn->prepare("SELECT id FROM fidelity_card_share_invitations WHERE fidelity_card_id = :fidelity_card_id AND recipient_email = :recipient_email AND owner_user_id = :owner_user_id");
            $stmt->execute([
                ':fidelity_card_id' => $new_card_id,
                ':recipient_email' => $recipient_email,
                ':owner_user_id' => $user_id
            ]);
            if (!$stmt->fetch()) {
                // Insert new invitation
                $stmt = $conn->prepare("
                    INSERT INTO fidelity_card_share_invitations (fidelity_card_id, owner_user_id, recipient_email)
                    VALUES (:fidelity_card_id, :owner_user_id, :recipient_email)
                ");
                $stmt->execute([
                    ':fidelity_card_id' => $new_card_id,
                    ':owner_user_id' => $user_id,
                    ':recipient_email' => $recipient_email
                ]);
            }
        }
        
        // Update user profile
        if (!empty($import_data['profile'])) {
            $profile = $import_data['profile'];
            $default_account_id = null;
            
            // Map default_account_id if set
            if (!empty($profile['default_account_id'])) {
                $old_default_account_id = $profile['default_account_id'];
                $default_account_id = $account_id_mapping[$old_default_account_id] ?? null;
            }
            
            $stmt = $conn->prepare("
                UPDATE users 
                SET default_currency = :default_currency, default_account_id = :default_account_id
                WHERE id = :user_id
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':default_currency' => $profile['default_currency'] ?? 'EUR',
                ':default_account_id' => $default_account_id
            ]);
        }
        
        // Commit transaction
        $conn->commit();
        
        // Clean up extraction directory
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($extract_dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ($files as $fileinfo) {
            $todo = ($fileinfo->isDir() ? 'rmdir' : 'unlink');
            $todo($fileinfo->getRealPath());
        }
        rmdir($extract_dir);
        
        echo json_encode([
            'success' => true,
            'message' => 'Data imported successfully',
            'imported' => [
                'accounts' => count($accounts) - $skipped_accounts,
                'expenses' => count($expenses),
                'transfers' => count($transfers),
                'securities' => count($securities),
                'prepaid_cards' => count($prepaid_cards),
                'fidelity_cards' => count($fidelity_cards)
            ],
            'skipped' => [
                'accounts' => $skipped_accounts
            ]
        ]);
        
    } catch (Exception $e) {
        $conn->rollBack();
        
        // Clean up extraction directory
        if (isset($extract_dir) && is_dir($extract_dir)) {
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($extract_dir, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::CHILD_FIRST
            );
            foreach ($files as $fileinfo) {
                $todo = ($fileinfo->isDir() ? 'rmdir' : 'unlink');
                $todo($fileinfo->getRealPath());
            }
            rmdir($extract_dir);
        }
        
        throw $e;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Import failed: ' . $e->getMessage()]);
    error_log('Import error: ' . $e->getMessage());
}
