<?php
/**
 * Securities Attachments API
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'POST':
        // Add attachment to security
        $security_id = $_POST['security_id'] ?? null;
        $type = $_POST['type'] ?? null;
        
        if (!$security_id || !$type || !in_array($type, ['PHOTO', 'DOCUMENT', 'LINK'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid parameters']);
            exit;
        }
        
        // Verify security belongs to user
        $stmt = $conn->prepare("
            SELECT s.id FROM securities s
            JOIN accounts a ON s.account_id = a.id
            WHERE s.id = :security_id AND a.user_id = :user_id
        ");
        $stmt->execute([':security_id' => $security_id, ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Security not found']);
            exit;
        }
        
        $file_path = null;
        $link_url = null;
        $link_text = null;
        
        if ($type === 'LINK') {
            $link_url = $_POST['link_url'] ?? null;
            $link_text = $_POST['link_text'] ?? null;
            
            if (!$link_url) {
                http_response_code(400);
                echo json_encode(['error' => 'Link URL required']);
                exit;
            }
        } else {
            // Handle file upload
            if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
                http_response_code(400);
                echo json_encode(['error' => 'File upload failed']);
                exit;
            }
            
            $file = $_FILES['file'];
            
            // Validate file size
            if ($file['size'] > MAX_UPLOAD_SIZE) {
                http_response_code(400);
                echo json_encode(['error' => 'File too large']);
                exit;
            }
            
            // Create upload directory if it doesn't exist
            $upload_dir = UPLOAD_DIR . 'securities/' . $security_id . '/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = uniqid() . '.' . $extension;
            $file_path = $upload_dir . $filename;
            
            if (!move_uploaded_file($file['tmp_name'], $file_path)) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to save file']);
                exit;
            }
            
            // Store relative path
            $file_path = 'securities/' . $security_id . '/' . $filename;
        }
        
        $stmt = $conn->prepare("
            INSERT INTO securities_attachments (security_id, type, file_path, link_url, link_text)
            VALUES (:security_id, :type, :file_path, :link_url, :link_text)
        ");
        
        $stmt->execute([
            ':security_id' => $security_id,
            ':type' => $type,
            ':file_path' => $file_path,
            ':link_url' => $link_url,
            ':link_text' => $link_text
        ]);
        
        $attachment_id = $conn->lastInsertId();
        
        echo json_encode(['success' => true, 'id' => $attachment_id]);
        break;
        
    case 'DELETE':
        // Delete attachment
        $attachment_id = $_GET['id'] ?? null;
        
        if (!$attachment_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Attachment ID required']);
            exit;
        }
        
        // Verify attachment belongs to user's security
        $stmt = $conn->prepare("
            SELECT sa.id, sa.file_path
            FROM securities_attachments sa
            JOIN securities s ON sa.security_id = s.id
            JOIN accounts a ON s.account_id = a.id
            WHERE sa.id = :attachment_id AND a.user_id = :user_id
        ");
        $stmt->execute([':attachment_id' => $attachment_id, ':user_id' => $user_id]);
        $attachment = $stmt->fetch();
        
        if (!$attachment) {
            http_response_code(404);
            echo json_encode(['error' => 'Attachment not found']);
            exit;
        }
        
        // Delete file if exists
        if ($attachment['file_path']) {
            $full_path = UPLOAD_DIR . $attachment['file_path'];
            if (file_exists($full_path)) {
                unlink($full_path);
            }
        }
        
        $stmt = $conn->prepare("DELETE FROM securities_attachments WHERE id = :id");
        $stmt->execute([':id' => $attachment_id]);
        
        echo json_encode(['success' => true]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}

