<?php
/**
 * Prepaid Card Sharing API
 * Prepaid cards can only be shared in write mode (always write-only)
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';
require_once __DIR__ . '/../middleware/notifications_helper.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];
    $prepaid_card_id = $_GET['prepaid_card_id'] ?? null;

    switch ($method) {
    case 'GET':
        if ($prepaid_card_id) {
            // Get shares for a specific prepaid card (owner view)
            $prepaid_card_id = (int)$prepaid_card_id;
            
            // Verify user owns the prepaid card
            $stmt = $conn->prepare("SELECT id FROM prepaid_cards WHERE id = :prepaid_card_id AND user_id = :user_id");
            $stmt->execute([':prepaid_card_id' => $prepaid_card_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Prepaid card not found']);
                break;
            }
            
            // Get active shares
            $stmt = $conn->prepare("
                SELECT 
                    pcs.id,
                    pcs.shared_with_user_id,
                    u.email,
                    u.name,
                    pcs.created_at
                FROM prepaid_card_shares pcs
                JOIN users u ON pcs.shared_with_user_id = u.id
                WHERE pcs.prepaid_card_id = :prepaid_card_id AND pcs.owner_user_id = :user_id
                ORDER BY pcs.created_at DESC
            ");
            $stmt->execute([':prepaid_card_id' => $prepaid_card_id, ':user_id' => $user_id]);
            $shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get pending invitations
            $stmt = $conn->prepare("
                SELECT 
                    id,
                    recipient_email,
                    created_at
                FROM prepaid_card_share_invitations
                WHERE prepaid_card_id = :prepaid_card_id AND owner_user_id = :user_id
                ORDER BY created_at DESC
            ");
            $stmt->execute([':prepaid_card_id' => $prepaid_card_id, ':user_id' => $user_id]);
            $invitations = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'shares' => $shares,
                'invitations' => $invitations
            ]);
        } else {
            // Get previously shared users for dropdown
            // Includes users the current user has shared with AND users who have shared with the current user
            // This includes users from both account shares and prepaid card shares
            $stmt = $conn->prepare("
                SELECT DISTINCT
                    u.id,
                    u.email,
                    u.name
                FROM users u
                WHERE u.id IN (
                    -- Users the current user has shared accounts with
                    SELECT DISTINCT shared_with_user_id 
                    FROM account_shares 
                    WHERE owner_user_id = :user_id1
                    UNION
                    -- Users who have shared accounts with the current user
                    SELECT DISTINCT owner_user_id 
                    FROM account_shares 
                    WHERE shared_with_user_id = :user_id2
                    UNION
                    -- Users the current user has shared prepaid cards with
                    SELECT DISTINCT shared_with_user_id 
                    FROM prepaid_card_shares 
                    WHERE owner_user_id = :user_id3
                    UNION
                    -- Users who have shared prepaid cards with the current user
                    SELECT DISTINCT owner_user_id 
                    FROM prepaid_card_shares 
                    WHERE shared_with_user_id = :user_id4
                )
                ORDER BY u.email
            ");
            $stmt->execute([
                ':user_id1' => $user_id, 
                ':user_id2' => $user_id,
                ':user_id3' => $user_id,
                ':user_id4' => $user_id
            ]);
            $previously_shared_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode($previously_shared_users);
        }
        break;
        
    case 'POST':
        // Share prepaid card with user
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['prepaid_card_id']) || !isset($data['recipient_email'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields: prepaid_card_id, recipient_email']);
            break;
        }
        
        $prepaid_card_id = (int)$data['prepaid_card_id'];
        $recipient_email = trim($data['recipient_email']);
        
        // Verify user owns the prepaid card
        $stmt = $conn->prepare("SELECT id, name FROM prepaid_cards WHERE id = :prepaid_card_id AND user_id = :user_id");
        $stmt->execute([':prepaid_card_id' => $prepaid_card_id, ':user_id' => $user_id]);
        $prepaid_card = $stmt->fetch();
        if (!$prepaid_card) {
            http_response_code(404);
            echo json_encode(['error' => 'Prepaid card not found']);
            break;
        }
        
        // Prevent sharing with yourself
        $stmt = $conn->prepare("SELECT id, email FROM users WHERE id = :user_id");
        $stmt->execute([':user_id' => $user_id]);
        $owner = $stmt->fetch();
        if (strtolower($owner['email']) === strtolower($recipient_email)) {
            http_response_code(400);
            echo json_encode(['error' => 'Cannot share prepaid card with yourself']);
            break;
        }
        
        // Check if recipient user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = :email");
        $stmt->execute([':email' => $recipient_email]);
        $recipient_user = $stmt->fetch();
        
        if ($recipient_user) {
            // User exists - create active share
            $recipient_user_id = $recipient_user['id'];
            
            // Check if share already exists
            $stmt = $conn->prepare("SELECT id FROM prepaid_card_shares WHERE prepaid_card_id = :prepaid_card_id AND shared_with_user_id = :shared_with_user_id");
            $stmt->execute([':prepaid_card_id' => $prepaid_card_id, ':shared_with_user_id' => $recipient_user_id]);
            if ($stmt->fetch()) {
                http_response_code(400);
                echo json_encode(['error' => 'Prepaid card is already shared with this user']);
                break;
            }
            
            // Create share (always write mode)
            $stmt = $conn->prepare("
                INSERT INTO prepaid_card_shares (prepaid_card_id, owner_user_id, shared_with_user_id)
                VALUES (:prepaid_card_id, :owner_user_id, :shared_with_user_id)
            ");
            $stmt->execute([
                ':prepaid_card_id' => $prepaid_card_id,
                ':owner_user_id' => $user_id,
                ':shared_with_user_id' => $recipient_user_id
            ]);
            
            // Get owner name for notification
            $stmt = $conn->prepare("SELECT name, email FROM users WHERE id = :user_id");
            $stmt->execute([':user_id' => $user_id]);
            $owner = $stmt->fetch();
            $owner_name = $owner['name'] ?: $owner['email'];
            
            // Create notification for recipient
            $card_name = $prepaid_card['name'];
            createNotification(
                $conn,
                $recipient_user_id,
                'share_created',
                'Carta prepagata condivisa',
                $owner_name . ' ha condiviso la carta prepagata "' . $card_name . '" con te',
                'prepaid_card',
                $prepaid_card_id
            );
            
            echo json_encode([
                'success' => true,
                'message' => 'Prepaid card shared successfully',
                'user_exists' => true
            ]);
        } else {
            // User doesn't exist - create invitation
            // Check if invitation already exists
            $stmt = $conn->prepare("SELECT id FROM prepaid_card_share_invitations WHERE prepaid_card_id = :prepaid_card_id AND recipient_email = :recipient_email");
            $stmt->execute([':prepaid_card_id' => $prepaid_card_id, ':recipient_email' => $recipient_email]);
            if ($stmt->fetch()) {
                http_response_code(400);
                echo json_encode(['error' => 'Invitation already sent to this email']);
                break;
            }
            
            // Create invitation
            $stmt = $conn->prepare("
                INSERT INTO prepaid_card_share_invitations (prepaid_card_id, owner_user_id, recipient_email)
                VALUES (:prepaid_card_id, :owner_user_id, :recipient_email)
            ");
            $stmt->execute([
                ':prepaid_card_id' => $prepaid_card_id,
                ':owner_user_id' => $user_id,
                ':recipient_email' => $recipient_email
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Invitation sent. Prepaid card will be shared when user registers.',
                'user_exists' => false
            ]);
        }
        break;
        
    case 'DELETE':
        // Remove sharing
        $share_id = $_GET['share_id'] ?? null;
        $invitation_id = $_GET['invitation_id'] ?? null;
        
        if ($share_id) {
            // Remove active share
            $share_id = (int)$share_id;
            
            // Verify user owns the prepaid card being shared OR is the recipient
            $stmt = $conn->prepare("
                SELECT pcs.id, pcs.prepaid_card_id, pcs.owner_user_id, pcs.shared_with_user_id
                FROM prepaid_card_shares pcs
                WHERE pcs.id = :share_id
            ");
            $stmt->execute([':share_id' => $share_id]);
            $share = $stmt->fetch();
            
            if (!$share) {
                http_response_code(404);
                echo json_encode(['error' => 'Share not found']);
                break;
            }
            
            // Check if user is owner or recipient
            if ($share['owner_user_id'] != $user_id && $share['shared_with_user_id'] != $user_id) {
                http_response_code(403);
                echo json_encode(['error' => 'Not authorized']);
                break;
            }
            
            // Get card name and user names for notification
            $stmt = $conn->prepare("SELECT name FROM prepaid_cards WHERE id = :card_id");
            $stmt->execute([':card_id' => $share['prepaid_card_id']]);
            $card = $stmt->fetch();
            $card_name = $card ? $card['name'] : 'Carta prepagata';
            
            $stmt = $conn->prepare("SELECT id, name, email FROM users WHERE id IN (:user_id1, :user_id2)");
            $stmt->execute([
                ':user_id1' => $share['owner_user_id'],
                ':user_id2' => $share['shared_with_user_id']
            ]);
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $owner_name = '';
            $recipient_name = '';
            foreach ($users as $u) {
                if ($u['id'] == $share['owner_user_id']) {
                    $owner_name = $u['name'] ?: $u['email'];
                }
                if ($u['id'] == $share['shared_with_user_id']) {
                    $recipient_name = $u['name'] ?: $u['email'];
                }
            }
            
            // Delete share
            $stmt = $conn->prepare("DELETE FROM prepaid_card_shares WHERE id = :share_id");
            $stmt->execute([':share_id' => $share_id]);
            
            // Create notification for the other user (not the one who removed it)
            $notified_user_id = ($share['owner_user_id'] == $user_id) ? $share['shared_with_user_id'] : $share['owner_user_id'];
            $remover_name = ($share['owner_user_id'] == $user_id) ? $owner_name : $recipient_name;
            
            createNotification(
                $conn,
                $notified_user_id,
                'share_removed',
                'Condivisione rimossa',
                $remover_name . ' ha rimosso la condivisione della carta prepagata "' . $card_name . '"',
                'prepaid_card',
                $share['prepaid_card_id']
            );
            
            echo json_encode(['success' => true, 'message' => 'Sharing removed']);
        } elseif ($invitation_id) {
            // Remove invitation (owner only)
            $invitation_id = (int)$invitation_id;
            
            // Verify user owns the invitation
            $stmt = $conn->prepare("
                SELECT pcsi.id 
                FROM prepaid_card_share_invitations pcsi
                JOIN prepaid_cards pc ON pcsi.prepaid_card_id = pc.id
                WHERE pcsi.id = :invitation_id AND pc.user_id = :user_id
            ");
            $stmt->execute([':invitation_id' => $invitation_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Invitation not found']);
                break;
            }
            
            // Delete invitation
            $stmt = $conn->prepare("DELETE FROM prepaid_card_share_invitations WHERE id = :invitation_id");
            $stmt->execute([':invitation_id' => $invitation_id]);
            
            echo json_encode(['success' => true, 'message' => 'Invitation removed']);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Missing share_id or invitation_id']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
    error_log('Prepaid card sharing API error: ' . $e->getMessage());
}
