<?php
/**
 * Authentication Middleware
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';

/**
 * Log session debug information
 * @param string $event Event name (e.g., 'session_check', 'session_expired', 'cookie_updated')
 * @param array $data Additional data to log
 */
function logSessionDebug($event, $data = []) {
    if (!defined('SESSION_DEBUG_ENABLED') || !SESSION_DEBUG_ENABLED) {
        return;
    }
    
    $logDir = dirname(SESSION_DEBUG_LOG_FILE);
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    $sessionId = session_id() ?: 'no-session';
    $userId = $_SESSION['user_id'] ?? 'no-user';
    $timestamp = date('Y-m-d H:i:s');
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    $logData = [
        'timestamp' => $timestamp,
        'event' => $event,
        'session_id' => substr($sessionId, 0, 16) . '...', // Log only first 16 chars for security
        'user_id' => $userId,
        'ip' => $ip,
        'user_agent' => substr($userAgent, 0, 100), // Limit length
        'data' => $data
    ];
    
    $logLine = json_encode($logData, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n";
    
    @file_put_contents(SESSION_DEBUG_LOG_FILE, $logLine, FILE_APPEND | LOCK_EX);
}

/**
 * Update session cookie expiration (sliding expiration: 7 days from now)
 * Also checks absolute expiration (180 days from first login)
 * Uses session-based first_login_at (per device, not per user)
 * @param int $user_id The user ID (not used, kept for compatibility)
 * @param bool $exit_on_expiration If true, exit with error when expired. If false, return false.
 * @return bool Returns false if session expired (when $exit_on_expiration is false), true otherwise
 */
function updateSessionExpiration($user_id, $exit_on_expiration = true) {
    // Get first_login_at from session (per device)
    // If not set, set it now (for sessions created before this change)
    if (!isset($_SESSION['first_login_at'])) {
        $_SESSION['first_login_at'] = time();
        logSessionDebug('first_login_at_set', [
            'first_login_at' => date('Y-m-d H:i:s', $_SESSION['first_login_at'])
        ]);
    }
    
    $first_login_at = $_SESSION['first_login_at'];
    
    // Check absolute expiration: 180 days from first login for this device
    $absolute_expiration = $first_login_at + (180 * 24 * 60 * 60); // 180 days in seconds
    $now = time();
    
    $daysUntilAbsoluteExpiration = ($absolute_expiration - $now) / (24 * 60 * 60);
    
    if ($now > $absolute_expiration) {
        // Absolute expiration reached for this device
        logSessionDebug('session_expired_absolute', [
            'reason' => 'Absolute expiration reached (180 days)',
            'first_login_at' => date('Y-m-d H:i:s', $first_login_at),
            'absolute_expiration' => date('Y-m-d H:i:s', $absolute_expiration),
            'now' => date('Y-m-d H:i:s', $now),
            'exit_on_expiration' => $exit_on_expiration
        ]);
        session_destroy();
        if ($exit_on_expiration) {
            http_response_code(401);
            echo json_encode(['error' => 'Session expired. Please login again.']);
            exit;
        } else {
            return false;
        }
    }
    
    // Update session cookie expiration: 7 days from now (sliding expiration)
    // But don't exceed absolute expiration
    $cookie_expiration = min($now + (7 * 24 * 60 * 60), $absolute_expiration); // 7 days or absolute expiration, whichever comes first
    
    // Set session cookie with new expiration
    // Always update the cookie if session exists (even if cookie expired, session might still be valid)
    // This ensures the cookie is always persistent and extends expiration on each request
    $cookie_params = session_get_cookie_params();
    
    // Build Set-Cookie header manually to include SameSite attribute
    // PHP's setcookie() doesn't support SameSite directly, so we use header() instead
    $cookie_value = session_id();
    $cookie_name = session_name();
    $cookie_string = sprintf(
        '%s=%s; expires=%s; path=%s; %s%s; SameSite=%s',
        urlencode($cookie_name),
        urlencode($cookie_value),
        gmdate('D, d-M-Y H:i:s T', $cookie_expiration),
        $cookie_params['path'],
        $cookie_params['domain'] ? 'domain=' . $cookie_params['domain'] . '; ' : '',
        $cookie_params['httponly'] ? 'HttpOnly; ' : '',
        $cookie_params['samesite'] ?? 'Lax'
    );
    
    if ($cookie_params['secure']) {
        $cookie_string .= '; Secure';
    }
    
    header('Set-Cookie: ' . $cookie_string, false);
    
    // Log cookie update
    logSessionDebug('cookie_updated', [
        'cookie_expiration' => date('Y-m-d H:i:s', $cookie_expiration),
        'days_until_expiration' => round(($cookie_expiration - $now) / (24 * 60 * 60), 2),
        'days_until_absolute_expiration' => round($daysUntilAbsoluteExpiration, 2),
        'cookie_secure' => $cookie_params['secure'],
        'cookie_samesite' => $cookie_params['samesite'] ?? 'Lax',
        'cookie_domain' => $cookie_params['domain'] ?: 'current'
    ]);
    
    return true;
}

function requireAuth() {
    logSessionDebug('requireAuth_called', [
        'has_session' => isset($_SESSION),
        'has_user_id' => isset($_SESSION['user_id']),
        'session_id_exists' => session_id() !== ''
    ]);
    
    if (!isset($_SESSION['user_id'])) {
        logSessionDebug('auth_failed_no_user_id', [
            'reason' => 'No user_id in session',
            'session_keys' => isset($_SESSION) ? array_keys($_SESSION) : []
        ]);
        http_response_code(401);
        echo json_encode(['error' => 'Authentication required']);
        exit;
    }
    
    // Check if user is blocked
    $db = new Database();
    $conn = $db->getConnection();
    $stmt = $conn->prepare("SELECT is_blocked FROM users WHERE id = :user_id");
    $stmt->execute([':user_id' => $_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if ($user && $user['is_blocked']) {
        logSessionDebug('session_destroyed_user_blocked', [
            'reason' => 'User account is blocked',
            'user_id' => $_SESSION['user_id']
        ]);
        session_destroy();
        http_response_code(403);
        echo json_encode(['error' => 'Account is blocked. Please contact administrator.']);
        exit;
    }
    
    // Update session expiration (sliding expiration + check absolute expiration)
    updateSessionExpiration($_SESSION['user_id']);
    
    logSessionDebug('auth_success', [
        'user_id' => $_SESSION['user_id']
    ]);
    
    return $_SESSION['user_id'];
}

function requireAdmin() {
    $user_id = requireAuth();
    
    $db = new Database();
    $conn = $db->getConnection();
    $stmt = $conn->prepare("SELECT is_admin FROM users WHERE id = :user_id");
    $stmt->execute([':user_id' => $user_id]);
    $user = $stmt->fetch();
    
    if (!$user || !$user['is_admin']) {
        http_response_code(403);
        echo json_encode(['error' => 'Admin access required']);
        exit;
    }
    
    return $user_id;
}

