<?php
/**
 * Generate secure token for manifest shortcuts
 * This endpoint generates a secure token that allows Chrome to read shortcuts
 * without exposing the user_id directly in the URL
 */

ob_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/middleware/auth.php';

ini_set('display_errors', 0);
ob_clean();

header('Content-Type: application/json');

try {
    // User must be authenticated to generate a token
    $user_id = requireAuth();
    
    // Verify the user_id in request matches the authenticated user
    $raw_input = file_get_contents('php://input');
    $data = json_decode($raw_input, true);
    
    $requested_user_id = isset($data['user_id']) ? intval($data['user_id']) : 0;
    
    // Security check: user can only generate token for themselves
    if ($requested_user_id !== $user_id) {
        http_response_code(403);
        echo json_encode(['error' => 'Unauthorized: can only generate token for your own account']);
        exit;
    }
    
    // Generate secure token
    // Token format: base64(user_id:timestamp:hmac)
    // HMAC is calculated as: hash_hmac('sha256', user_id . timestamp, MANIFEST_TOKEN_SECRET)
    $timestamp = time();
    $hmac = hash_hmac('sha256', $user_id . ':' . $timestamp, MANIFEST_TOKEN_SECRET);
    $token_data = $user_id . ':' . $timestamp . ':' . $hmac;
    $token = base64_encode($token_data);
    
    echo json_encode([
        'token' => $token,
        'expires_in' => 86400 // 24 hours
    ]);
    
} catch (Exception $e) {
    http_response_code(401);
    echo json_encode(['error' => 'Authentication required']);
    error_log('Manifest token generation error: ' . $e->getMessage());
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error']);
    error_log('Manifest token generation fatal error: ' . $e->getMessage());
}
