<?php
/**
 * Fidelity Cards API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints to prevent HTML in JSON responses
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$card_id = isset($_GET['id']) ? intval($_GET['id']) : null;

switch ($method) {
    case 'GET':
        // Get all fidelity cards for user (owned + shared)
        // Use different parameter names for each part of UNION to avoid PDO parameter conflict
        $query = "
            SELECT 
                fc.*,
                NULL as sharing_mode,
                NULL as owner_name,
                NULL as owner_email,
                NULL as share_id,
                'owned' as card_status,
                CASE WHEN EXISTS (
                    SELECT 1 FROM fidelity_card_shares fcs WHERE fcs.fidelity_card_id = fc.id
                    UNION
                    SELECT 1 FROM fidelity_card_share_invitations fcsi WHERE fcsi.fidelity_card_id = fc.id
                ) THEN 1 ELSE 0 END as has_shares,
                COALESCE(ufco.display_order, 999999) as display_order,
                COALESCE(ufcc.click_count, 0) as click_count
            FROM fidelity_cards fc
            LEFT JOIN user_fidelity_card_order ufco ON ufco.user_id = :user_id_owned_order AND ufco.fidelity_card_id = fc.id
            LEFT JOIN user_fidelity_card_clicks ufcc ON ufcc.user_id = :user_id_owned_clicks AND ufcc.fidelity_card_id = fc.id
            WHERE fc.user_id = :user_id_owned
            
            UNION ALL
            
            SELECT 
                fc.*,
                'read' as sharing_mode,
                owner.name as owner_name,
                owner.email as owner_email,
                fcs.id as share_id,
                'shared' as card_status,
                0 as has_shares,
                COALESCE(ufco.display_order, 999999) as display_order,
                COALESCE(ufcc.click_count, 0) as click_count
            FROM fidelity_card_shares fcs
            JOIN fidelity_cards fc ON fcs.fidelity_card_id = fc.id
            JOIN users owner ON fcs.owner_user_id = owner.id
            LEFT JOIN user_fidelity_card_order ufco ON ufco.user_id = :user_id_shared_order AND ufco.fidelity_card_id = fc.id
            LEFT JOIN user_fidelity_card_clicks ufcc ON ufcc.user_id = :user_id_shared_clicks AND ufcc.fidelity_card_id = fc.id
            WHERE fcs.shared_with_user_id = :user_id_shared
            
            ORDER BY display_order ASC, click_count DESC, name ASC
        ";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':user_id_owned' => $user_id,
            ':user_id_owned_order' => $user_id,
            ':user_id_owned_clicks' => $user_id,
            ':user_id_shared' => $user_id,
            ':user_id_shared_order' => $user_id,
            ':user_id_shared_clicks' => $user_id
        ]);
        $cards = $stmt->fetchAll();
        
        echo json_encode($cards);
        break;
        
    case 'POST':
        // Create or update fidelity card - handle both JSON and FormData (for file uploads)
        $data = [];
        $logo_path = null;
        $card_image_path = null;
        $is_update = false;
        
        // Check if it's FormData or JSON
        $content_type = $_SERVER['CONTENT_TYPE'] ?? '';
        $is_multipart = strpos($content_type, 'multipart/form-data') !== false || !empty($_POST) || !empty($_FILES);
        
        if ($is_multipart) {
            // Handle FormData request
            $data = $_POST;
            
            // Check if this is an update (has 'id' field)
            if (!empty($data['id'])) {
                $is_update = true;
                $card_id = intval($data['id']);
                
                // Verify card belongs to user
                $stmt = $conn->prepare("SELECT id FROM fidelity_cards WHERE id = :card_id AND user_id = :user_id");
                $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
                if (!$stmt->fetch()) {
                    http_response_code(404);
                    echo json_encode(['error' => 'Card not found']);
                    break;
                }
            }
            
            // Handle logo from brand or upload
            if (!empty($data['brand_id'])) {
                // Copy logo from brand
                $brand_id = intval($data['brand_id']);
                $stmt = $conn->prepare("SELECT logo_path FROM fidelity_card_brands WHERE id = :brand_id");
                $stmt->execute([':brand_id' => $brand_id]);
                $brand = $stmt->fetch();
                
                if ($brand && $brand['logo_path']) {
                    $source_file = UPLOAD_DIR . $brand['logo_path'];
                    if (file_exists($source_file)) {
                        $upload_dir = UPLOAD_DIR . 'fidelity_cards/logos/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        // Delete old logo if updating
                        if ($is_update) {
                            $stmt = $conn->prepare("SELECT logo_path FROM fidelity_cards WHERE id = :card_id");
                            $stmt->execute([':card_id' => $card_id]);
                            $old_card = $stmt->fetch();
                            if ($old_card && $old_card['logo_path']) {
                                $old_file_path = UPLOAD_DIR . $old_card['logo_path'];
                                if (file_exists($old_file_path)) {
                                    unlink($old_file_path);
                                }
                            }
                        }
                        
                        $extension = pathinfo($brand['logo_path'], PATHINFO_EXTENSION);
                        $filename = uniqid() . '.' . $extension;
                        $dest_file = $upload_dir . $filename;
                        
                        if (copy($source_file, $dest_file)) {
                            $logo_path = 'fidelity_cards/logos/' . $filename;
                        }
                    }
                }
            } else if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                // Handle logo upload if provided
                $file = $_FILES['logo'];
                
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'fidelity_cards/logos/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Delete old logo if updating
                if ($is_update) {
                    $stmt = $conn->prepare("SELECT logo_path FROM fidelity_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $old_card = $stmt->fetch();
                    if ($old_card && $old_card['logo_path']) {
                        $old_file_path = UPLOAD_DIR . $old_card['logo_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $logo_path = 'fidelity_cards/logos/' . $filename;
                }
            } else if ($is_update) {
                // Get current logo path if not uploading new one
                $remove_logo = isset($data['remove_logo']) && ($data['remove_logo'] === true || $data['remove_logo'] === 'on');
                
                if ($remove_logo) {
                    // Delete old logo
                    $stmt = $conn->prepare("SELECT logo_path FROM fidelity_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $old_card = $stmt->fetch();
                    if ($old_card && $old_card['logo_path']) {
                        $old_file_path = UPLOAD_DIR . $old_card['logo_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                    $logo_path = null;
                } else {
                    // Keep current logo
                    $stmt = $conn->prepare("SELECT logo_path FROM fidelity_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $current_card = $stmt->fetch();
                    $logo_path = $current_card['logo_path'] ?? null;
                }
            }
            
            // Handle card image upload if provided
            if (isset($_FILES['card_image']) && $_FILES['card_image']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['card_image'];
                
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'fidelity_cards/images/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Delete old card image if updating
                if ($is_update) {
                    $stmt = $conn->prepare("SELECT card_image_path FROM fidelity_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $old_card = $stmt->fetch();
                    if ($old_card && $old_card['card_image_path']) {
                        $old_file_path = UPLOAD_DIR . $old_card['card_image_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $card_image_path = 'fidelity_cards/images/' . $filename;
                }
            } else if ($is_update) {
                // Get current card image path if not uploading new one
                $remove_card_image = isset($data['remove_card_image']) && ($data['remove_card_image'] === true || $data['remove_card_image'] === 'on');
                
                if ($remove_card_image) {
                    // Delete old card image
                    $stmt = $conn->prepare("SELECT card_image_path FROM fidelity_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $old_card = $stmt->fetch();
                    if ($old_card && $old_card['card_image_path']) {
                        $old_file_path = UPLOAD_DIR . $old_card['card_image_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                    $card_image_path = null;
                } else {
                    // Keep current card image
                    $stmt = $conn->prepare("SELECT card_image_path FROM fidelity_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $current_card = $stmt->fetch();
                    $card_image_path = $current_card['card_image_path'] ?? null;
                }
            }
        } else {
            // Handle JSON request
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
        }
        
        $required = ['name', 'store_name', 'card_code'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || $data[$field] === '') {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        if ($is_update) {
            // Update existing card
            $update_fields = [
                'name' => $data['name'],
                'store_name' => $data['store_name'],
                'card_code' => $data['card_code'],
                'description' => $data['description'] ?? null
            ];
            
            if ($logo_path !== null) {
                $update_fields['logo_path'] = $logo_path;
            }
            if ($card_image_path !== null) {
                $update_fields['card_image_path'] = $card_image_path;
            }
            
            $set_clause = [];
            $params = [':card_id' => $card_id, ':user_id' => $user_id];
            
            foreach ($update_fields as $field => $value) {
                $set_clause[] = "$field = :$field";
                $params[":$field"] = $value;
            }
            
            $stmt = $conn->prepare("
                UPDATE fidelity_cards 
                SET " . implode(', ', $set_clause) . "
                WHERE id = :card_id AND user_id = :user_id
            ");
            
            $stmt->execute($params);
            
            echo json_encode(['success' => true]);
        } else {
            // Create new card
            $stmt = $conn->prepare("
                INSERT INTO fidelity_cards (user_id, name, store_name, card_code, description, logo_path, card_image_path)
                VALUES (:user_id, :name, :store_name, :card_code, :description, :logo_path, :card_image_path)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':name' => $data['name'],
                ':store_name' => $data['store_name'],
                ':card_code' => $data['card_code'],
                ':description' => $data['description'] ?? null,
                ':logo_path' => $logo_path,
                ':card_image_path' => $card_image_path
            ]);
            
            $card_id = $conn->lastInsertId();
            
            echo json_encode(['success' => true, 'id' => $card_id]);
        }
        break;
        
    case 'PUT':
        // Update fidelity card
        if (!$card_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Card ID is required']);
            break;
        }
        
        // Verify card belongs to user
        $stmt = $conn->prepare("SELECT id FROM fidelity_cards WHERE id = :card_id AND user_id = :user_id");
        $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Card not found']);
            break;
        }
        
        // Handle FormData (for file uploads) or JSON
        $content_type = $_SERVER['CONTENT_TYPE'] ?? '';
        $is_multipart = strpos($content_type, 'multipart/form-data') !== false || !empty($_POST) || !empty($_FILES);
        
        if ($is_multipart) {
            if (!empty($_POST)) {
                $data = $_POST;
            } else if (!empty($_REQUEST)) {
                $data = $_REQUEST;
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'FormData fields not received']);
                break;
            }
        } else {
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
        }
        
        $required = ['name', 'store_name', 'card_code'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || $data[$field] === '') {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        $logo_path = null;
        $card_image_path = null;
        $remove_logo = isset($data['remove_logo']) && $data['remove_logo'] === true;
        $remove_card_image = isset($data['remove_card_image']) && $data['remove_card_image'] === true;
        
        // Handle logo from brand or upload
        if (!empty($data['brand_id'])) {
            // Copy logo from brand
            $brand_id = intval($data['brand_id']);
            $stmt = $conn->prepare("SELECT logo_path FROM fidelity_card_brands WHERE id = :brand_id");
            $stmt->execute([':brand_id' => $brand_id]);
            $brand = $stmt->fetch();
            
            if ($brand && $brand['logo_path']) {
                $source_file = UPLOAD_DIR . $brand['logo_path'];
                if (file_exists($source_file)) {
                    $upload_dir = UPLOAD_DIR . 'fidelity_cards/logos/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    // Delete old logo
                    if ($current_card['logo_path']) {
                        $old_file_path = UPLOAD_DIR . $current_card['logo_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                    
                    $extension = pathinfo($brand['logo_path'], PATHINFO_EXTENSION);
                    $filename = uniqid() . '.' . $extension;
                    $dest_file = $upload_dir . $filename;
                    
                    if (copy($source_file, $dest_file)) {
                        $logo_path = 'fidelity_cards/logos/' . $filename;
                    }
                }
            }
        } else if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
            // Handle logo upload if provided
            $file = $_FILES['logo'];
            
            if ($file['size'] > MAX_UPLOAD_SIZE) {
                http_response_code(400);
                echo json_encode(['error' => 'File too large']);
                break;
            }
            
            $upload_dir = UPLOAD_DIR . 'fidelity_cards/logos/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            // Delete old logo
            if ($current_card['logo_path']) {
                $old_file_path = UPLOAD_DIR . $current_card['logo_path'];
                if (file_exists($old_file_path)) {
                    unlink($old_file_path);
                }
            }
            
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = uniqid() . '.' . $extension;
            $file_path = $upload_dir . $filename;
            
            if (move_uploaded_file($file['tmp_name'], $file_path)) {
                $logo_path = 'fidelity_cards/logos/' . $filename;
            }
        }
        
        // Handle card image upload if provided
        if (isset($_FILES['card_image']) && $_FILES['card_image']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['card_image'];
            
            if ($file['size'] > MAX_UPLOAD_SIZE) {
                http_response_code(400);
                echo json_encode(['error' => 'File too large']);
                break;
            }
            
            $upload_dir = UPLOAD_DIR . 'fidelity_cards/images/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = uniqid() . '.' . $extension;
            $file_path = $upload_dir . $filename;
            
            if (move_uploaded_file($file['tmp_name'], $file_path)) {
                $card_image_path = 'fidelity_cards/images/' . $filename;
            }
        }
        
        // Get current paths if not removing and not uploading new ones
        $stmt = $conn->prepare("SELECT logo_path, card_image_path FROM fidelity_cards WHERE id = :card_id");
        $stmt->execute([':card_id' => $card_id]);
        $current_card = $stmt->fetch();
        
        if (!$remove_logo && !$logo_path) {
            $logo_path = $current_card['logo_path'];
        }
        if (!$remove_card_image && !$card_image_path) {
            $card_image_path = $current_card['card_image_path'];
        }
        
        // Delete old files if removing or uploading new ones
        if ($remove_logo || ($logo_path && isset($_FILES['logo']))) {
            if ($current_card['logo_path']) {
                $old_file_path = UPLOAD_DIR . $current_card['logo_path'];
                if (file_exists($old_file_path)) {
                    unlink($old_file_path);
                }
            }
        }
        
        if ($remove_card_image || ($card_image_path && isset($_FILES['card_image']))) {
            if ($current_card['card_image_path']) {
                $old_file_path = UPLOAD_DIR . $current_card['card_image_path'];
                if (file_exists($old_file_path)) {
                    unlink($old_file_path);
                }
            }
        }
        
        $stmt = $conn->prepare("
            UPDATE fidelity_cards 
            SET name = :name, 
                store_name = :store_name, 
                card_code = :card_code, 
                description = :description,
                logo_path = :logo_path,
                card_image_path = :card_image_path
            WHERE id = :card_id AND user_id = :user_id
        ");
        
        $stmt->execute([
            ':card_id' => $card_id,
            ':user_id' => $user_id,
            ':name' => $data['name'],
            ':store_name' => $data['store_name'],
            ':card_code' => $data['card_code'],
            ':description' => $data['description'] ?? null,
            ':logo_path' => $logo_path,
            ':card_image_path' => $card_image_path
        ]);
        
        echo json_encode(['success' => true]);
        break;
        
    case 'DELETE':
        // Delete fidelity card
        if (!$card_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Card ID is required']);
            break;
        }
        
        // Verify card belongs to user and get file paths
        $stmt = $conn->prepare("SELECT logo_path, card_image_path FROM fidelity_cards WHERE id = :card_id AND user_id = :user_id");
        $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
        $card = $stmt->fetch();
        
        if (!$card) {
            http_response_code(404);
            echo json_encode(['error' => 'Card not found']);
            break;
        }
        
        // Delete files if exist
        if ($card['logo_path']) {
            $logo_file_path = UPLOAD_DIR . $card['logo_path'];
            if (file_exists($logo_file_path)) {
                unlink($logo_file_path);
            }
        }
        
        if ($card['card_image_path']) {
            $image_file_path = UPLOAD_DIR . $card['card_image_path'];
            if (file_exists($image_file_path)) {
                unlink($image_file_path);
            }
        }
        
        // Delete card
        $stmt = $conn->prepare("DELETE FROM fidelity_cards WHERE id = :card_id AND user_id = :user_id");
        $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
        
        echo json_encode(['success' => true]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error processing fidelity card: ' . $e->getMessage()]);
    error_log('Fidelity card API error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
    error_log('Fidelity card API fatal error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
}
