<?php
/**
 * Fidelity Card Brands API
 * Admin-only API for managing fidelity card brand templates
 */

ob_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

ini_set('display_errors', 0);
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();
    
    // Check if user is admin
    $stmt = $conn->prepare("SELECT is_admin FROM users WHERE id = :user_id");
    $stmt->execute([':user_id' => $user_id]);
    $user = $stmt->fetch();
    
    if (!$user || !$user['is_admin']) {
        http_response_code(403);
        echo json_encode(['error' => 'Access denied. Admin only.']);
        exit;
    }
    
    $method = $_SERVER['REQUEST_METHOD'];
    $brand_id = isset($_GET['id']) ? intval($_GET['id']) : null;
    
    switch ($method) {
        case 'GET':
            if ($brand_id) {
                // Get single brand
                $stmt = $conn->prepare("SELECT * FROM fidelity_card_brands WHERE id = :id");
                $stmt->execute([':id' => $brand_id]);
                $brand = $stmt->fetch();
                
                if ($brand) {
                    echo json_encode($brand);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Brand not found']);
                }
            } else {
                // Get all brands
                $stmt = $conn->prepare("SELECT * FROM fidelity_card_brands ORDER BY name ASC");
                $stmt->execute();
                $brands = $stmt->fetchAll(PDO::FETCH_ASSOC);
                echo json_encode($brands);
            }
            break;
            
        case 'POST':
            // Create new brand
            $data = [];
            $logo_path = null;
            
            // Check if it's FormData or JSON
            if (!empty($_POST)) {
                $data = $_POST;
            } else {
                $input = file_get_contents('php://input');
                $data = json_decode($input, true) ?? [];
            }
            
            if (empty($data['name'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Brand name is required']);
                break;
            }
            
            // Check if brand name already exists
            $stmt = $conn->prepare("SELECT id FROM fidelity_card_brands WHERE name = :name");
            $stmt->execute([':name' => $data['name']]);
            if ($stmt->fetch()) {
                http_response_code(400);
                echo json_encode(['error' => 'Brand name already exists']);
                break;
            }
            
            // Handle logo upload
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['logo'];
                
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'fidelity_cards/brands/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $logo_path = 'fidelity_cards/brands/' . $filename;
                } else {
                    http_response_code(500);
                    echo json_encode(['error' => 'Failed to upload logo']);
                    break;
                }
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Logo file is required']);
                break;
            }
            
            // Insert brand
            $stmt = $conn->prepare("
                INSERT INTO fidelity_card_brands (name, logo_path)
                VALUES (:name, :logo_path)
            ");
            $stmt->execute([
                ':name' => $data['name'],
                ':logo_path' => $logo_path
            ]);
            
            $new_brand_id = $conn->lastInsertId();
            
            // Return created brand
            $stmt = $conn->prepare("SELECT * FROM fidelity_card_brands WHERE id = :id");
            $stmt->execute([':id' => $new_brand_id]);
            $brand = $stmt->fetch();
            
            echo json_encode($brand);
            break;
            
        case 'PUT':
            // Update brand
            if (!$brand_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Brand ID is required']);
                break;
            }
            
            $data = [];
            $logo_path = null;
            
            // Check if it's FormData or JSON
            if (!empty($_POST)) {
                $data = $_POST;
            } else {
                $input = file_get_contents('php://input');
                $data = json_decode($input, true) ?? [];
            }
            
            // Get current brand
            $stmt = $conn->prepare("SELECT * FROM fidelity_card_brands WHERE id = :id");
            $stmt->execute([':id' => $brand_id]);
            $current_brand = $stmt->fetch();
            
            if (!$current_brand) {
                http_response_code(404);
                echo json_encode(['error' => 'Brand not found']);
                break;
            }
            
            // Check if name is being changed and if new name already exists
            if (isset($data['name']) && $data['name'] !== $current_brand['name']) {
                $stmt = $conn->prepare("SELECT id FROM fidelity_card_brands WHERE name = :name AND id != :id");
                $stmt->execute([':name' => $data['name'], ':id' => $brand_id]);
                if ($stmt->fetch()) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Brand name already exists']);
                    break;
                }
            }
            
            // Handle logo upload if provided
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['logo'];
                
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'fidelity_cards/brands/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Delete old logo
                if ($current_brand['logo_path']) {
                    $old_file_path = UPLOAD_DIR . $current_brand['logo_path'];
                    if (file_exists($old_file_path)) {
                        unlink($old_file_path);
                    }
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $logo_path = 'fidelity_cards/brands/' . $filename;
                } else {
                    http_response_code(500);
                    echo json_encode(['error' => 'Failed to upload logo']);
                    break;
                }
            } else {
                // Keep current logo if not uploading new one
                $remove_logo = isset($data['remove_logo']) && ($data['remove_logo'] === true || $data['remove_logo'] === 'on');
                
                if ($remove_logo) {
                    // Delete old logo
                    if ($current_brand['logo_path']) {
                        $old_file_path = UPLOAD_DIR . $current_brand['logo_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                    $logo_path = null;
                } else {
                    $logo_path = $current_brand['logo_path'];
                }
            }
            
            // Update brand
            $update_fields = [];
            $params = [':id' => $brand_id];
            
            if (isset($data['name'])) {
                $update_fields[] = 'name = :name';
                $params[':name'] = $data['name'];
            }
            
            if ($logo_path !== null) {
                $update_fields[] = 'logo_path = :logo_path';
                $params[':logo_path'] = $logo_path;
            }
            
            if (empty($update_fields)) {
                http_response_code(400);
                echo json_encode(['error' => 'No fields to update']);
                break;
            }
            
            $sql = "UPDATE fidelity_card_brands SET " . implode(', ', $update_fields) . " WHERE id = :id";
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            // Return updated brand
            $stmt = $conn->prepare("SELECT * FROM fidelity_card_brands WHERE id = :id");
            $stmt->execute([':id' => $brand_id]);
            $brand = $stmt->fetch();
            
            echo json_encode($brand);
            break;
            
        case 'DELETE':
            // Delete brand
            if (!$brand_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Brand ID is required']);
                break;
            }
            
            // Get brand to delete logo file
            $stmt = $conn->prepare("SELECT logo_path FROM fidelity_card_brands WHERE id = :id");
            $stmt->execute([':id' => $brand_id]);
            $brand = $stmt->fetch();
            
            if (!$brand) {
                http_response_code(404);
                echo json_encode(['error' => 'Brand not found']);
                break;
            }
            
            // Delete logo file
            if ($brand['logo_path']) {
                $file_path = UPLOAD_DIR . $brand['logo_path'];
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
            }
            
            // Delete brand
            $stmt = $conn->prepare("DELETE FROM fidelity_card_brands WHERE id = :id");
            $stmt->execute([':id' => $brand_id]);
            
            echo json_encode(['success' => true, 'message' => 'Brand deleted']);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
    error_log('Fidelity card brands API error: ' . $e->getMessage());
}
