<?php
/**
 * Expenses API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints to prevent HTML in JSON responses
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
    case 'GET':
        // Get expenses - filter by account_id(s), prepaid_card_id(s), date range, description keywords, and deductible status
        $account_id = $_GET['account_id'] ?? null; // Single account (backward compatibility)
        $account_ids = $_GET['account_ids'] ?? null; // Multiple accounts (can be array or comma-separated)
        $prepaid_card_ids = $_GET['prepaid_card_ids'] ?? null; // Multiple prepaid cards (can be array or comma-separated)
        $start_date = $_GET['start_date'] ?? null;
        $end_date = $_GET['end_date'] ?? null;
        $description_search = $_GET['description_search'] ?? null;
        $is_tax_deductible = $_GET['is_tax_deductible'] ?? null;
        
        // Handle account_ids - support both array and comma-separated string
        $account_ids_array = [];
        if ($account_ids) {
            if (is_array($account_ids)) {
                $account_ids_array = array_filter(array_map('intval', $account_ids));
            } else {
                $account_ids_array = array_filter(array_map('intval', explode(',', $account_ids)));
            }
        }
        // If single account_id is provided, add it to the array
        if ($account_id && !in_array((int)$account_id, $account_ids_array)) {
            $account_ids_array[] = (int)$account_id;
        }
        
        // Handle prepaid_card_ids - support both array and comma-separated string
        $prepaid_card_ids_array = [];
        if ($prepaid_card_ids) {
            if (is_array($prepaid_card_ids)) {
                $prepaid_card_ids_array = array_filter(array_map('intval', $prepaid_card_ids));
            } else {
                $prepaid_card_ids_array = array_filter(array_map('intval', explode(',', $prepaid_card_ids)));
            }
        }
        
        // Build SQL query parts based on filters
        $sql_parts = [];
        $params = [];
        
        // Build WHERE conditions for filtering (will be applied after UNION)
        $where_conditions = [];
        
        // Include account expenses if:
        // - No filters specified (show all)
        // - Filtering by accounts
        // - Filtering by both accounts and prepaid cards
        if (empty($account_ids_array) && empty($prepaid_card_ids_array) || !empty($account_ids_array)) {
            $sql_account_expenses = "
                SELECT e.*, 
                       a.name as account_name,
                       NULL as prepaid_card_name,
                       a.name as source_name,
                       a.user_id as account_owner_id,
                       tdc.name as tax_category_name, tdc.code as tax_category_code,
                       NULL as transfer_id,
                       0 as is_transfer,
                       NULL as transfer_type,
                       creator.name as created_by_name,
                       creator.email as created_by_email
                FROM expenses e
                JOIN accounts a ON e.account_id = a.id
                LEFT JOIN tax_deduction_categories tdc ON e.tax_category_id = tdc.id
                LEFT JOIN users creator ON e.user_id = creator.id
                WHERE e.account_id IS NOT NULL
                AND (
                    e.user_id = :user_id_acc 
                    OR EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = e.account_id 
                        AND ash.shared_with_user_id = :user_id_shared_acc
                    )
                    OR a.user_id = :user_id_owner_acc
                )
            ";
            
            // Add parameters for account expenses
            $params[':user_id_acc'] = $user_id;
            $params[':user_id_shared_acc'] = $user_id;
            $params[':user_id_owner_acc'] = $user_id;
            
            // Add account filter if specified
            if (!empty($account_ids_array)) {
                $placeholders = [];
                foreach ($account_ids_array as $idx => $acc_id) {
                    $key = ':account_id_' . $idx;
                    $placeholders[] = $key;
                    $params[$key] = $acc_id;
                }
                $sql_account_expenses .= " AND e.account_id IN (" . implode(', ', $placeholders) . ")";
            }
            
            $sql_parts[] = $sql_account_expenses;
        }
        
        // Include prepaid card expenses if:
        // - No filters specified (show all)
        // - Filtering by prepaid cards
        // - Filtering by both accounts and prepaid cards
        if (empty($account_ids_array) && empty($prepaid_card_ids_array) || !empty($prepaid_card_ids_array)) {
            $sql_prepaid_expenses = "
                SELECT e.*,
                       NULL as account_name,
                       pc.name as prepaid_card_name,
                       pc.name as source_name,
                       pc.user_id as account_owner_id,
                       tdc.name as tax_category_name, tdc.code as tax_category_code,
                       NULL as transfer_id,
                       0 as is_transfer,
                       NULL as transfer_type,
                       creator.name as created_by_name,
                       creator.email as created_by_email
                FROM expenses e
                JOIN prepaid_cards pc ON e.prepaid_card_id = pc.id
                LEFT JOIN tax_deduction_categories tdc ON e.tax_category_id = tdc.id
                LEFT JOIN users creator ON e.user_id = creator.id
                WHERE e.prepaid_card_id IS NOT NULL
                AND (
                    e.user_id = :user_id_prepaid
                    OR EXISTS (
                        SELECT 1 FROM prepaid_card_shares pcs 
                        WHERE pcs.prepaid_card_id = e.prepaid_card_id 
                        AND pcs.shared_with_user_id = :user_id_shared_prepaid
                    )
                    OR pc.user_id = :user_id_owner_prepaid
                )
            ";
            
            // Add parameters for prepaid card expenses
            $params[':user_id_prepaid'] = $user_id;
            $params[':user_id_shared_prepaid'] = $user_id;
            $params[':user_id_owner_prepaid'] = $user_id;
            
            // Add prepaid card filter if specified
            if (!empty($prepaid_card_ids_array)) {
                $placeholders = [];
                foreach ($prepaid_card_ids_array as $idx => $card_id) {
                    $key = ':prepaid_card_id_' . $idx;
                    $placeholders[] = $key;
                    $params[$key] = $card_id;
                }
                $sql_prepaid_expenses .= " AND e.prepaid_card_id IN (" . implode(', ', $placeholders) . ")";
            }
            
            $sql_parts[] = $sql_prepaid_expenses;
        }
        
        // Combine parts with UNION ALL
        // If no parts, return empty result
        if (empty($sql_parts)) {
            $sql_expenses = "SELECT NULL as id WHERE 1=0";
        } else {
            $sql_expenses = implode(' UNION ALL ', $sql_parts);
        }
        
        if ($start_date) {
            $where_conditions[] = "date >= :start_date";
            $params[':start_date'] = $start_date;
        }
        
        if ($end_date) {
            $where_conditions[] = "date <= :end_date";
            $params[':end_date'] = $end_date;
        }
        
        if ($description_search) {
            $where_conditions[] = "description LIKE :description_search";
            $params[':description_search'] = '%' . $description_search . '%';
        }
        
        if ($is_tax_deductible !== null && $is_tax_deductible !== '') {
            $where_conditions[] = "is_tax_deductible = :is_tax_deductible";
            $params[':is_tax_deductible'] = ($is_tax_deductible === '1' || $is_tax_deductible === 'true' || $is_tax_deductible === true) ? 1 : 0;
        }
        
        // Wrap the UNION query and apply WHERE conditions if any
        if (!empty($where_conditions) && !empty($sql_parts)) {
            $sql_expenses = "SELECT * FROM (" . $sql_expenses . ") AS combined_expenses WHERE " . implode(' AND ', $where_conditions);
        } else if (!empty($sql_parts)) {
            // Even if no WHERE conditions, wrap in SELECT to ensure consistent column names
            $sql_expenses = "SELECT * FROM (" . $sql_expenses . ") AS combined_expenses";
        }
        
        // Get transfers as expenses (outgoing from source account)
        // Include transfers made by user OR transfers from accounts owned by user OR transfers from shared accounts (read-only or write access)
        $sql_transfers_out = "
            SELECT 
                NULL as id,
                ft.user_id,
                ft.from_account_id as account_id,
                ft.transfer_date as date,
                ft.arrival_date,
                CONCAT('Trasferimento a ', ta.name) as description,
                -ft.amount as amount,
                ft.from_currency as currency,
                0 as is_tax_deductible,
                NULL as tax_category_id,
                NULL as created_at,
                NULL as updated_at,
                fa.name as account_name,
                fa.user_id as account_owner_id,
                NULL as tax_category_name,
                NULL as tax_category_code,
                ft.id as transfer_id,
                1 as is_transfer,
                'OUTGOING' as transfer_type,
                creator.name as created_by_name,
                creator.email as created_by_email
            FROM fund_transfers ft
            JOIN accounts fa ON ft.from_account_id = fa.id
            JOIN accounts ta ON ft.to_account_id = ta.id
            LEFT JOIN users creator ON ft.user_id = creator.id
            WHERE (
                ft.user_id = :user_id
                OR fa.user_id = :user_id_owner
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = ft.from_account_id 
                    AND ash.shared_with_user_id = :user_id_transfer_out
                )
            )
        ";
        
        // Create separate params array for transfers to avoid conflicts
        $params_transfers_out = [
            ':user_id' => $user_id, 
            ':user_id_owner' => $user_id,
            ':user_id_transfer_out' => $user_id
        ];
        
        if (!empty($account_ids_array)) {
            $placeholders = [];
            foreach ($account_ids_array as $idx => $acc_id) {
                $key = ':transfer_out_account_id_' . $idx;
                $placeholders[] = $key;
                $params_transfers_out[$key] = $acc_id;
            }
            $sql_transfers_out .= " AND ft.from_account_id IN (" . implode(', ', $placeholders) . ")";
        }
        
        if ($start_date) {
            $sql_transfers_out .= " AND ft.transfer_date >= :transfer_out_start_date";
            $params_transfers_out[':transfer_out_start_date'] = $start_date;
        }
        
        if ($end_date) {
            $sql_transfers_out .= " AND ft.transfer_date <= :transfer_out_end_date";
            $params_transfers_out[':transfer_out_end_date'] = $end_date;
        }
        
        // Note: Transfers don't support description search or deductible filter
        
        // Get transfers as expenses (incoming to destination account)
        // Include transfers made by user OR transfers to accounts owned by user OR transfers to shared accounts (read-only or write access)
        $sql_transfers_in = "
            SELECT 
                NULL as id,
                ft.user_id,
                ft.to_account_id as account_id,
                COALESCE(ft.arrival_date, ft.transfer_date) as date,
                ft.arrival_date,
                CONCAT('Trasferimento da ', fa.name) as description,
                ft.converted_amount as amount,
                ft.to_currency as currency,
                0 as is_tax_deductible,
                NULL as tax_category_id,
                NULL as created_at,
                NULL as updated_at,
                ta.name as account_name,
                ta.user_id as account_owner_id,
                NULL as tax_category_name,
                NULL as tax_category_code,
                ft.id as transfer_id,
                1 as is_transfer,
                'INCOMING' as transfer_type,
                creator.name as created_by_name,
                creator.email as created_by_email
            FROM fund_transfers ft
            JOIN accounts fa ON ft.from_account_id = fa.id
            JOIN accounts ta ON ft.to_account_id = ta.id
            LEFT JOIN users creator ON ft.user_id = creator.id
            WHERE (
                ft.user_id = :user_id
                OR ta.user_id = :user_id_owner_to
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = ft.to_account_id 
                    AND ash.shared_with_user_id = :user_id_transfer_in
                )
            )
        ";
        
        // Create separate params array for transfers to avoid conflicts
        $params_transfers_in = [
            ':user_id' => $user_id, 
            ':user_id_owner_to' => $user_id,
            ':user_id_transfer_in' => $user_id
        ];
        
        if (!empty($account_ids_array)) {
            $placeholders = [];
            foreach ($account_ids_array as $idx => $acc_id) {
                $key = ':transfer_in_account_id_' . $idx;
                $placeholders[] = $key;
                $params_transfers_in[$key] = $acc_id;
            }
            $sql_transfers_in .= " AND ft.to_account_id IN (" . implode(', ', $placeholders) . ")";
        }
        
        if ($start_date) {
            $sql_transfers_in .= " AND ft.transfer_date >= :transfer_in_start_date";
            $params_transfers_in[':transfer_in_start_date'] = $start_date;
        }
        
        if ($end_date) {
            $sql_transfers_in .= " AND ft.transfer_date <= :transfer_in_end_date";
            $params_transfers_in[':transfer_in_end_date'] = $end_date;
        }
        
        // Note: Transfers don't support description search or deductible filter
        
        // Execute queries separately and combine results
        $all_expenses = [];
        
        // Get regular expenses
        $stmt = $conn->prepare($sql_expenses);
        $stmt->execute($params);
        $expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $all_expenses = array_merge($all_expenses, $expenses);
        
        // Only include transfers if:
        // 1. Not filtering by description or deductible status (transfers don't support these filters)
        // 2. Not filtering only by prepaid cards (transfers are only between accounts, not prepaid cards)
        // 3. If filtering by accounts, include transfers; if filtering only by prepaid cards, exclude transfers
        $should_include_transfers = !$description_search && 
                                   ($is_tax_deductible === null || $is_tax_deductible === '') &&
                                   (empty($prepaid_card_ids_array) || !empty($account_ids_array));
        
        if ($should_include_transfers) {
            // Get outgoing transfers
            $stmt = $conn->prepare($sql_transfers_out);
            $stmt->execute($params_transfers_out);
            $transfers_out = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $all_expenses = array_merge($all_expenses, $transfers_out);
            
            // Get incoming transfers
            $stmt = $conn->prepare($sql_transfers_in);
            $stmt->execute($params_transfers_in);
            $transfers_in = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $all_expenses = array_merge($all_expenses, $transfers_in);
        }
        
        // Sort by date DESC, then by created_at DESC
        usort($all_expenses, function($a, $b) {
            $dateA = $a['date'] ?? $a['created_at'] ?? '';
            $dateB = $b['date'] ?? $b['created_at'] ?? '';
            if ($dateA === $dateB) {
                $createdA = $a['created_at'] ?? '';
                $createdB = $b['created_at'] ?? '';
                return strcmp($createdB, $createdA);
            }
            return strcmp($dateB, $dateA);
        });
        
        // Get attachments for each expense
        foreach ($all_expenses as &$expense) {
            if (!$expense['is_transfer'] && $expense['id']) {
                $stmt = $conn->prepare("SELECT * FROM expense_attachments WHERE expense_id = :expense_id");
                $stmt->execute([':expense_id' => $expense['id']]);
                $expense['attachments'] = $stmt->fetchAll();
            } else {
                // Get transfer attachments if it's a transfer
                if ($expense['transfer_id']) {
                    $stmt = $conn->prepare("SELECT * FROM transfer_attachments WHERE transfer_id = :transfer_id");
                    $stmt->execute([':transfer_id' => $expense['transfer_id']]);
                    $expense['attachments'] = $stmt->fetchAll();
                } else {
                    $expense['attachments'] = [];
                }
            }
        }
        
        echo json_encode($all_expenses);
        break;
        
    case 'POST':
        // Create new expense
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Require either account_id or prepaid_card_id
        if (!isset($data['account_id']) && !isset($data['prepaid_card_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Either account_id or prepaid_card_id is required']);
            exit;
        }
        
        $required = ['date', 'description', 'amount', 'currency'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                exit;
            }
        }
        
        $account_id = null;
        $prepaid_card_id = null;
        
        if (isset($data['account_id'])) {
            // Verify account belongs to user OR user has write access via sharing
            $stmt = $conn->prepare("
                SELECT 
                    a.id, 
                    at.code as type_code,
                    CASE 
                        WHEN a.user_id = :user_id_check_post THEN 'owned'
                        WHEN EXISTS (
                            SELECT 1 FROM account_shares ash 
                            WHERE ash.account_id = a.id 
                            AND ash.shared_with_user_id = :user_id_shared_check_post
                            AND ash.sharing_mode = 'write'
                        ) THEN 'shared_write'
                        ELSE 'no_access'
                    END as access_type
                FROM accounts a
                JOIN account_types at ON a.type_id = at.id
                WHERE a.id = :account_id 
                AND (
                    a.user_id = :user_id_owner_post 
                    OR EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = a.id 
                        AND ash.shared_with_user_id = :user_id_shared_post
                        AND ash.sharing_mode = 'write'
                    )
                )
            ");
            $stmt->execute([
                ':account_id' => $data['account_id'], 
                ':user_id_check_post' => $user_id,
                ':user_id_shared_check_post' => $user_id,
                ':user_id_owner_post' => $user_id,
                ':user_id_shared_post' => $user_id
            ]);
            $account = $stmt->fetch();
            
            if (!$account) {
                http_response_code(404);
                echo json_encode(['error' => 'Account not found or you do not have write access']);
                exit;
            }
            
            if ($account['type_code'] === 'SECURITIES') {
                http_response_code(400);
                echo json_encode(['error' => 'Cannot add expenses to securities accounts. Use securities purchases/sales instead.']);
                exit;
            }
            
            if ($account['access_type'] === 'no_access') {
                http_response_code(403);
                echo json_encode(['error' => 'You do not have write access to this account']);
                exit;
            }
            
            $account_id = $data['account_id'];
        } else if (isset($data['prepaid_card_id'])) {
            // Verify prepaid card belongs to user OR is shared with user and is enabled
            $stmt = $conn->prepare("
                SELECT 
                    pc.id,
                    CASE 
                        WHEN pc.user_id = :user_id_check THEN 'owned'
                        WHEN EXISTS (
                            SELECT 1 FROM prepaid_card_shares pcs 
                            WHERE pcs.prepaid_card_id = pc.id 
                            AND pcs.shared_with_user_id = :user_id_shared_check
                        ) THEN 'shared'
                        ELSE 'no_access'
                    END as access_type
                FROM prepaid_cards pc
                WHERE pc.id = :prepaid_card_id 
                AND pc.enabled = 1
                AND pc.archived = 0
                AND (
                    pc.user_id = :user_id_owner 
                    OR EXISTS (
                        SELECT 1 FROM prepaid_card_shares pcs 
                        WHERE pcs.prepaid_card_id = pc.id 
                        AND pcs.shared_with_user_id = :user_id_shared
                    )
                )
            ");
            $stmt->execute([
                ':prepaid_card_id' => $data['prepaid_card_id'],
                ':user_id_check' => $user_id,
                ':user_id_shared_check' => $user_id,
                ':user_id_owner' => $user_id,
                ':user_id_shared' => $user_id
            ]);
            $prepaid_card = $stmt->fetch();
            
            if (!$prepaid_card) {
                http_response_code(404);
                echo json_encode(['error' => 'Prepaid card not found, not enabled, or you do not have access']);
                exit;
            }
            
            if ($prepaid_card['access_type'] === 'no_access') {
                http_response_code(403);
                echo json_encode(['error' => 'You do not have access to this prepaid card']);
                exit;
            }
            
            $prepaid_card_id = $data['prepaid_card_id'];
        }
        
        // Validate tax category if tax deductible
        $tax_category_id = null;
        if (!empty($data['is_tax_deductible']) && isset($data['tax_category_id'])) {
            $stmt = $conn->prepare("SELECT id FROM tax_deduction_categories WHERE id = :category_id");
            $stmt->execute([':category_id' => $data['tax_category_id']]);
            if ($stmt->fetch()) {
                $tax_category_id = $data['tax_category_id'];
            }
        }
        
        $stmt = $conn->prepare("
            INSERT INTO expenses (user_id, account_id, prepaid_card_id, date, description, amount, currency, is_tax_deductible, tax_category_id)
            VALUES (:user_id, :account_id, :prepaid_card_id, :date, :description, :amount, :currency, :is_tax_deductible, :tax_category_id)
        ");
        
        $stmt->execute([
            ':user_id' => $user_id,
            ':account_id' => $account_id,
            ':prepaid_card_id' => $prepaid_card_id,
            ':date' => $data['date'],
            ':description' => $data['description'],
            ':amount' => $data['amount'],
            ':currency' => $data['currency'],
            ':is_tax_deductible' => !empty($data['is_tax_deductible']) ? 1 : 0,
            ':tax_category_id' => $tax_category_id
        ]);
        
        $expense_id = $conn->lastInsertId();
        
        // If expense is on a prepaid card, check if balance is now 0 and suggest archiving
        $suggest_archive = false;
        if ($prepaid_card_id) {
            // Get card balance - use GROUP BY to fix sql_mode=only_full_group_by issue
            $stmt_balance = $conn->prepare("
                SELECT 
                    COALESCE(pc.nominal_amount, 0) as nominal_amount,
                    COALESCE(SUM(e.amount), 0) as total_expenses
                FROM prepaid_cards pc
                LEFT JOIN expenses e ON e.prepaid_card_id = pc.id
                WHERE pc.id = :prepaid_card_id
                GROUP BY pc.id, pc.nominal_amount
            ");
            $stmt_balance->execute([':prepaid_card_id' => $prepaid_card_id]);
            $balance_result = $stmt_balance->fetch();
            
            if ($balance_result) {
                $nominal_amount = floatval($balance_result['nominal_amount']);
                $total_expenses = floatval($balance_result['total_expenses']);
                $remaining_balance = $nominal_amount + $total_expenses;
                
                // Suggest archiving if balance is 0 or less
                if ($remaining_balance <= 0) {
                    $suggest_archive = true;
                }
            }
        }
        
        echo json_encode([
            'success' => true, 
            'id' => $expense_id,
            'suggest_archive_prepaid_card' => $suggest_archive ? $prepaid_card_id : null
        ]);
        break;
        
    case 'PUT':
        // Update existing expense
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Expense ID is required']);
            break;
        }
        
        $expense_id = (int)$data['id'];
        
        // Verify expense belongs to user OR is on a shared account/prepaid card user has write access to OR user owns the account/card
        // Also get the original prepaid_card_id to check if it was archived
        $stmt = $conn->prepare("
            SELECT e.id, e.account_id, e.prepaid_card_id, pc.archived as original_card_archived
            FROM expenses e
            LEFT JOIN accounts a ON e.account_id = a.id
            LEFT JOIN prepaid_cards pc ON e.prepaid_card_id = pc.id
            WHERE e.id = :id 
            AND (
                e.user_id = :user_id_owner 
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = e.account_id 
                    AND ash.shared_with_user_id = :user_id_shared
                    AND ash.sharing_mode = 'write'
                )
                OR a.user_id = :user_id_account_owner
                OR EXISTS (
                    SELECT 1 FROM prepaid_card_shares pcs 
                    WHERE pcs.prepaid_card_id = e.prepaid_card_id 
                    AND pcs.shared_with_user_id = :user_id_shared_prepaid
                )
                OR pc.user_id = :user_id_prepaid_card_owner
            )
        ");
        $stmt->execute([
            ':id' => $expense_id, 
            ':user_id_owner' => $user_id,
            ':user_id_shared' => $user_id,
            ':user_id_account_owner' => $user_id,
            ':user_id_shared_prepaid' => $user_id,
            ':user_id_prepaid_card_owner' => $user_id
        ]);
        $expense = $stmt->fetch();
        
        if (!$expense) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found or you do not have write access']);
            break;
        }
        
        // Save original prepaid card info for later check
        $original_prepaid_card_id = $expense['prepaid_card_id'];
        $original_card_archived = $expense['original_card_archived'] == 1;
        
        // Require either account_id or prepaid_card_id
        if (!isset($data['account_id']) && !isset($data['prepaid_card_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Either account_id or prepaid_card_id is required']);
            break;
        }
        
        $required = ['date', 'description', 'amount', 'currency'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        $account_id = null;
        $prepaid_card_id = null;
        
        if (isset($data['account_id'])) {
            // Verify account belongs to user OR user has write access via sharing
            $stmt = $conn->prepare("
                SELECT 
                    a.id, 
                    at.code as type_code,
                    CASE 
                        WHEN a.user_id = :user_id_check2 THEN 'owned'
                        WHEN EXISTS (
                            SELECT 1 FROM account_shares ash 
                            WHERE ash.account_id = a.id 
                            AND ash.shared_with_user_id = :user_id_shared_check2
                            AND ash.sharing_mode = 'write'
                        ) THEN 'shared_write'
                        ELSE 'no_access'
                    END as access_type
                FROM accounts a
                JOIN account_types at ON a.type_id = at.id
                WHERE a.id = :account_id 
                AND (
                    a.user_id = :user_id_owner2 
                    OR EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = a.id 
                        AND ash.shared_with_user_id = :user_id_shared2
                        AND ash.sharing_mode = 'write'
                    )
                )
            ");
            $stmt->execute([
                ':account_id' => $data['account_id'], 
                ':user_id_check2' => $user_id,
                ':user_id_shared_check2' => $user_id,
                ':user_id_owner2' => $user_id,
                ':user_id_shared2' => $user_id
            ]);
            $account = $stmt->fetch();
            
            if (!$account) {
                http_response_code(404);
                echo json_encode(['error' => 'Account not found or you do not have write access']);
                break;
            }
            
            if ($account['type_code'] === 'SECURITIES') {
                http_response_code(400);
                echo json_encode(['error' => 'Cannot update expenses on securities accounts. Use securities purchases/sales instead.']);
                break;
            }
            
            if ($account['access_type'] === 'no_access') {
                http_response_code(403);
                echo json_encode(['error' => 'You do not have write access to this account']);
                break;
            }
            
            $account_id = $data['account_id'];
        } else if (isset($data['prepaid_card_id'])) {
            // Verify prepaid card belongs to user OR is shared with user and is enabled
            $stmt = $conn->prepare("
                SELECT 
                    pc.id,
                    CASE 
                        WHEN pc.user_id = :user_id_check THEN 'owned'
                        WHEN EXISTS (
                            SELECT 1 FROM prepaid_card_shares pcs 
                            WHERE pcs.prepaid_card_id = pc.id 
                            AND pcs.shared_with_user_id = :user_id_shared_check
                        ) THEN 'shared'
                        ELSE 'no_access'
                    END as access_type
                FROM prepaid_cards pc
                WHERE pc.id = :prepaid_card_id 
                AND pc.enabled = 1
                AND (
                    pc.user_id = :user_id_owner 
                    OR EXISTS (
                        SELECT 1 FROM prepaid_card_shares pcs 
                        WHERE pcs.prepaid_card_id = pc.id 
                        AND pcs.shared_with_user_id = :user_id_shared
                    )
                )
            ");
            $stmt->execute([
                ':prepaid_card_id' => $data['prepaid_card_id'],
                ':user_id_check' => $user_id,
                ':user_id_shared_check' => $user_id,
                ':user_id_owner' => $user_id,
                ':user_id_shared' => $user_id
            ]);
            $prepaid_card = $stmt->fetch();
            
            if (!$prepaid_card) {
                http_response_code(404);
                echo json_encode(['error' => 'Prepaid card not found, not enabled, or you do not have access']);
                break;
            }
            
            if ($prepaid_card['access_type'] === 'no_access') {
                http_response_code(403);
                echo json_encode(['error' => 'You do not have access to this prepaid card']);
                break;
            }
            
            $prepaid_card_id = $data['prepaid_card_id'];
        }
        
        // Validate tax category if tax deductible
        $tax_category_id = null;
        if (!empty($data['is_tax_deductible']) && isset($data['tax_category_id'])) {
            $stmt = $conn->prepare("SELECT id FROM tax_deduction_categories WHERE id = :category_id");
            $stmt->execute([':category_id' => $data['tax_category_id']]);
            if ($stmt->fetch()) {
                $tax_category_id = $data['tax_category_id'];
            }
        }
        
        $stmt = $conn->prepare("
            UPDATE expenses 
            SET account_id = :account_id, 
                prepaid_card_id = :prepaid_card_id,
                date = :date, 
                description = :description, 
                amount = :amount, 
                currency = :currency, 
                is_tax_deductible = :is_tax_deductible, 
                tax_category_id = :tax_category_id
            WHERE id = :id 
            AND (
                user_id = :user_id_owner_update 
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = expenses.account_id 
                    AND ash.shared_with_user_id = :user_id_shared_update
                    AND ash.sharing_mode = 'write'
                )
                OR EXISTS (
                    SELECT 1 FROM accounts a 
                    WHERE a.id = expenses.account_id 
                    AND a.user_id = :user_id_account_owner_update
                )
                OR EXISTS (
                    SELECT 1 FROM prepaid_card_shares pcs 
                    WHERE pcs.prepaid_card_id = expenses.prepaid_card_id 
                    AND pcs.shared_with_user_id = :user_id_shared_prepaid_update
                )
                OR EXISTS (
                    SELECT 1 FROM prepaid_cards pc 
                    WHERE pc.id = expenses.prepaid_card_id 
                    AND pc.user_id = :user_id_prepaid_card_owner_update
                )
            )
        ");
        
        $stmt->execute([
            ':id' => $expense_id,
            ':user_id_owner_update' => $user_id,
            ':user_id_shared_update' => $user_id,
            ':user_id_account_owner_update' => $user_id,
            ':user_id_shared_prepaid_update' => $user_id,
            ':user_id_prepaid_card_owner_update' => $user_id,
            ':account_id' => $account_id,
            ':prepaid_card_id' => $prepaid_card_id,
            ':date' => $data['date'],
            ':description' => $data['description'],
            ':amount' => $data['amount'],
            ':currency' => $data['currency'],
            ':is_tax_deductible' => !empty($data['is_tax_deductible']) ? 1 : 0,
            ':tax_category_id' => $tax_category_id
        ]);
        
        // Check if we should suggest restoring the prepaid card
        $suggest_restore = false;
        $card_to_check = null;
        
        // If expense was moved to a different prepaid card, check both the original and new one
        if ($prepaid_card_id && $prepaid_card_id == $original_prepaid_card_id && $original_card_archived) {
            // Same card, check if balance is now positive
            $card_to_check = $prepaid_card_id;
        } else if ($prepaid_card_id && $prepaid_card_id != $original_prepaid_card_id) {
            // Card changed, check the new card
            $card_to_check = $prepaid_card_id;
        } else if ($original_prepaid_card_id && !$prepaid_card_id) {
            // Card removed (moved to account), check original card
            $card_to_check = $original_prepaid_card_id;
        }
        
        if ($card_to_check) {
            // Check if card is archived and balance is now positive
            $stmt_check = $conn->prepare("
                SELECT 
                    pc.archived,
                    COALESCE(pc.nominal_amount, 0) as nominal_amount,
                    COALESCE(SUM(e.amount), 0) as total_expenses
                FROM prepaid_cards pc
                LEFT JOIN expenses e ON e.prepaid_card_id = pc.id
                WHERE pc.id = :prepaid_card_id
                GROUP BY pc.id, pc.nominal_amount, pc.archived
            ");
            $stmt_check->execute([':prepaid_card_id' => $card_to_check]);
            $card_check = $stmt_check->fetch();
            
            if ($card_check && $card_check['archived'] == 1) {
                $nominal_amount = floatval($card_check['nominal_amount']);
                $total_expenses = floatval($card_check['total_expenses']);
                $remaining_balance = $nominal_amount + $total_expenses;
                
                // Suggest restore if balance is positive
                if ($remaining_balance > 0) {
                    $suggest_restore = true;
                }
            }
        }
        
        echo json_encode([
            'success' => true, 
            'id' => $expense_id,
            'suggest_restore_prepaid_card' => $suggest_restore ? $card_to_check : null
        ]);
        break;
        
    case 'DELETE':
        // Delete expense
        $expense_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$expense_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Expense ID is required']);
            break;
        }
        
        // Verify expense belongs to user OR is on a shared account/prepaid card user has write access to OR user owns the account/card
        // Also get prepaid_card_id and archived status before deleting
        $stmt = $conn->prepare("
            SELECT e.id, e.prepaid_card_id, pc.archived as card_archived
            FROM expenses e
            LEFT JOIN accounts a ON e.account_id = a.id
            LEFT JOIN prepaid_cards pc ON e.prepaid_card_id = pc.id
            WHERE e.id = :id 
            AND (
                e.user_id = :user_id_owner 
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = e.account_id 
                    AND ash.shared_with_user_id = :user_id_shared
                    AND ash.sharing_mode = 'write'
                )
                OR a.user_id = :user_id_account_owner_delete_check
                OR EXISTS (
                    SELECT 1 FROM prepaid_card_shares pcs 
                    WHERE pcs.prepaid_card_id = e.prepaid_card_id 
                    AND pcs.shared_with_user_id = :user_id_shared_prepaid_delete
                )
                OR pc.user_id = :user_id_prepaid_card_owner_delete_check
            )
        ");
        $stmt->execute([
            ':id' => $expense_id, 
            ':user_id_owner' => $user_id,
            ':user_id_shared' => $user_id,
            ':user_id_account_owner_delete_check' => $user_id,
            ':user_id_shared_prepaid_delete' => $user_id,
            ':user_id_prepaid_card_owner_delete_check' => $user_id
        ]);
        $expense_to_delete = $stmt->fetch();
        if (!$expense_to_delete) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found or you do not have write access']);
            break;
        }
        
        // Save prepaid card info before deleting
        $deleted_prepaid_card_id = $expense_to_delete['prepaid_card_id'];
        $deleted_card_archived = $expense_to_delete['card_archived'] == 1;
        
        // Delete all attachment files before deleting expense
        $stmt = $conn->prepare("
            SELECT file_path 
            FROM expense_attachments 
            WHERE expense_id = :expense_id AND file_path IS NOT NULL
        ");
        $stmt->execute([':expense_id' => $expense_id]);
        $attachments = $stmt->fetchAll();
        
        foreach ($attachments as $attachment) {
            if ($attachment['file_path']) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
            }
        }
        
        // Delete the expense directory if it exists
        $expense_dir = UPLOAD_DIR . 'expenses/' . $expense_id . '/';
        if (is_dir($expense_dir)) {
            // Remove all files in directory
            $files = glob($expense_dir . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
            // Remove directory
            rmdir($expense_dir);
        }
        
        // Delete expense (attachments will be deleted via CASCADE if foreign key is set up)
        $stmt = $conn->prepare("
            DELETE e FROM expenses e
            LEFT JOIN accounts a ON e.account_id = a.id
            LEFT JOIN prepaid_cards pc ON e.prepaid_card_id = pc.id
            WHERE e.id = :id 
            AND (
                e.user_id = :user_id_owner_delete 
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = e.account_id 
                    AND ash.shared_with_user_id = :user_id_shared_delete
                    AND ash.sharing_mode = 'write'
                )
                OR a.user_id = :user_id_account_owner_delete
                OR EXISTS (
                    SELECT 1 FROM prepaid_card_shares pcs 
                    WHERE pcs.prepaid_card_id = e.prepaid_card_id 
                    AND pcs.shared_with_user_id = :user_id_shared_prepaid_delete_final
                )
                OR pc.user_id = :user_id_prepaid_card_owner_delete
            )
        ");
        $stmt->execute([
            ':id' => $expense_id, 
            ':user_id_owner_delete' => $user_id,
            ':user_id_shared_delete' => $user_id,
            ':user_id_account_owner_delete' => $user_id,
            ':user_id_shared_prepaid_delete_final' => $user_id,
            ':user_id_prepaid_card_owner_delete' => $user_id
        ]);
        
        // Check if we should suggest restoring the prepaid card
        $suggest_restore = false;
        if ($deleted_prepaid_card_id && $deleted_card_archived) {
            // Check if balance is now positive after deletion
            $stmt_check = $conn->prepare("
                SELECT 
                    COALESCE(pc.nominal_amount, 0) as nominal_amount,
                    COALESCE(SUM(e.amount), 0) as total_expenses
                FROM prepaid_cards pc
                LEFT JOIN expenses e ON e.prepaid_card_id = pc.id
                WHERE pc.id = :prepaid_card_id
                GROUP BY pc.id, pc.nominal_amount
            ");
            $stmt_check->execute([':prepaid_card_id' => $deleted_prepaid_card_id]);
            $card_check = $stmt_check->fetch();
            
            if ($card_check) {
                $nominal_amount = floatval($card_check['nominal_amount']);
                $total_expenses = floatval($card_check['total_expenses']);
                $remaining_balance = $nominal_amount + $total_expenses;
                
                // Suggest restore if balance is positive
                if ($remaining_balance > 0) {
                    $suggest_restore = true;
                }
            }
        }
        
        echo json_encode([
            'success' => true,
            'suggest_restore_prepaid_card' => $suggest_restore ? $deleted_prepaid_card_id : null
        ]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

