<?php
/**
 * Session Debug Logs API (Admin only)
 * Returns session debug logs for troubleshooting session issues
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

try {
    requireAdmin();
    
    // Check if debug is enabled
    if (!defined('SESSION_DEBUG_ENABLED') || !SESSION_DEBUG_ENABLED) {
        http_response_code(400);
        echo json_encode([
            'error' => 'Session debug is not enabled',
            'message' => 'Set SESSION_DEBUG_ENABLED to true in config/config.php to enable debug logging'
        ]);
        exit;
    }
    
    $logFile = SESSION_DEBUG_LOG_FILE;
    
    if (!file_exists($logFile)) {
        echo json_encode([
            'enabled' => true,
            'log_file' => $logFile,
            'logs' => [],
            'message' => 'No logs found yet. Logs will appear here once session debug is active.'
        ]);
        exit;
    }
    
    // Read log file
    $logs = [];
    $handle = fopen($logFile, 'r');
    if ($handle) {
        while (($line = fgets($handle)) !== false) {
            $line = trim($line);
            if (empty($line)) {
                continue;
            }
            $logEntry = json_decode($line, true);
            if ($logEntry) {
                $logs[] = $logEntry;
            }
        }
        fclose($handle);
    }
    
    // Get query parameters
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 100;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    $event = isset($_GET['event']) ? $_GET['event'] : null;
    $userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
    
    // Filter logs
    $filteredLogs = $logs;
    if ($event) {
        $filteredLogs = array_filter($filteredLogs, function($log) use ($event) {
            return isset($log['event']) && $log['event'] === $event;
        });
    }
    if ($userId) {
        $filteredLogs = array_filter($filteredLogs, function($log) use ($userId) {
            return isset($log['user_id']) && $log['user_id'] == $userId;
        });
    }
    
    // Reverse to show newest first
    $filteredLogs = array_reverse($filteredLogs);
    
    // Apply pagination
    $total = count($filteredLogs);
    $paginatedLogs = array_slice($filteredLogs, $offset, $limit);
    
    // Get log file stats
    $fileSize = filesize($logFile);
    $fileModified = filemtime($logFile);
    
    echo json_encode([
        'enabled' => true,
        'log_file' => $logFile,
        'file_size' => $fileSize,
        'file_modified' => date('Y-m-d H:i:s', $fileModified),
        'total_logs' => $total,
        'showing' => count($paginatedLogs),
        'offset' => $offset,
        'limit' => $limit,
        'filters' => [
            'event' => $event,
            'user_id' => $userId
        ],
        'logs' => array_values($paginatedLogs)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
