<?php
/**
 * Google OAuth Callback Handler
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

if (!isset($_GET['code'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Authorization code not provided']);
    exit;
}

$code = $_GET['code'];

// Exchange authorization code for access token
$token_url = 'https://oauth2.googleapis.com/token';
$token_data = [
    'code' => $code,
    'client_id' => GOOGLE_CLIENT_ID,
    'client_secret' => GOOGLE_CLIENT_SECRET,
    'redirect_uri' => GOOGLE_REDIRECT_URI,
    'grant_type' => 'authorization_code'
];

$ch = curl_init($token_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($token_data));
$token_response = curl_exec($ch);
curl_close($ch);

$token_info = json_decode($token_response, true);

if (!isset($token_info['access_token'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Failed to obtain access token']);
    exit;
}

// Get user info from Google
$user_info_url = 'https://www.googleapis.com/oauth2/v2/userinfo?access_token=' . $token_info['access_token'];
$user_info_response = file_get_contents($user_info_url);
$user_info = json_decode($user_info_response, true);

if (!isset($user_info['id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Failed to get user information']);
    exit;
}

// Create or update user in database
$db = new Database();
$conn = $db->getConnection();

// Check if this is the first user (will become admin)
$stmt = $conn->prepare("SELECT COUNT(*) as user_count FROM users");
$stmt->execute();
$count_result = $stmt->fetch();
$is_first_user = ($count_result['user_count'] == 0);

// Check if user already exists
$stmt = $conn->prepare("SELECT id, is_blocked FROM users WHERE google_id = :google_id");
$stmt->execute([':google_id' => $user_info['id']]);
$existing_user = $stmt->fetch();

// Check if user is blocked
if ($existing_user && $existing_user['is_blocked']) {
    http_response_code(403);
    echo json_encode(['error' => 'Account is blocked. Please contact administrator.']);
    exit;
}

// Prepare picture URL - truncate if too long (temporary fix until migration is applied)
$picture_url = $user_info['picture'] ?? null;
if ($picture_url && strlen($picture_url) > 500) {
    // Log warning but truncate to prevent error
    error_log("Warning: picture_url truncated from " . strlen($picture_url) . " to 500 characters for user " . $user_info['email']);
    $picture_url = substr($picture_url, 0, 500);
}

if ($existing_user) {
    // Update existing user
    $stmt = $conn->prepare("
        UPDATE users 
        SET email = :email,
            name = :name,
            picture_url = :picture_url,
            updated_at = CURRENT_TIMESTAMP
        WHERE google_id = :google_id
    ");
    $stmt->execute([
        ':google_id' => $user_info['id'],
        ':email' => $user_info['email'],
        ':name' => $user_info['name'] ?? null,
        ':picture_url' => $picture_url
    ]);
    $user_id = $existing_user['id'];
} else {
    // Insert new user
    $stmt = $conn->prepare("
        INSERT INTO users (google_id, email, name, picture_url, is_admin, first_login_at)
        VALUES (:google_id, :email, :name, :picture_url, :is_admin, CURRENT_TIMESTAMP)
    ");
    $stmt->execute([
        ':google_id' => $user_info['id'],
        ':email' => $user_info['email'],
        ':name' => $user_info['name'] ?? null,
        ':picture_url' => $picture_url,
        ':is_admin' => $is_first_user ? 1 : 0
    ]);
    $user_id = $conn->lastInsertId();
    
    // Create default "No Account" for new users
    // Get the "OTHER" account type ID
    $stmt = $conn->prepare("SELECT id FROM account_types WHERE code = 'OTHER'");
    $stmt->execute();
    $account_type = $stmt->fetch();
    
    if ($account_type) {
        // Get user's default currency or use EUR
        $default_currency = 'EUR';
        
        // Create the "No Account" account
        $stmt = $conn->prepare("
            INSERT INTO accounts (user_id, name, type_id, base_currency, opening_date)
            VALUES (:user_id, 'No Account', :type_id, :base_currency, CURDATE())
        ");
        $stmt->execute([
            ':user_id' => $user_id,
            ':type_id' => $account_type['id'],
            ':base_currency' => $default_currency
        ]);
    }
    
    // Convert pending invitations to active shares for this new user
    $stmt = $conn->prepare("
        SELECT id, account_id, owner_user_id, sharing_mode
        FROM account_share_invitations
        WHERE recipient_email = :email
    ");
    $stmt->execute([':email' => $user_info['email']]);
    $pending_invitations = $stmt->fetchAll();
    
    foreach ($pending_invitations as $invitation) {
        // Check if share already exists (shouldn't, but just in case)
        $stmt = $conn->prepare("
            SELECT id FROM account_shares 
            WHERE account_id = :account_id AND shared_with_user_id = :shared_with_user_id
        ");
        $stmt->execute([
            ':account_id' => $invitation['account_id'],
            ':shared_with_user_id' => $user_id
        ]);
        
        if (!$stmt->fetch()) {
            // Create active share
            $stmt = $conn->prepare("
                INSERT INTO account_shares (account_id, owner_user_id, shared_with_user_id, sharing_mode)
                VALUES (:account_id, :owner_user_id, :shared_with_user_id, :sharing_mode)
            ");
            $stmt->execute([
                ':account_id' => $invitation['account_id'],
                ':owner_user_id' => $invitation['owner_user_id'],
                ':shared_with_user_id' => $user_id,
                ':sharing_mode' => $invitation['sharing_mode']
            ]);
        }
        
        // Delete the invitation
        $stmt = $conn->prepare("DELETE FROM account_share_invitations WHERE id = :id");
        $stmt->execute([':id' => $invitation['id']]);
    }
}

// Get user data
$stmt = $conn->prepare("SELECT id FROM users WHERE google_id = :google_id");
$stmt->execute([':google_id' => $user_info['id']]);
$user = $stmt->fetch();

// Set session
$_SESSION['user_id'] = $user['id'];
$_SESSION['google_id'] = $user_info['id'];
$_SESSION['email'] = $user_info['email'];
$_SESSION['name'] = $user_info['name'] ?? null;
// Store first_login_at in session (per device, not per user)
// This ensures each device has its own 180-day period
$_SESSION['first_login_at'] = time();

// Set session cookie with proper expiration
// Calculate expiration: 7 days from now, but not more than 180 days from first login
$now = time();
$first_login_at = $_SESSION['first_login_at'];
$absolute_expiration = $first_login_at + (180 * 24 * 60 * 60); // 180 days from first login
$cookie_expiration = min($now + (7 * 24 * 60 * 60), $absolute_expiration); // 7 days or absolute expiration

// Build Set-Cookie header manually to include SameSite attribute
$cookie_params = session_get_cookie_params();
$cookie_value = session_id();
$cookie_name = session_name();
$cookie_string = sprintf(
    '%s=%s; expires=%s; path=%s; %s%s; SameSite=%s',
    urlencode($cookie_name),
    urlencode($cookie_value),
    gmdate('D, d-M-Y H:i:s T', $cookie_expiration),
    $cookie_params['path'],
    $cookie_params['domain'] ? 'domain=' . $cookie_params['domain'] . '; ' : '',
    $cookie_params['httponly'] ? 'HttpOnly; ' : '',
    $cookie_params['samesite'] ?? 'Lax'
);

if ($cookie_params['secure']) {
    $cookie_string .= '; Secure';
}

header('Set-Cookie: ' . $cookie_string, false);

// Save session ID before closing session
$currentSessionId = session_id();
$sessionSavePath = session_save_path();
if (empty($sessionSavePath)) {
    $sessionSavePath = sys_get_temp_dir();
}
$sessionFilePath = $sessionSavePath . '/sess_' . $currentSessionId;
$sessionKeys = array_keys($_SESSION);

// Force session write before redirect to ensure data is saved
// PHP normally writes sessions at script end, but redirects can interfere
// Note: After session_write_close(), we can't access $_SESSION anymore
// But the session data should be written to disk
if (function_exists('session_write_close')) {
    session_write_close();
}

// Verify session was written correctly AFTER write_close
$sessionFileExists = file_exists($sessionFilePath);
$sessionFileSize = $sessionFileExists ? filesize($sessionFilePath) : 0;
$sessionFileWritable = $sessionFileExists ? is_writable($sessionFilePath) : false;

// Read session file content to verify data was written
$sessionFileContent = null;
if ($sessionFileExists && $sessionFileSize > 0) {
    $sessionFileContent = @file_get_contents($sessionFilePath);
    if ($sessionFileContent !== false) {
        $sessionFileContent = 'has_content_' . strlen($sessionFileContent) . '_bytes';
    }
} else {
    $sessionFileContent = 'empty_or_not_readable';
}

// Log session creation
logSessionDebug('session_created_login', [
    'reason' => 'Google OAuth login successful',
    'user_id' => $user['id'],
    'email' => $user_info['email'],
    'first_login_at' => date('Y-m-d H:i:s', $first_login_at),
    'cookie_expiration' => date('Y-m-d H:i:s', $cookie_expiration),
    'absolute_expiration' => date('Y-m-d H:i:s', $absolute_expiration),
    'days_until_cookie_expiration' => round(($cookie_expiration - $now) / (24 * 60 * 60), 2),
    'days_until_absolute_expiration' => round(($absolute_expiration - $now) / (24 * 60 * 60), 2),
    'cookie_secure' => $cookie_params['secure'],
    'cookie_samesite' => $cookie_params['samesite'] ?? 'Lax',
    'cookie_domain' => $cookie_params['domain'] ?: 'current',
    'is_new_user' => !$existing_user,
    'session_id' => session_id(),
    'session_keys' => $sessionKeys,
    'session_count' => count($_SESSION),
    'session_file_exists' => $sessionFileExists,
    'session_file_size' => $sessionFileSize,
    'session_file_writable' => $sessionFileWritable,
    'session_file_content' => $sessionFileContent,
    'session_save_path' => $sessionSavePath,
    'session_save_path_writable' => is_writable($sessionSavePath),
    'session_write_close_called' => true
]);

// Redirect to main application (use root, .htaccess will handle index.html)
header('Location: ' . rtrim(APP_URL, '/') . '/');
exit;

