<?php
/**
 * Account Sharing API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';
require_once __DIR__ . '/../middleware/notifications_helper.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? null;
    $account_id = $_GET['account_id'] ?? null;

    switch ($method) {
    case 'GET':
        if ($account_id) {
            // Get shares for a specific account (owner view)
            $account_id = (int)$account_id;
            
            // Verify user owns the account
            $stmt = $conn->prepare("SELECT id FROM accounts WHERE id = :account_id AND user_id = :user_id");
            $stmt->execute([':account_id' => $account_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Account not found']);
                break;
            }
            
            // Get active shares
            $stmt = $conn->prepare("
                SELECT 
                    ash.id,
                    ash.shared_with_user_id,
                    ash.sharing_mode,
                    u.email,
                    u.name,
                    ash.created_at
                FROM account_shares ash
                JOIN users u ON ash.shared_with_user_id = u.id
                WHERE ash.account_id = :account_id AND ash.owner_user_id = :user_id
                ORDER BY ash.created_at DESC
            ");
            $stmt->execute([':account_id' => $account_id, ':user_id' => $user_id]);
            $shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get pending invitations
            $stmt = $conn->prepare("
                SELECT 
                    id,
                    recipient_email,
                    sharing_mode,
                    created_at
                FROM account_share_invitations
                WHERE account_id = :account_id AND owner_user_id = :user_id
                ORDER BY created_at DESC
            ");
            $stmt->execute([':account_id' => $account_id, ':user_id' => $user_id]);
            $invitations = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'shares' => $shares,
                'invitations' => $invitations
            ]);
        } else {
            // Get previously shared users for dropdown
            // Includes users the current user has shared with AND users who have shared with the current user
            // This includes users from both account shares and prepaid card shares
            $stmt = $conn->prepare("
                SELECT DISTINCT
                    u.id,
                    u.email,
                    u.name
                FROM users u
                WHERE u.id IN (
                    -- Users the current user has shared accounts with
                    SELECT DISTINCT shared_with_user_id 
                    FROM account_shares 
                    WHERE owner_user_id = :user_id1
                    UNION
                    -- Users who have shared accounts with the current user
                    SELECT DISTINCT owner_user_id 
                    FROM account_shares 
                    WHERE shared_with_user_id = :user_id2
                    UNION
                    -- Users the current user has shared prepaid cards with
                    SELECT DISTINCT shared_with_user_id 
                    FROM prepaid_card_shares 
                    WHERE owner_user_id = :user_id3
                    UNION
                    -- Users who have shared prepaid cards with the current user
                    SELECT DISTINCT owner_user_id 
                    FROM prepaid_card_shares 
                    WHERE shared_with_user_id = :user_id4
                )
                ORDER BY u.email
            ");
            $stmt->execute([
                ':user_id1' => $user_id, 
                ':user_id2' => $user_id,
                ':user_id3' => $user_id,
                ':user_id4' => $user_id
            ]);
            $previously_shared_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode($previously_shared_users);
        }
        break;
        
    case 'POST':
        // Share account with user
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['account_id']) || !isset($data['recipient_email']) || !isset($data['sharing_mode'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields: account_id, recipient_email, sharing_mode']);
            break;
        }
        
        $account_id = (int)$data['account_id'];
        $recipient_email = trim($data['recipient_email']);
        $sharing_mode = $data['sharing_mode'];
        
        if (!in_array($sharing_mode, ['read-only', 'write'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid sharing_mode. Must be "read-only" or "write"']);
            break;
        }
        
        // Verify user owns the account
        $stmt = $conn->prepare("SELECT id, name FROM accounts WHERE id = :account_id AND user_id = :user_id");
        $stmt->execute([':account_id' => $account_id, ':user_id' => $user_id]);
        $account = $stmt->fetch();
        if (!$account) {
            http_response_code(404);
            echo json_encode(['error' => 'Account not found']);
            break;
        }
        
        // Prevent sharing "No Account"
        if ($account['name'] === 'No Account') {
            http_response_code(400);
            echo json_encode(['error' => 'Cannot share "No Account"']);
            break;
        }
        
        // Prevent sharing with yourself
        $stmt = $conn->prepare("SELECT id, email FROM users WHERE id = :user_id");
        $stmt->execute([':user_id' => $user_id]);
        $owner = $stmt->fetch();
        if (strtolower($owner['email']) === strtolower($recipient_email)) {
            http_response_code(400);
            echo json_encode(['error' => 'Cannot share account with yourself']);
            break;
        }
        
        // Check if recipient user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = :email");
        $stmt->execute([':email' => $recipient_email]);
        $recipient_user = $stmt->fetch();
        
        if ($recipient_user) {
            // User exists - create active share
            $recipient_user_id = $recipient_user['id'];
            
            // Check if share already exists
            $stmt = $conn->prepare("SELECT id FROM account_shares WHERE account_id = :account_id AND shared_with_user_id = :shared_with_user_id");
            $stmt->execute([':account_id' => $account_id, ':shared_with_user_id' => $recipient_user_id]);
            if ($stmt->fetch()) {
                http_response_code(400);
                echo json_encode(['error' => 'Account is already shared with this user']);
                break;
            }
            
            // Create share
            $stmt = $conn->prepare("
                INSERT INTO account_shares (account_id, owner_user_id, shared_with_user_id, sharing_mode)
                VALUES (:account_id, :owner_user_id, :shared_with_user_id, :sharing_mode)
            ");
            $stmt->execute([
                ':account_id' => $account_id,
                ':owner_user_id' => $user_id,
                ':shared_with_user_id' => $recipient_user_id,
                ':sharing_mode' => $sharing_mode
            ]);
            
            // Get owner name for notification
            $stmt = $conn->prepare("SELECT name, email FROM users WHERE id = :user_id");
            $stmt->execute([':user_id' => $user_id]);
            $owner = $stmt->fetch();
            $owner_name = $owner['name'] ?: $owner['email'];
            
            // Create notification for recipient
            $account_name = $account['name'];
            $mode_text = $sharing_mode === 'write' ? 'scrittura' : 'sola lettura';
            createNotification(
                $conn,
                $recipient_user_id,
                'share_created',
                'Account condiviso',
                $owner_name . ' ha condiviso l\'account "' . $account_name . '" con te in modalità ' . $mode_text,
                'account',
                $account_id
            );
            
            echo json_encode([
                'success' => true,
                'message' => 'Account shared successfully',
                'user_exists' => true
            ]);
        } else {
            // User doesn't exist - create invitation
            // Check if invitation already exists
            $stmt = $conn->prepare("SELECT id FROM account_share_invitations WHERE account_id = :account_id AND recipient_email = :recipient_email");
            $stmt->execute([':account_id' => $account_id, ':recipient_email' => $recipient_email]);
            if ($stmt->fetch()) {
                http_response_code(400);
                echo json_encode(['error' => 'Invitation already sent to this email']);
                break;
            }
            
            // Create invitation
            $stmt = $conn->prepare("
                INSERT INTO account_share_invitations (account_id, owner_user_id, recipient_email, sharing_mode)
                VALUES (:account_id, :owner_user_id, :recipient_email, :sharing_mode)
            ");
            $stmt->execute([
                ':account_id' => $account_id,
                ':owner_user_id' => $user_id,
                ':recipient_email' => $recipient_email,
                ':sharing_mode' => $sharing_mode
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Invitation sent. Account will be shared when user registers.',
                'user_exists' => false
            ]);
        }
        break;
        
    case 'PUT':
        // Update sharing mode (for both active shares and pending invitations)
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['sharing_mode'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required field: sharing_mode']);
            break;
        }
        
        $sharing_mode = $data['sharing_mode'];
        
        if (!in_array($sharing_mode, ['read-only', 'write'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid sharing_mode. Must be "read-only" or "write"']);
            break;
        }
        
        if (isset($data['share_id'])) {
            // Update active share
            $share_id = (int)$data['share_id'];
            
            // Verify user owns the account being shared
            $stmt = $conn->prepare("
                SELECT ash.id 
                FROM account_shares ash
                JOIN accounts a ON ash.account_id = a.id
                WHERE ash.id = :share_id AND a.user_id = :user_id
            ");
            $stmt->execute([':share_id' => $share_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Share not found']);
                break;
            }
            
            // Get share details for notification
            $stmt = $conn->prepare("
                SELECT ash.account_id, ash.shared_with_user_id, a.name as account_name
                FROM account_shares ash
                JOIN accounts a ON ash.account_id = a.id
                WHERE ash.id = :share_id
            ");
            $stmt->execute([':share_id' => $share_id]);
            $share_details = $stmt->fetch();
            
            // Update sharing mode
            $stmt = $conn->prepare("UPDATE account_shares SET sharing_mode = :sharing_mode WHERE id = :share_id");
            $stmt->execute([':sharing_mode' => $sharing_mode, ':share_id' => $share_id]);
            
            // Create notification for recipient about mode change
            if ($share_details) {
                $stmt = $conn->prepare("SELECT name, email FROM users WHERE id = :user_id");
                $stmt->execute([':user_id' => $user_id]);
                $owner = $stmt->fetch();
                $owner_name = $owner['name'] ?: $owner['email'];
                
                $mode_text = $sharing_mode === 'write' ? 'scrittura' : 'sola lettura';
                createNotification(
                    $conn,
                    $share_details['shared_with_user_id'],
                    'share_updated',
                    'Modalità condivisione modificata',
                    $owner_name . ' ha modificato la modalità di condivisione dell\'account "' . $share_details['account_name'] . '" in ' . $mode_text,
                    'account',
                    $share_details['account_id']
                );
            }
            
            echo json_encode(['success' => true, 'message' => 'Sharing mode updated']);
        } elseif (isset($data['invitation_id'])) {
            // Update pending invitation
            $invitation_id = (int)$data['invitation_id'];
            
            // Verify user owns the invitation
            $stmt = $conn->prepare("
                SELECT asi.id 
                FROM account_share_invitations asi
                JOIN accounts a ON asi.account_id = a.id
                WHERE asi.id = :invitation_id AND a.user_id = :user_id
            ");
            $stmt->execute([':invitation_id' => $invitation_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Invitation not found']);
                break;
            }
            
            // Update sharing mode
            $stmt = $conn->prepare("UPDATE account_share_invitations SET sharing_mode = :sharing_mode WHERE id = :invitation_id");
            $stmt->execute([':sharing_mode' => $sharing_mode, ':invitation_id' => $invitation_id]);
            
            echo json_encode(['success' => true, 'message' => 'Invitation sharing mode updated']);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Missing share_id or invitation_id']);
        }
        break;
        
    case 'DELETE':
        // Remove sharing
        $share_id = $_GET['share_id'] ?? null;
        $invitation_id = $_GET['invitation_id'] ?? null;
        
        if ($share_id) {
            // Remove active share
            $share_id = (int)$share_id;
            
            // Verify user owns the account being shared OR is the recipient
            $stmt = $conn->prepare("
                SELECT ash.id, ash.account_id, ash.owner_user_id, ash.shared_with_user_id
                FROM account_shares ash
                WHERE ash.id = :share_id
            ");
            $stmt->execute([':share_id' => $share_id]);
            $share = $stmt->fetch();
            
            if (!$share) {
                http_response_code(404);
                echo json_encode(['error' => 'Share not found']);
                break;
            }
            
            // Check if user is owner or recipient
            if ($share['owner_user_id'] != $user_id && $share['shared_with_user_id'] != $user_id) {
                http_response_code(403);
                echo json_encode(['error' => 'Not authorized']);
                break;
            }
            
            // Get account name and user names for notification
            $stmt = $conn->prepare("SELECT name FROM accounts WHERE id = :account_id");
            $stmt->execute([':account_id' => $share['account_id']]);
            $account = $stmt->fetch();
            $account_name = $account ? $account['name'] : 'Account';
            
            $stmt = $conn->prepare("SELECT id, name, email FROM users WHERE id IN (:user_id1, :user_id2)");
            $stmt->execute([
                ':user_id1' => $share['owner_user_id'],
                ':user_id2' => $share['shared_with_user_id']
            ]);
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $owner_name = '';
            $recipient_name = '';
            foreach ($users as $u) {
                if ($u['id'] == $share['owner_user_id']) {
                    $owner_name = $u['name'] ?: $u['email'];
                }
                if ($u['id'] == $share['shared_with_user_id']) {
                    $recipient_name = $u['name'] ?: $u['email'];
                }
            }
            
            // Delete share
            $stmt = $conn->prepare("DELETE FROM account_shares WHERE id = :share_id");
            $stmt->execute([':share_id' => $share_id]);
            
            // Create notification for the other user (not the one who removed it)
            $notified_user_id = ($share['owner_user_id'] == $user_id) ? $share['shared_with_user_id'] : $share['owner_user_id'];
            $remover_name = ($share['owner_user_id'] == $user_id) ? $owner_name : $recipient_name;
            
            createNotification(
                $conn,
                $notified_user_id,
                'share_removed',
                'Condivisione rimossa',
                $remover_name . ' ha rimosso la condivisione dell\'account "' . $account_name . '"',
                'account',
                $share['account_id']
            );
            
            echo json_encode(['success' => true, 'message' => 'Sharing removed']);
        } elseif ($invitation_id) {
            // Remove invitation (owner only)
            $invitation_id = (int)$invitation_id;
            
            // Verify user owns the invitation
            $stmt = $conn->prepare("
                SELECT asi.id 
                FROM account_share_invitations asi
                JOIN accounts a ON asi.account_id = a.id
                WHERE asi.id = :invitation_id AND a.user_id = :user_id
            ");
            $stmt->execute([':invitation_id' => $invitation_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Invitation not found']);
                break;
            }
            
            // Delete invitation
            $stmt = $conn->prepare("DELETE FROM account_share_invitations WHERE id = :invitation_id");
            $stmt->execute([':invitation_id' => $invitation_id]);
            
            echo json_encode(['success' => true, 'message' => 'Invitation removed']);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Missing share_id or invitation_id']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
    error_log('Account sharing API error: ' . $e->getMessage());
}

