<?php
/**
 * Account Dashboard Visibility API
 * Toggle dashboard visibility for accounts per user
 */

ob_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

ini_set('display_errors', 0);
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();
    
    $method = $_SERVER['REQUEST_METHOD'];
    $account_id = isset($_GET['account_id']) ? intval($_GET['account_id']) : null;
    
    switch ($method) {
        case 'PUT':
            // Toggle dashboard visibility
            if (!$account_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Account ID is required']);
                break;
            }
            
            $data = json_decode(file_get_contents('php://input'), true);
            $show_in_dashboard = isset($data['show_in_dashboard']) 
                ? ($data['show_in_dashboard'] === true || $data['show_in_dashboard'] === '1' || $data['show_in_dashboard'] === 1)
                : true;
            
            // Verify user has access to this account (owned or shared)
            $stmt = $conn->prepare("
                SELECT a.id 
                FROM accounts a
                WHERE a.id = :account_id AND (
                    a.user_id = :user_id1
                    OR EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = :account_id2 
                        AND ash.shared_with_user_id = :user_id2
                    )
                )
            ");
            $stmt->execute([
                ':account_id' => $account_id,
                ':account_id2' => $account_id,
                ':user_id1' => $user_id,
                ':user_id2' => $user_id
            ]);
            
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Account not found or access denied']);
                break;
            }
            
            if ($show_in_dashboard) {
                // Add to dashboard
                $stmt = $conn->prepare("
                    INSERT INTO user_account_dashboard (user_id, account_id)
                    VALUES (:user_id, :account_id)
                    ON DUPLICATE KEY UPDATE user_id = user_id
                ");
                $stmt->execute([':user_id' => $user_id, ':account_id' => $account_id]);
            } else {
                // Remove from dashboard
                $stmt = $conn->prepare("
                    DELETE FROM user_account_dashboard 
                    WHERE user_id = :user_id AND account_id = :account_id
                ");
                $stmt->execute([':user_id' => $user_id, ':account_id' => $account_id]);
            }
            
            echo json_encode([
                'success' => true,
                'show_in_dashboard' => $show_in_dashboard
            ]);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
    error_log('Account dashboard API error: ' . $e->getMessage());
}
