<?php
/**
 * Account Balance API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $account_id = $_GET['account_id'] ?? null;

    if (!$account_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Account ID required']);
        exit;
    }

    // Verify account belongs to user OR is shared with user and get account type
    $stmt = $conn->prepare("
        SELECT 
            a.id, 
            a.name, 
            a.base_currency, 
            at.code as type_code,
            CASE 
                WHEN a.user_id = :user_id THEN 'owned'
                ELSE 'shared'
            END as access_type
        FROM accounts a
        JOIN account_types at ON a.type_id = at.id
        WHERE a.id = :account_id 
        AND (
            a.user_id = :user_id_owner 
            OR EXISTS (
                SELECT 1 FROM account_shares ash 
                WHERE ash.account_id = a.id 
                AND ash.shared_with_user_id = :user_id_shared
            )
        )
    ");
    $stmt->execute([
        ':account_id' => $account_id, 
        ':user_id' => $user_id,
        ':user_id_owner' => $user_id,
        ':user_id_shared' => $user_id
    ]);
    $account = $stmt->fetch();

    if (!$account) {
        http_response_code(404);
        echo json_encode(['error' => 'Account not found or you do not have access']);
        exit;
    }

    // "No Account" doesn't have a balance
    if ($account['name'] === 'No Account') {
        echo json_encode([
            'account_id' => $account_id,
            'currency' => $account['base_currency'],
            'type_code' => $account['type_code'],
            'balance' => 0,
            'expenses' => 0,
            'transfers_in' => 0,
            'transfers_out' => 0
        ]);
        exit;
    }

    if ($account['type_code'] === 'SECURITIES') {
        // For securities accounts, calculate total value by currency
        // purchase_price is the total amount, so we sum it directly
        // Sales subtract from the total
        $stmt = $conn->prepare("
            SELECT 
                currency,
                SUM(CASE 
                    WHEN transaction_type = 'PURCHASE' OR transaction_type IS NULL THEN purchase_price 
                    WHEN transaction_type = 'SALE' THEN -purchase_price 
                    ELSE 0 
                END) as total_value
            FROM securities
            WHERE account_id = :account_id
            GROUP BY currency
        ");
        $stmt->execute([':account_id' => $account_id]);
        $securities_by_currency = $stmt->fetchAll();
        
        // Ensure total_value is a float
        foreach ($securities_by_currency as &$sec) {
            $sec['total_value'] = floatval($sec['total_value']);
        }
        unset($sec); // Break reference
        
        echo json_encode([
            'account_id' => $account_id,
            'currency' => $account['base_currency'],
            'type_code' => $account['type_code'],
            'balance' => 0, // Will be calculated on frontend with currency conversion
            'securities_by_currency' => $securities_by_currency,
            'expenses' => 0,
            'transfers_in' => 0,
            'transfers_out' => 0
        ]);
    } else {
        // Calculate balance from expenses and transfers
        // Expenses are stored as negative values for expenses, positive for income
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(amount), 0) as total_expenses
            FROM expenses
            WHERE account_id = :account_id
        ");
        $stmt->execute([':account_id' => $account_id]);
        $expenses = $stmt->fetch();
        $total_expenses = floatval($expenses['total_expenses']);
        
        // Transfers out reduce balance
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(amount), 0) as total_out
            FROM fund_transfers
            WHERE from_account_id = :account_id
        ");
        $stmt->execute([':account_id' => $account_id]);
        $transfers_out = $stmt->fetch();
        $total_out = floatval($transfers_out['total_out']);
        
        // Transfers in increase balance (converted to account currency)
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(converted_amount), 0) as total_in
            FROM fund_transfers
            WHERE to_account_id = :account_id
        ");
        $stmt->execute([':account_id' => $account_id]);
        $transfers_in = $stmt->fetch();
        $total_in = floatval($transfers_in['total_in']);
        
        // Note: This is simplified. In a real system, you'd need to track initial balances
        // and handle currency conversions properly
        // Expenses are stored as negative values, income as positive values, so we add them
        $balance = $total_in - $total_out + $total_expenses;
        
        echo json_encode([
            'account_id' => $account_id,
            'currency' => $account['base_currency'],
            'type_code' => $account['type_code'],
            'balance' => $balance,
            'expenses' => $total_expenses,
            'transfers_in' => $total_in,
            'transfers_out' => $total_out
        ]);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error loading account balance: ' . $e->getMessage()]);
    error_log('Account balance API error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
    error_log('Account balance API fatal error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
}
