package com.personalexpenses.app;

import android.content.Context;
import android.content.SharedPreferences;

import com.google.gson.Gson;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class ApiClient {
    private static final MediaType JSON = MediaType.get("application/json; charset=utf-8");
    private final OkHttpClient client;
    private final String baseUrl;
    private final SharedPreferences prefs;
    private final Gson gson;

    public ApiClient(Context context, String baseUrl) {
        this.baseUrl = baseUrl;
        this.prefs = context.getSharedPreferences("PersonalExpenses", Context.MODE_PRIVATE);
        this.gson = new Gson();
        
        this.client = new OkHttpClient.Builder()
                .connectTimeout(30, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .build();
    }

    public void get(String endpoint, ApiCallback callback) {
        Request request = new Request.Builder()
                .url(baseUrl + endpoint)
                .get()
                .build();

        client.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                callback.onError(e.getMessage());
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                if (response.isSuccessful()) {
                    String body = response.body().string();
                    callback.onSuccess(body);
                } else {
                    callback.onError("Error: " + response.code());
                }
            }
        });
    }

    public void post(String endpoint, Object data, ApiCallback callback) {
        String json = gson.toJson(data);
        RequestBody body = RequestBody.create(json, JSON);
        
        Request request = new Request.Builder()
                .url(baseUrl + endpoint)
                .post(body)
                .build();

        client.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                callback.onError(e.getMessage());
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                if (response.isSuccessful()) {
                    String responseBody = response.body().string();
                    callback.onSuccess(responseBody);
                } else {
                    callback.onError("Error: " + response.code());
                }
            }
        });
    }

    public interface ApiCallback {
        void onSuccess(String response);
        void onError(String error);
    }
}

