/**
 * API Client
 */

const API_BASE = '/api';

class API {
    static async request(endpoint, options = {}) {
        const url = API_BASE + endpoint;
        const defaultOptions = {
            headers: {
                'Content-Type': 'application/json',
            },
            credentials: 'include', // Always include cookies for session management
        };

        const config = { ...defaultOptions, ...options };
        
        // Handle FormData (for file uploads)
        if (config.body instanceof FormData) {
            delete config.headers['Content-Type'];
        } else if (config.body && typeof config.body === 'object') {
            config.body = JSON.stringify(config.body);
        }

        try {
            // Log request (avoid JSON.parse for FormData)
            let logBody = null;
            if (config.body instanceof FormData) {
                logBody = '[FormData]';
            } else if (config.body && typeof config.body === 'string') {
                try {
                    logBody = JSON.parse(config.body);
                } catch (e) {
                    logBody = config.body;
                }
            } else {
                logBody = config.body;
            }
            console.log('API Request:', url, config.method || 'GET', logBody);
            const response = await fetch(url, config);
            
            let data;
            try {
                const text = await response.text();
                console.log('API Response text:', text);
                data = JSON.parse(text);
            } catch (parseError) {
                console.error('API Response parse error:', parseError);
                throw new Error('Invalid JSON response');
            }

            // Handle authentication check specially - 401 with authenticated:false is not an error
            if (!response.ok) {
                // For check_session endpoint, 401 with authenticated:false is a valid response
                if (url.includes('/auth/check_session.php') && data && data.authenticated === false) {
                    return data;
                }
                console.error('API Error response:', data);
                throw new Error(data.error || 'Request failed');
            }

            return data;
        } catch (error) {
            console.error('API Error:', error);
            throw error;
        }
    }

    // Auth
    static checkSession() {
        return this.request('/auth/check_session.php');
    }

    static logout() {
        return this.request('/auth/logout.php');
    }

    // Users
    static getProfile() {
        return this.request('/users/profile.php');
    }

    static updateProfile(data) {
        return this.request('/users/profile.php', {
            method: 'PUT',
            body: data
        });
    }

    // Notifications
    static getNotifications(options = {}) {
        const params = new URLSearchParams();
        if (options.unread_only) params.append('unread_only', 'true');
        if (options.limit) params.append('limit', options.limit);
        const queryString = params.toString();
        return this.request('/notifications/index.php' + (queryString ? '?' + queryString : ''));
    }

    static markNotificationRead(notificationId) {
        return this.request(`/notifications/index.php?id=${notificationId}`, {
            method: 'PUT',
            body: { is_read: true }
        });
    }

    static markAllNotificationsRead() {
        return this.request('/notifications/index.php', {
            method: 'PUT',
            body: { mark_all_read: true }
        });
    }

    static deleteNotification(notificationId) {
        return this.request(`/notifications/index.php?id=${notificationId}`, {
            method: 'DELETE'
        });
    }

    static deleteAllReadNotifications() {
        return this.request('/notifications/index.php?delete_all_read=true', {
            method: 'DELETE'
        });
    }

    // User Management (Admin only)
    static getAllUsers() {
        return this.request('/users/index.php');
    }

    static updateUser(userId, data) {
        return this.request('/users/index.php', {
            method: 'PUT',
            body: { id: userId, ...data }
        });
    }

    static getUserDataCounts(userId) {
        return this.request(`/users/index.php?user_id=${userId}`);
    }

    // Debug (Admin only)
    static getSessionDebugLogs(options = {}) {
        const params = new URLSearchParams();
        if (options.limit) params.append('limit', options.limit);
        if (options.offset) params.append('offset', options.offset);
        if (options.event) params.append('event', options.event);
        if (options.user_id) params.append('user_id', options.user_id);
        const queryString = params.toString();
        return this.request(`/debug/session_logs.php${queryString ? '?' + queryString : ''}`);
    }

    static deleteUser(userId, deleteFlags) {
        return this.request('/users/index.php', {
            method: 'DELETE',
            body: { id: userId, delete_flags: deleteFlags }
        });
    }

    // Accounts
    static getAccounts() {
        return this.request('/accounts/index.php');
    }

    static createAccount(data) {
        return this.request('/accounts/index.php', {
            method: 'POST',
            body: data
        });
    }

    static createAccountWithLogo(formData) {
        return this.request('/accounts/index.php', {
            method: 'POST',
            body: formData
        });
    }

    static updateAccount(data) {
        return this.request('/accounts/index.php', {
            method: 'PUT',
            body: data
        });
    }

    static updateAccountWithLogo(formData) {
        // Use POST for FormData because PHP doesn't populate $_POST for PUT requests
        return this.request('/accounts/index.php', {
            method: 'POST',
            body: formData
        });
    }

    static deleteAccount(accountId) {
        return this.request(`/accounts/index.php?id=${accountId}`, {
            method: 'DELETE'
        });
    }

    static getAccountTypes() {
        return this.request('/accounts/types.php');
    }

    static getAccountBalance(accountId) {
        return this.request(`/accounts/balance.php?account_id=${accountId}`);
    }

    static getAccountTotals() {
        return this.request('/accounts/totals.php');
    }

    static saveAccountOrder(accountOrders) {
        return this.request('/accounts/order.php', {
            method: 'POST',
            body: {
                account_orders: accountOrders
            }
        });
    }

    // Account Sharing
    static getAccountShares(accountId) {
        return this.request(`/accounts/share.php?account_id=${accountId}`);
    }

    static getPreviouslySharedUsers() {
        return this.request('/accounts/share.php');
    }

    static shareAccount(accountId, recipientEmail, sharingMode) {
        return this.request('/accounts/share.php', {
            method: 'POST',
            body: {
                account_id: accountId,
                recipient_email: recipientEmail,
                sharing_mode: sharingMode
            }
        });
    }

    static updateShareMode(shareId, sharingMode) {
        return this.request('/accounts/share.php', {
            method: 'PUT',
            body: {
                share_id: shareId,
                sharing_mode: sharingMode
            }
        });
    }

    static updateInvitationMode(invitationId, sharingMode) {
        return this.request('/accounts/share.php', {
            method: 'PUT',
            body: {
                invitation_id: invitationId,
                sharing_mode: sharingMode
            }
        });
    }

    static removeShare(shareId) {
        return this.request(`/accounts/share.php?share_id=${shareId}`, {
            method: 'DELETE'
        });
    }

    static removeInvitation(invitationId) {
        return this.request(`/accounts/share.php?invitation_id=${invitationId}`, {
            method: 'DELETE'
        });
    }

    // Expenses
    static getExpenses(filters = {}) {
        // Filter out empty values to avoid sending empty parameters
        const cleanFilters = {};
        Object.keys(filters).forEach(key => {
            if (filters[key] !== null && filters[key] !== undefined && filters[key] !== '') {
                cleanFilters[key] = filters[key];
            }
        });
        
        const params = new URLSearchParams(cleanFilters);
        const queryString = params.toString();
        const url = queryString ? `/expenses/index.php?${queryString}` : '/expenses/index.php';
        return this.request(url);
    }

    static createExpense(data) {
        return this.request('/expenses/index.php', {
            method: 'POST',
            body: data
        });
    }

    static updateExpense(data) {
        return this.request('/expenses/index.php', {
            method: 'PUT',
            body: data
        });
    }

    static deleteExpense(expenseId) {
        return this.request(`/expenses/index.php?id=${expenseId}`, {
            method: 'DELETE'
        });
    }

    static getTaxCategories() {
        return this.request('/expenses/tax_categories.php');
    }

    static addExpenseAttachment(expenseId, type, file = null, linkUrl = null, linkText = null) {
        const formData = new FormData();
        formData.append('expense_id', expenseId);
        formData.append('type', type);
        
        if (type === 'LINK') {
            formData.append('link_url', linkUrl);
            if (linkText) formData.append('link_text', linkText);
        } else if (file) {
            formData.append('file', file);
        }

        return this.request('/expenses/attachments.php', {
            method: 'POST',
            body: formData
        });
    }

    static deleteExpenseAttachment(attachmentId) {
        return this.request(`/expenses/attachments.php?id=${attachmentId}`, {
            method: 'DELETE'
        });
    }

    // Data Export/Import
    static exportData() {
        // Export returns a file download, so we handle it differently
        return fetch(API_BASE + '/users/export.php', {
            method: 'GET',
            credentials: 'include'
        }).then(response => {
            if (!response.ok) {
                return response.json().then(err => Promise.reject(new Error(err.error || 'Export failed')));
            }
            return response.blob().then(blob => {
                // Create download link
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `personal_expenses_export_${new Date().toISOString().split('T')[0]}.zip`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
                return { success: true };
            });
        });
    }

    static importData(file) {
        const formData = new FormData();
        formData.append('import_file', file);
        
        return this.request('/users/import.php', {
            method: 'POST',
            body: formData
        });
    }

    // Transfers
    static getTransfers() {
        return this.request('/transfers/index.php');
    }

    static createTransfer(data) {
        return this.request('/transfers/index.php', {
            method: 'POST',
            body: data
        });
    }

    static updateTransfer(data) {
        return this.request('/transfers/index.php', {
            method: 'PUT',
            body: data
        });
    }

    static deleteTransfer(transferId) {
        return this.request(`/transfers/index.php?id=${transferId}`, {
            method: 'DELETE'
        });
    }

    static getTransfer(transferId) {
        return this.request(`/transfers/index.php?id=${transferId}`);
    }

    static getExchangeRate(from, to) {
        return this.request(`/transfers/exchange_rate.php?from=${from}&to=${to}`);
    }

    static addTransferAttachment(transferId, type, file = null, linkUrl = null, linkText = null) {
        const formData = new FormData();
        formData.append('transfer_id', transferId);
        formData.append('type', type);
        
        if (type === 'LINK') {
            formData.append('link_url', linkUrl);
            if (linkText) formData.append('link_text', linkText);
        } else if (file) {
            formData.append('file', file);
        }

        return this.request('/transfers/attachments.php', {
            method: 'POST',
            body: formData
        });
    }

    // Securities
    static getSecurities(accountId = null) {
        const url = accountId 
            ? `/securities/index.php?account_id=${accountId}`
            : '/securities/index.php';
        return this.request(url);
    }

    static createSecurity(data) {
        return this.request('/securities/index.php', {
            method: 'POST',
            body: data
        });
    }

    static updateSecurity(data) {
        return this.request('/securities/index.php', {
            method: 'PUT',
            body: data
        });
    }

    static deleteSecurity(securityId) {
        return this.request(`/securities/index.php?id=${securityId}`, {
            method: 'DELETE'
        });
    }

    static getSecuritiesSummary(accountId = null) {
        const url = accountId
            ? `/securities/summary.php?account_id=${accountId}`
            : '/securities/summary.php';
        return this.request(url);
    }

    static addSecurityAttachment(securityId, type, file = null, linkUrl = null, linkText = null) {
        const formData = new FormData();
        formData.append('security_id', securityId);
        formData.append('type', type);
        
        if (type === 'LINK') {
            formData.append('link_url', linkUrl);
            if (linkText) formData.append('link_text', linkText);
        } else if (file) {
            formData.append('file', file);
        }

        return this.request('/securities/attachments.php', {
            method: 'POST',
            body: formData
        });
    }

    static deleteSecurityAttachment(attachmentId) {
        return this.request(`/securities/attachments.php?id=${attachmentId}`, {
            method: 'DELETE'
        });
    }

    // Prepaid Cards
    static getPrepaidCards(params = {}) {
        const searchParams = new URLSearchParams();
        if (params.enabled_only) {
            searchParams.append('enabled_only', 'true');
        }
        if (params.archived_only) {
            searchParams.append('archived_only', 'true');
        }
        const queryString = searchParams.toString();
        return this.request('/prepaid_cards/index.php' + (queryString ? '?' + queryString : ''));
    }

    static createPrepaidCard(data, photo = null) {
        const formData = new FormData();
        Object.keys(data).forEach(key => {
            formData.append(key, data[key]);
        });
        if (photo) {
            formData.append('photo', photo);
        }

        return this.request('/prepaid_cards/index.php', {
            method: 'POST',
            body: formData
        });
    }

    static updatePrepaidCard(cardId, data, photo = null) {
        const formData = new FormData();
        // Add id to FormData to indicate this is an update
        formData.append('id', cardId);
        Object.keys(data).forEach(key => {
            formData.append(key, data[key]);
        });
        if (photo) {
            formData.append('photo', photo);
        }

        // Use POST instead of PUT for FormData (PHP doesn't populate $_POST for PUT requests)
        return this.request('/prepaid_cards/index.php', {
            method: 'POST',
            body: formData
        });
    }

    static deletePrepaidCard(cardId) {
        return this.request(`/prepaid_cards/index.php?id=${cardId}`, {
            method: 'DELETE'
        });
    }

    static archivePrepaidCard(cardId, archived = true) {
        return this.request(`/prepaid_cards/index.php?id=${cardId}`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ archived })
        });
    }

    static getPreviouslySharedUsersForPrepaidCards() {
        return this.request('/prepaid_cards/share.php');
    }

    static getPrepaidCardShares(prepaidCardId) {
        return this.request(`/prepaid_cards/share.php?prepaid_card_id=${prepaidCardId}`);
    }

    static sharePrepaidCard(prepaidCardId, recipientEmail) {
        return this.request('/prepaid_cards/share.php', {
            method: 'POST',
            body: {
                prepaid_card_id: prepaidCardId,
                recipient_email: recipientEmail
            }
        });
    }

    static removePrepaidCardShare(shareId) {
        return this.request(`/prepaid_cards/share.php?share_id=${shareId}`, {
            method: 'DELETE'
        });
    }

    static removePrepaidCardInvitation(invitationId) {
        return this.request(`/prepaid_cards/share.php?invitation_id=${invitationId}`, {
            method: 'DELETE'
        });
    }

    // Fidelity Cards
    static getFidelityCards() {
        return this.request('/fidelity_cards/index.php');
    }

    static createFidelityCard(data, logo = null, cardImage = null) {
        const formData = new FormData();
        Object.keys(data).forEach(key => {
            formData.append(key, data[key]);
        });
        if (logo) {
            formData.append('logo', logo);
        }
        if (cardImage) {
            formData.append('card_image', cardImage);
        }

        return this.request('/fidelity_cards/index.php', {
            method: 'POST',
            body: formData
        });
    }

    static updateFidelityCard(cardId, data, logo = null, cardImage = null) {
        const formData = new FormData();
        formData.append('id', cardId);
        Object.keys(data).forEach(key => {
            formData.append(key, data[key]);
        });
        if (logo) {
            formData.append('logo', logo);
        }
        if (cardImage) {
            formData.append('card_image', cardImage);
        }

        return this.request('/fidelity_cards/index.php', {
            method: 'POST',
            body: formData
        });
    }

    static deleteFidelityCard(cardId) {
        return this.request(`/fidelity_cards/index.php?id=${cardId}`, {
            method: 'DELETE'
        });
    }

    static saveFidelityCardOrder(cardOrders) {
        return this.request('/fidelity_cards/order.php', {
            method: 'POST',
            body: {
                card_orders: cardOrders
            }
        });
    }

    static trackFidelityCardClick(cardId) {
        return this.request(`/fidelity_cards/click.php?id=${cardId}`, {
            method: 'POST'
        });
    }

    static getFidelityCardShares(fidelityCardId) {
        return this.request(`/fidelity_cards/share.php?fidelity_card_id=${fidelityCardId}`);
    }

    static getPreviouslySharedUsersForFidelityCards() {
        return this.request('/fidelity_cards/share.php');
    }

    static shareFidelityCard(fidelityCardId, recipientEmail) {
        return this.request('/fidelity_cards/share.php', {
            method: 'POST',
            body: {
                fidelity_card_id: fidelityCardId,
                recipient_email: recipientEmail
            }
        });
    }

    static removeFidelityCardShare(shareId) {
        return this.request(`/fidelity_cards/share.php?share_id=${shareId}`, {
            method: 'DELETE'
        });
    }

    static removeFidelityCardInvitation(invitationId) {
        return this.request(`/fidelity_cards/share.php?invitation_id=${invitationId}`, {
            method: 'DELETE'
        });
    }
}

