<?php
/**
 * User Profile API
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        // Get user profile
        $stmt = $conn->prepare("
            SELECT id, email, name, picture_url, default_currency, default_account_id, default_view, 
                   expiration_notification_days, is_admin,
                   (SELECT name FROM accounts WHERE id = default_account_id) as default_account_name
            FROM users
            WHERE id = :user_id
        ");
        $stmt->execute([':user_id' => $user_id]);
        $profile = $stmt->fetch();
        
        if ($profile) {
            // Ensure default_view has a value (for existing users without this field)
            if (empty($profile['default_view'])) {
                $profile['default_view'] = 'dashboard';
            }
            echo json_encode($profile);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Profile not found']);
        }
        break;
        
    case 'PUT':
        // Update user profile
        $data = json_decode(file_get_contents('php://input'), true);
        
        $update_fields = [];
        $params = [':user_id' => $user_id];
        
        if (isset($data['default_currency'])) {
            $update_fields[] = 'default_currency = :default_currency';
            $params[':default_currency'] = $data['default_currency'];
        }
        
        if (isset($data['default_account_id'])) {
            // Verify account belongs to user (only owned accounts can be default)
            $stmt = $conn->prepare("SELECT id FROM accounts WHERE id = :account_id AND user_id = :user_id");
            $stmt->execute([':account_id' => $data['default_account_id'], ':user_id' => $user_id]);
            if ($stmt->fetch()) {
                $update_fields[] = 'default_account_id = :default_account_id';
                $params[':default_account_id'] = $data['default_account_id'];
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Account not found or not owned by user']);
                break;
            }
        }
        
        if (isset($data['default_view'])) {
            // Validate default_view value
            $allowed_views = ['dashboard', 'expenses', 'securities', 'prepaid-cards', 'fidelity-cards'];
            if (in_array($data['default_view'], $allowed_views)) {
                $update_fields[] = 'default_view = :default_view';
                $params[':default_view'] = $data['default_view'];
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid default_view value']);
                break;
            }
        }
        
        if (isset($data['expiration_notification_days'])) {
            $days = (int)$data['expiration_notification_days'];
            if ($days >= 0 && $days <= 365) {
                $update_fields[] = 'expiration_notification_days = :expiration_notification_days';
                $params[':expiration_notification_days'] = $days;
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'expiration_notification_days must be between 0 and 365']);
                break;
            }
        }
        
        if (empty($update_fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            break;
        }
        
        $sql = "UPDATE users SET " . implode(', ', $update_fields) . " WHERE id = :user_id";
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        
        echo json_encode(['success' => true, 'message' => 'Profile updated']);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}

