<?php
/**
 * Export User Data API - Creates a ZIP file with JSON and attachments
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

try {
    // Check if ZipArchive is available
    if (!class_exists('ZipArchive')) {
        throw new Exception('ZipArchive class not available. Please install php-zip extension.');
    }
    
    // Create temporary ZIP file
    $zip_filename = sys_get_temp_dir() . '/export_' . $user_id . '_' . time() . '.zip';
    $zip = new ZipArchive();
    
    if ($zip->open($zip_filename, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
        throw new Exception('Cannot create ZIP file');
    }
    
    // Get user profile
    $stmt = $conn->prepare("SELECT default_currency, default_account_id FROM users WHERE id = :user_id");
    $stmt->execute([':user_id' => $user_id]);
    $profile = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get accounts with logos
    $stmt = $conn->prepare("
        SELECT id, name, type_id, base_currency, opening_date, id_number, expire_date, closure_date, tax_type, logo_path
        FROM accounts
        WHERE user_id = :user_id
        ORDER BY id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get account types mapping (for reference)
    $stmt = $conn->prepare("SELECT id, code FROM account_types");
    $stmt->execute();
    $account_types_map = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $account_types_map[$row['id']] = $row['code'];
    }
    
    // Add account logos to ZIP and update paths
    foreach ($accounts as $idx => $account) {
        if (!empty($account['logo_path'])) {
            $logo_file = UPLOAD_DIR . $account['logo_path'];
            if (file_exists($logo_file)) {
                // Add file to ZIP maintaining directory structure
                $zip_path = 'attachments/' . $account['logo_path'];
                $zip->addFile($logo_file, $zip_path);
                // Store relative path for import
                $accounts[$idx]['logo_path_export'] = $zip_path;
            }
        }
    }
    
    // Get expenses
    $stmt = $conn->prepare("
        SELECT id, account_id, prepaid_card_id, date, description, amount, currency, is_tax_deductible, tax_category_id
        FROM expenses
        WHERE user_id = :user_id
        ORDER BY id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get expense attachments
    $expense_ids = array_column($expenses, 'id');
    $attachments = [];
    if (!empty($expense_ids)) {
        $placeholders = implode(',', array_fill(0, count($expense_ids), '?'));
        $stmt = $conn->prepare("
            SELECT id, expense_id, type, file_path, link_url, link_text
            FROM expense_attachments
            WHERE expense_id IN ($placeholders)
            ORDER BY expense_id, id
        ");
        $stmt->execute($expense_ids);
        $attachments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Add attachment files to ZIP
        foreach ($attachments as $idx => $attachment) {
            if ($attachment['type'] !== 'LINK' && !empty($attachment['file_path'])) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    // Add file to ZIP maintaining directory structure
                    $zip_path = 'attachments/' . $attachment['file_path'];
                    $zip->addFile($file_path, $zip_path);
                    // Store relative path for import
                    $attachments[$idx]['file_path_export'] = $zip_path;
                }
            }
        }
    }
    
    // Group attachments by expense_id
    $attachments_by_expense = [];
    foreach ($attachments as $attachment) {
        $expense_id = $attachment['expense_id'];
        if (!isset($attachments_by_expense[$expense_id])) {
            $attachments_by_expense[$expense_id] = [];
        }
        $attachments_by_expense[$expense_id][] = $attachment;
    }
    
    // Add attachments to expenses
    foreach ($expenses as $idx => $expense) {
        $expense_id = $expense['id'];
        $expenses[$idx]['attachments'] = $attachments_by_expense[$expense_id] ?? [];
    }
    
    // Get fund transfers
    $stmt = $conn->prepare("
        SELECT id, from_account_id, to_account_id, amount, from_currency, to_currency, exchange_rate, converted_amount, transfer_date, arrival_date, notes
        FROM fund_transfers
        WHERE user_id = :user_id
        ORDER BY id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $transfers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get transfer attachments
    $transfer_ids = array_column($transfers, 'id');
    $transfer_attachments = [];
    if (!empty($transfer_ids)) {
        $placeholders = implode(',', array_fill(0, count($transfer_ids), '?'));
        $stmt = $conn->prepare("
            SELECT id, transfer_id, type, file_path, link_url, link_text
            FROM transfer_attachments
            WHERE transfer_id IN ($placeholders)
            ORDER BY transfer_id, id
        ");
        $stmt->execute($transfer_ids);
        $transfer_attachments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Add attachment files to ZIP
        foreach ($transfer_attachments as $idx => $attachment) {
            if ($attachment['type'] !== 'LINK' && !empty($attachment['file_path'])) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    // Add file to ZIP maintaining directory structure
                    $zip_path = 'attachments/' . $attachment['file_path'];
                    $zip->addFile($file_path, $zip_path);
                    // Store relative path for import
                    $transfer_attachments[$idx]['file_path_export'] = $zip_path;
                }
            }
        }
    }
    
    // Group attachments by transfer_id
    $attachments_by_transfer = [];
    foreach ($transfer_attachments as $attachment) {
        $transfer_id = $attachment['transfer_id'];
        if (!isset($attachments_by_transfer[$transfer_id])) {
            $attachments_by_transfer[$transfer_id] = [];
        }
        $attachments_by_transfer[$transfer_id][] = $attachment;
    }
    
    // Add attachments to transfers
    foreach ($transfers as $idx => $transfer) {
        $transfer_id = $transfer['id'];
        $transfers[$idx]['attachments'] = $attachments_by_transfer[$transfer_id] ?? [];
    }
    
    // Get securities
    $stmt = $conn->prepare("
        SELECT id, account_id, symbol, quantity, purchase_price, purchase_date, currency, notes, transaction_type
        FROM securities
        WHERE account_id IN (SELECT id FROM accounts WHERE user_id = :user_id)
        ORDER BY id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $securities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get securities attachments
    $security_ids = array_column($securities, 'id');
    $securities_attachments = [];
    if (!empty($security_ids)) {
        $placeholders = implode(',', array_fill(0, count($security_ids), '?'));
        $stmt = $conn->prepare("
            SELECT id, security_id, type, file_path, link_url, link_text
            FROM securities_attachments
            WHERE security_id IN ($placeholders)
            ORDER BY security_id, id
        ");
        $stmt->execute($security_ids);
        $securities_attachments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Add attachment files to ZIP
        foreach ($securities_attachments as $idx => $attachment) {
            if ($attachment['type'] !== 'LINK' && !empty($attachment['file_path'])) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    // Add file to ZIP maintaining directory structure
                    $zip_path = 'attachments/' . $attachment['file_path'];
                    $zip->addFile($file_path, $zip_path);
                    // Store relative path for import
                    $securities_attachments[$idx]['file_path_export'] = $zip_path;
                }
            }
        }
    }
    
    // Group attachments by security_id
    $attachments_by_security = [];
    foreach ($securities_attachments as $attachment) {
        $security_id = $attachment['security_id'];
        if (!isset($attachments_by_security[$security_id])) {
            $attachments_by_security[$security_id] = [];
        }
        $attachments_by_security[$security_id][] = $attachment;
    }
    
    // Add attachments to securities
    foreach ($securities as $idx => $security) {
        $security_id = $security['id'];
        $securities[$idx]['attachments'] = $attachments_by_security[$security_id] ?? [];
    }
    
    // Get prepaid cards
    $stmt = $conn->prepare("
        SELECT id, name, card_number, base_currency, nominal_amount, opening_date, expire_date, enabled, photo_path
        FROM prepaid_cards
        WHERE user_id = :user_id
        ORDER BY id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $cards = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Add card photos to ZIP
    foreach ($cards as $idx => $card) {
        if (!empty($card['photo_path'])) {
            $photo_file = UPLOAD_DIR . $card['photo_path'];
            if (file_exists($photo_file)) {
                // Add file to ZIP maintaining directory structure
                $zip_path = 'attachments/' . $card['photo_path'];
                $zip->addFile($photo_file, $zip_path);
                // Store relative path for import
                $cards[$idx]['photo_path_export'] = $zip_path;
            }
        }
    }
    
    // Get fidelity cards
    $stmt = $conn->prepare("
        SELECT id, name, store_name, card_code, description, logo_path, card_image_path
        FROM fidelity_cards
        WHERE user_id = :user_id
        ORDER BY id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $fidelity_cards = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Add fidelity card logos and images to ZIP
    foreach ($fidelity_cards as $idx => $card) {
        if (!empty($card['logo_path'])) {
            $logo_file = UPLOAD_DIR . $card['logo_path'];
            if (file_exists($logo_file)) {
                $zip_path = 'attachments/' . $card['logo_path'];
                $zip->addFile($logo_file, $zip_path);
                $fidelity_cards[$idx]['logo_path_export'] = $zip_path;
            }
        }
        if (!empty($card['card_image_path'])) {
            $image_file = UPLOAD_DIR . $card['card_image_path'];
            if (file_exists($image_file)) {
                $zip_path = 'attachments/' . $card['card_image_path'];
                $zip->addFile($image_file, $zip_path);
                $fidelity_cards[$idx]['card_image_path_export'] = $zip_path;
            }
        }
    }
    
    // Get account order
    $stmt = $conn->prepare("
        SELECT account_id, display_order
        FROM user_account_order
        WHERE user_id = :user_id
        ORDER BY display_order
    ");
    $stmt->execute([':user_id' => $user_id]);
    $account_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get fidelity card order
    $stmt = $conn->prepare("
        SELECT fidelity_card_id, display_order
        FROM user_fidelity_card_order
        WHERE user_id = :user_id
        ORDER BY display_order
    ");
    $stmt->execute([':user_id' => $user_id]);
    $fidelity_card_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get account shares (where user is owner)
    $stmt = $conn->prepare("
        SELECT ashr.account_id, ashr.sharing_mode, u.email as shared_with_email
        FROM account_shares ashr
        JOIN users u ON ashr.shared_with_user_id = u.id
        WHERE ashr.owner_user_id = :user_id
        ORDER BY ashr.account_id, ashr.id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $account_shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get account share invitations (where user is owner)
    $stmt = $conn->prepare("
        SELECT account_id, recipient_email, sharing_mode
        FROM account_share_invitations
        WHERE owner_user_id = :user_id
        ORDER BY account_id, id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $account_share_invitations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get prepaid card shares (where user is owner)
    $stmt = $conn->prepare("
        SELECT pcs.prepaid_card_id, u.email as shared_with_email
        FROM prepaid_card_shares pcs
        JOIN users u ON pcs.shared_with_user_id = u.id
        WHERE pcs.owner_user_id = :user_id
        ORDER BY pcs.prepaid_card_id, pcs.id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $prepaid_card_shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get prepaid card share invitations (where user is owner)
    $stmt = $conn->prepare("
        SELECT prepaid_card_id, recipient_email
        FROM prepaid_card_share_invitations
        WHERE owner_user_id = :user_id
        ORDER BY prepaid_card_id, id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $prepaid_card_share_invitations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get fidelity card shares (where user is owner)
    $stmt = $conn->prepare("
        SELECT fcs.fidelity_card_id, u.email as shared_with_email
        FROM fidelity_card_shares fcs
        JOIN users u ON fcs.shared_with_user_id = u.id
        WHERE fcs.owner_user_id = :user_id
        ORDER BY fcs.fidelity_card_id, fcs.id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $fidelity_card_shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get fidelity card share invitations (where user is owner)
    $stmt = $conn->prepare("
        SELECT fidelity_card_id, recipient_email
        FROM fidelity_card_share_invitations
        WHERE owner_user_id = :user_id
        ORDER BY fidelity_card_id, id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $fidelity_card_share_invitations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Build export data (without base64, just paths)
    $export_data = [
        'version' => '1.0',
        'export_date' => date('Y-m-d H:i:s'),
        'profile' => $profile,
        'accounts' => $accounts,
        'account_orders' => $account_orders,
        'account_shares' => $account_shares,
        'account_share_invitations' => $account_share_invitations,
        'expenses' => $expenses,
        'transfers' => $transfers,
        'securities' => $securities,
        'prepaid_cards' => $cards,
        'prepaid_card_shares' => $prepaid_card_shares,
        'prepaid_card_share_invitations' => $prepaid_card_share_invitations,
        'fidelity_cards' => $fidelity_cards,
        'fidelity_card_orders' => $fidelity_card_orders,
        'fidelity_card_shares' => $fidelity_card_shares,
        'fidelity_card_share_invitations' => $fidelity_card_share_invitations,
        'account_types_map' => $account_types_map
    ];
    
    // Add JSON data file to ZIP
    $json_content = json_encode($export_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    $zip->addFromString('data.json', $json_content);
    
    // Close ZIP
    $zip->close();
    
    // Send ZIP file
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="personal_expenses_export_' . date('Y-m-d') . '.zip"');
    header('Content-Length: ' . filesize($zip_filename));
    header('Cache-Control: no-cache, must-revalidate');
    
    readfile($zip_filename);
    
    // Clean up temporary file
    unlink($zip_filename);
    
} catch (Exception $e) {
    // Clean up ZIP file if it exists
    if (isset($zip_filename) && file_exists($zip_filename)) {
        unlink($zip_filename);
    }
    
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Export failed: ' . $e->getMessage()]);
    error_log('Export error: ' . $e->getMessage());
}
