<?php
/**
 * Fund Transfers API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
    case 'GET':
        // Get transfer(s) - if ID provided, get single transfer, otherwise get all
        $transfer_id = $_GET['id'] ?? null;
        
        if ($transfer_id) {
            // Get single transfer
            $stmt = $conn->prepare("
                SELECT ft.*,
                       fa.name as from_account_name, fa.base_currency as from_account_currency,
                       ta.name as to_account_name, ta.base_currency as to_account_currency
                FROM fund_transfers ft
                JOIN accounts fa ON ft.from_account_id = fa.id
                JOIN accounts ta ON ft.to_account_id = ta.id
                WHERE ft.id = :id AND ft.user_id = :user_id
            ");
            $stmt->execute([':id' => $transfer_id, ':user_id' => $user_id]);
            $transfer = $stmt->fetch();
            
            if (!$transfer) {
                http_response_code(404);
                echo json_encode(['error' => 'Transfer not found']);
                break;
            }
            
            echo json_encode($transfer);
        } else {
            // Get all transfers for user
            // Include transfers made by user OR transfers from/to accounts owned by user OR transfers from/to shared accounts
            $stmt = $conn->prepare("
                SELECT ft.*,
                       fa.name as from_account_name, fa.base_currency as from_account_currency,
                       ta.name as to_account_name, ta.base_currency as to_account_currency
                FROM fund_transfers ft
                JOIN accounts fa ON ft.from_account_id = fa.id
                JOIN accounts ta ON ft.to_account_id = ta.id
                WHERE (
                    ft.user_id = :user_id
                    OR fa.user_id = :user_id_owner_from
                    OR ta.user_id = :user_id_owner_to
                    OR EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = ft.from_account_id 
                        AND ash.shared_with_user_id = :user_id_shared_from
                    )
                    OR EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = ft.to_account_id 
                        AND ash.shared_with_user_id = :user_id_shared_to
                    )
                )
                ORDER BY ft.transfer_date DESC, ft.created_at DESC
            ");
            $stmt->execute([
                ':user_id' => $user_id,
                ':user_id_owner_from' => $user_id,
                ':user_id_owner_to' => $user_id,
                ':user_id_shared_from' => $user_id,
                ':user_id_shared_to' => $user_id
            ]);
            $transfers = $stmt->fetchAll();
            
            echo json_encode($transfers);
        }
        break;
        
    case 'POST':
        // Create new transfer
        $data = json_decode(file_get_contents('php://input'), true);
        
        $required = ['from_account_id', 'to_account_id', 'amount', 'transfer_date'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                exit;
            }
        }
        
        $arrival_date = isset($data['arrival_date']) && !empty($data['arrival_date']) ? $data['arrival_date'] : null;
        
        // Verify both accounts belong to user OR user has write access via sharing
        // For transfers, at least one account must be owned by user (can't transfer between two shared accounts)
        $stmt = $conn->prepare("
            SELECT 
                a.id, 
                a.base_currency,
                CASE 
                    WHEN a.user_id = :user_id_owner THEN 'owned'
                    WHEN EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = a.id 
                        AND ash.shared_with_user_id = :user_id_shared
                        AND ash.sharing_mode = 'write'
                    ) THEN 'shared_write'
                    ELSE 'no_access'
                END as access_type
            FROM accounts a
            WHERE a.id IN (:id1, :id2)
            AND (
                a.user_id = :user_id_check 
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = a.id 
                    AND ash.shared_with_user_id = :user_id_check2
                    AND ash.sharing_mode = 'write'
                )
            )
        ");
        $stmt->execute([
            ':id1' => $data['from_account_id'],
            ':id2' => $data['to_account_id'],
            ':user_id_owner' => $user_id,
            ':user_id_shared' => $user_id,
            ':user_id_check' => $user_id,
            ':user_id_check2' => $user_id
        ]);
        $accounts = $stmt->fetchAll();
        
        if (count($accounts) !== 2) {
            http_response_code(404);
            echo json_encode(['error' => 'One or both accounts not found or you do not have write access']);
            exit;
        }
        
        // Check that at least one account is owned (not both shared)
        $owned_count = 0;
        foreach ($accounts as $acc) {
            if ($acc['access_type'] === 'owned') {
                $owned_count++;
            }
        }
        
        if ($owned_count === 0) {
            http_response_code(403);
            echo json_encode(['error' => 'Cannot transfer between two shared accounts. At least one account must be owned by you.']);
            exit;
        }
        
        $from_account = $accounts[0]['id'] == $data['from_account_id'] ? $accounts[0] : $accounts[1];
        $to_account = $accounts[0]['id'] == $data['to_account_id'] ? $accounts[0] : $accounts[1];
        
        $from_currency = $from_account['base_currency'];
        $to_currency = $to_account['base_currency'];
        
        // Handle currency conversion
        $exchange_rate = 1.0;
        $converted_amount = $data['amount'];
        
        if ($from_currency !== $to_currency) {
            if (isset($data['exchange_rate']) && $data['exchange_rate'] > 0) {
                $exchange_rate = $data['exchange_rate'];
                $converted_amount = $data['amount'] * $exchange_rate;
            } else {
                // Try to fetch current exchange rate
                $exchange_rate = fetchExchangeRate($from_currency, $to_currency);
                if ($exchange_rate > 0) {
                    $converted_amount = $data['amount'] * $exchange_rate;
                } else {
                    http_response_code(400);
                    echo json_encode(['error' => 'Exchange rate required for currency conversion']);
                    exit;
                }
            }
        }
        
        $stmt = $conn->prepare("
            INSERT INTO fund_transfers (user_id, from_account_id, to_account_id, amount, from_currency, to_currency, exchange_rate, converted_amount, transfer_date, arrival_date, notes)
            VALUES (:user_id, :from_account_id, :to_account_id, :amount, :from_currency, :to_currency, :exchange_rate, :converted_amount, :transfer_date, :arrival_date, :notes)
        ");
        
        $stmt->execute([
            ':user_id' => $user_id,
            ':from_account_id' => $data['from_account_id'],
            ':to_account_id' => $data['to_account_id'],
            ':amount' => $data['amount'],
            ':from_currency' => $from_currency,
            ':to_currency' => $to_currency,
            ':exchange_rate' => $exchange_rate,
            ':converted_amount' => $converted_amount,
            ':transfer_date' => $data['transfer_date'],
            ':arrival_date' => $arrival_date,
            ':notes' => $data['notes'] ?? null
        ]);
        
        $transfer_id = $conn->lastInsertId();
        
        echo json_encode(['success' => true, 'id' => $transfer_id]);
        break;
        
    case 'PUT':
        // Update existing transfer
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Transfer ID is required']);
            break;
        }
        
        $transfer_id = (int)$data['id'];
        
        // Get transfer with account information
        $stmt = $conn->prepare("
            SELECT ft.*, 
                   fa.user_id as from_account_owner_id,
                   ta.user_id as to_account_owner_id
            FROM fund_transfers ft
            JOIN accounts fa ON ft.from_account_id = fa.id
            JOIN accounts ta ON ft.to_account_id = ta.id
            WHERE ft.id = :id
        ");
        $stmt->execute([':id' => $transfer_id]);
        $transfer = $stmt->fetch();
        
        if (!$transfer) {
            http_response_code(404);
            echo json_encode(['error' => 'Transfer not found']);
            break;
        }
        
        // Get original account IDs
        $original_from_account_id = $transfer['from_account_id'];
        $original_to_account_id = $transfer['to_account_id'];
        
        // Verify both original accounts belong to user OR user has write access via sharing
        $from_account_owner_id = $transfer['from_account_owner_id'];
        $to_account_owner_id = $transfer['to_account_owner_id'];
        
        // Check if user owns from_account
        $owns_from_account = ($from_account_owner_id == $user_id);
        
        // Check if user has write access to from_account via sharing
        $stmt = $conn->prepare("
            SELECT id FROM account_shares 
            WHERE account_id = :account_id 
            AND shared_with_user_id = :user_id 
            AND sharing_mode = 'write'
        ");
        $stmt->execute([':account_id' => $original_from_account_id, ':user_id' => $user_id]);
        $has_write_access_from = (bool)$stmt->fetch();
        
        // Check if user owns to_account
        $owns_to_account = ($to_account_owner_id == $user_id);
        
        // Check if user has write access to to_account via sharing
        $stmt = $conn->prepare("
            SELECT id FROM account_shares 
            WHERE account_id = :account_id 
            AND shared_with_user_id = :user_id 
            AND sharing_mode = 'write'
        ");
        $stmt->execute([':account_id' => $original_to_account_id, ':user_id' => $user_id]);
        $has_write_access_to = (bool)$stmt->fetch();
        
        // User must have write access to both original accounts
        $can_modify_from = $owns_from_account || $has_write_access_from;
        $can_modify_to = $owns_to_account || $has_write_access_to;
        
        if (!$can_modify_from || !$can_modify_to) {
            http_response_code(403);
            echo json_encode([
                'error' => 'Non hai i permessi necessari per modificare questo trasferimento. Devi avere accesso in scrittura ad entrambi gli account coinvolti. Contatta l\'utente che ha creato il trasferimento.',
                'requires_contact' => true
            ]);
            break;
        }
        
        // Verify both accounts belong to user OR user has write access via sharing
        $from_account_id = isset($data['from_account_id']) ? $data['from_account_id'] : $transfer['from_account_id'];
        $to_account_id = isset($data['to_account_id']) ? $data['to_account_id'] : $transfer['to_account_id'];
        
        $stmt = $conn->prepare("
            SELECT 
                a.id, 
                a.base_currency,
                CASE 
                    WHEN a.user_id = :user_id_owner THEN 'owned'
                    WHEN EXISTS (
                        SELECT 1 FROM account_shares ash 
                        WHERE ash.account_id = a.id 
                        AND ash.shared_with_user_id = :user_id_shared
                        AND ash.sharing_mode = 'write'
                    ) THEN 'shared_write'
                    ELSE 'no_access'
                END as access_type
            FROM accounts a
            WHERE a.id IN (:id1, :id2)
            AND (
                a.user_id = :user_id_check 
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = a.id 
                    AND ash.shared_with_user_id = :user_id_check2
                    AND ash.sharing_mode = 'write'
                )
            )
        ");
        $stmt->execute([
            ':id1' => $from_account_id,
            ':id2' => $to_account_id,
            ':user_id_owner' => $user_id,
            ':user_id_shared' => $user_id,
            ':user_id_check' => $user_id,
            ':user_id_check2' => $user_id
        ]);
        $accounts = $stmt->fetchAll();
        
        if (count($accounts) !== 2) {
            http_response_code(404);
            echo json_encode(['error' => 'One or both accounts not found or you do not have write access']);
            break;
        }
        
        // Check that at least one account is owned (not both shared)
        $owned_count = 0;
        foreach ($accounts as $acc) {
            if ($acc['access_type'] === 'owned') {
                $owned_count++;
            }
        }
        
        if ($owned_count === 0) {
            http_response_code(403);
            echo json_encode(['error' => 'Cannot transfer between two shared accounts. At least one account must be owned by you.']);
            break;
        }
        
        $from_account = $accounts[0]['id'] == $from_account_id ? $accounts[0] : $accounts[1];
        $to_account = $accounts[0]['id'] == $to_account_id ? $accounts[0] : $accounts[1];
        
        $from_currency = $from_account['base_currency'];
        $to_currency = $to_account['base_currency'];
        
        // Handle currency conversion
        $amount = isset($data['amount']) ? $data['amount'] : $transfer['amount'];
        $exchange_rate = $transfer['exchange_rate'];
        $converted_amount = $transfer['converted_amount'];
        
        if ($from_currency !== $to_currency) {
            if (isset($data['exchange_rate']) && $data['exchange_rate'] > 0) {
                $exchange_rate = $data['exchange_rate'];
                $converted_amount = $amount * $exchange_rate;
            } else if ($from_currency !== $transfer['from_currency'] || $to_currency !== $transfer['to_currency']) {
                // Currencies changed, need new exchange rate
                $exchange_rate = fetchExchangeRate($from_currency, $to_currency);
                if ($exchange_rate > 0) {
                    $converted_amount = $amount * $exchange_rate;
                } else {
                    http_response_code(400);
                    echo json_encode(['error' => 'Exchange rate required for currency conversion']);
                    break;
                }
            }
        } else {
            $exchange_rate = 1.0;
            $converted_amount = $amount;
        }
        
        $transfer_date = isset($data['transfer_date']) ? $data['transfer_date'] : $transfer['transfer_date'];
        $arrival_date = isset($data['arrival_date']) && !empty($data['arrival_date']) ? $data['arrival_date'] : null;
        $notes = isset($data['notes']) ? $data['notes'] : $transfer['notes'];
        
        $stmt = $conn->prepare("
            UPDATE fund_transfers 
            SET from_account_id = :from_account_id, to_account_id = :to_account_id, 
                amount = :amount, from_currency = :from_currency, to_currency = :to_currency,
                exchange_rate = :exchange_rate, converted_amount = :converted_amount,
                transfer_date = :transfer_date, arrival_date = :arrival_date, notes = :notes
            WHERE id = :id AND user_id = :user_id
        ");
        
        $stmt->execute([
            ':id' => $transfer_id,
            ':user_id' => $user_id,
            ':from_account_id' => $from_account_id,
            ':to_account_id' => $to_account_id,
            ':amount' => $amount,
            ':from_currency' => $from_currency,
            ':to_currency' => $to_currency,
            ':exchange_rate' => $exchange_rate,
            ':converted_amount' => $converted_amount,
            ':transfer_date' => $transfer_date,
            ':arrival_date' => $arrival_date,
            ':notes' => $notes
        ]);
        
        echo json_encode(['success' => true, 'id' => $transfer_id]);
        break;
        
    case 'DELETE':
        // Delete transfer
        $transfer_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$transfer_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Transfer ID is required']);
            break;
        }
        
        // Get transfer with account information
        $stmt = $conn->prepare("
            SELECT ft.id, ft.user_id, ft.from_account_id, ft.to_account_id,
                   fa.user_id as from_account_owner_id,
                   ta.user_id as to_account_owner_id
            FROM fund_transfers ft
            JOIN accounts fa ON ft.from_account_id = fa.id
            JOIN accounts ta ON ft.to_account_id = ta.id
            WHERE ft.id = :id
        ");
        $stmt->execute([':id' => $transfer_id]);
        $transfer = $stmt->fetch();
        
        if (!$transfer) {
            http_response_code(404);
            echo json_encode(['error' => 'Transfer not found']);
            break;
        }
        
        // Check if user has write access to both accounts
        // User must either:
        // 1. Own both accounts, OR
        // 2. Have write access to both accounts via sharing
        
        $from_account_id = $transfer['from_account_id'];
        $to_account_id = $transfer['to_account_id'];
        $from_account_owner_id = $transfer['from_account_owner_id'];
        $to_account_owner_id = $transfer['to_account_owner_id'];
        
        // Check if user owns from_account
        $owns_from_account = ($from_account_owner_id == $user_id);
        
        // Check if user has write access to from_account via sharing
        $stmt = $conn->prepare("
            SELECT id FROM account_shares 
            WHERE account_id = :account_id 
            AND shared_with_user_id = :user_id 
            AND sharing_mode = 'write'
        ");
        $stmt->execute([':account_id' => $from_account_id, ':user_id' => $user_id]);
        $has_write_access_from = (bool)$stmt->fetch();
        
        // Check if user owns to_account
        $owns_to_account = ($to_account_owner_id == $user_id);
        
        // Check if user has write access to to_account via sharing
        $stmt = $conn->prepare("
            SELECT id FROM account_shares 
            WHERE account_id = :account_id 
            AND shared_with_user_id = :user_id 
            AND sharing_mode = 'write'
        ");
        $stmt->execute([':account_id' => $to_account_id, ':user_id' => $user_id]);
        $has_write_access_to = (bool)$stmt->fetch();
        
        // User must have write access to both accounts
        $can_modify_from = $owns_from_account || $has_write_access_from;
        $can_modify_to = $owns_to_account || $has_write_access_to;
        
        if (!$can_modify_from || !$can_modify_to) {
            http_response_code(403);
            echo json_encode([
                'error' => 'Non hai i permessi necessari per eliminare questo trasferimento. Devi avere accesso in scrittura ad entrambi gli account coinvolti. Contatta l\'utente che ha creato il trasferimento.',
                'requires_contact' => true
            ]);
            break;
        }
        
        // Delete all attachment files before deleting transfer
        $stmt = $conn->prepare("
            SELECT file_path 
            FROM transfer_attachments 
            WHERE transfer_id = :transfer_id AND file_path IS NOT NULL
        ");
        $stmt->execute([':transfer_id' => $transfer_id]);
        $attachments = $stmt->fetchAll();
        
        foreach ($attachments as $attachment) {
            if ($attachment['file_path']) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
            }
        }
        
        // Delete the transfer directory if it exists
        $transfer_dir = UPLOAD_DIR . 'transfers/' . $transfer_id . '/';
        if (is_dir($transfer_dir)) {
            // Remove all files in directory
            $files = glob($transfer_dir . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
            // Remove directory
            rmdir($transfer_dir);
        }
        
        // Delete transfer (attachments will be deleted via CASCADE)
        $stmt = $conn->prepare("DELETE FROM fund_transfers WHERE id = :id");
        $stmt->execute([':id' => $transfer_id]);
        
        echo json_encode(['success' => true]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error processing transfer: ' . $e->getMessage()]);
    error_log('Transfer API error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
    error_log('Transfer API fatal error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
}

function fetchExchangeRate($from, $to) {
    try {
        $url = EXCHANGE_RATE_API_URL . $from;
        $response = file_get_contents($url);
        $data = json_decode($response, true);
        
        if (isset($data['rates'][$to])) {
            return floatval($data['rates'][$to]);
        }
    } catch (Exception $e) {
        error_log("Exchange rate fetch error: " . $e->getMessage());
    }
    
    return 0;
}

