<?php
/**
 * Transfer Attachments API
 */

// Start output buffering
ob_start();

// Disable error display
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled
ini_set('display_errors', 0);

// Clear any output
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();
    
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'POST':
            // Add attachment to transfer
            $transfer_id = isset($_POST['transfer_id']) ? (int)$_POST['transfer_id'] : null;
            $type = isset($_POST['type']) ? $_POST['type'] : null;
            
            if (!$transfer_id || !$type) {
                http_response_code(400);
                echo json_encode(['error' => 'Transfer ID and type are required']);
                break;
            }
            
            // Verify transfer belongs to user
            $stmt = $conn->prepare("SELECT id FROM fund_transfers WHERE id = :transfer_id AND user_id = :user_id");
            $stmt->execute([':transfer_id' => $transfer_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Transfer not found']);
                break;
            }
            
            $file_path = null;
            $link_url = null;
            $link_text = null;
            
            if ($type === 'LINK') {
                $link_url = $_POST['link_url'] ?? null;
                $link_text = $_POST['link_text'] ?? $link_url;
                
                if (!$link_url) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Link URL is required']);
                    break;
                }
            } else {
                // Handle file upload
                if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File upload required']);
                    break;
                }
                
                $file = $_FILES['file'];
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'transfers/' . $transfer_id . '/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path_full = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path_full)) {
                    $file_path = 'transfers/' . $transfer_id . '/' . $filename;
                } else {
                    http_response_code(500);
                    echo json_encode(['error' => 'Failed to upload file']);
                    break;
                }
            }
            
            $stmt = $conn->prepare("
                INSERT INTO transfer_attachments (transfer_id, type, file_path, link_url, link_text)
                VALUES (:transfer_id, :type, :file_path, :link_url, :link_text)
            ");
            
            $stmt->execute([
                ':transfer_id' => $transfer_id,
                ':type' => $type,
                ':file_path' => $file_path,
                ':link_url' => $link_url,
                ':link_text' => $link_text
            ]);
            
            echo json_encode(['success' => true, 'id' => $conn->lastInsertId()]);
            break;
            
        case 'DELETE':
            // Delete attachment
            $attachment_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
            
            if (!$attachment_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Attachment ID is required']);
                break;
            }
            
            // Get attachment and verify transfer belongs to user
            $stmt = $conn->prepare("
                SELECT ta.id, ta.file_path, ft.user_id
                FROM transfer_attachments ta
                JOIN fund_transfers ft ON ta.transfer_id = ft.id
                WHERE ta.id = :attachment_id AND ft.user_id = :user_id
            ");
            $stmt->execute([':attachment_id' => $attachment_id, ':user_id' => $user_id]);
            $attachment = $stmt->fetch();
            
            if (!$attachment) {
                http_response_code(404);
                echo json_encode(['error' => 'Attachment not found']);
                break;
            }
            
            // Delete file if exists
            if ($attachment['file_path']) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
            }
            
            // Delete attachment record
            $stmt = $conn->prepare("DELETE FROM transfer_attachments WHERE id = :id");
            $stmt->execute([':id' => $attachment_id]);
            
            echo json_encode(['success' => true]);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error: ' . $e->getMessage()]);
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
}

