<?php
/**
 * Securities API
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        // Get securities - filter by account_id if provided
        $account_id = $_GET['account_id'] ?? null;
        
        if ($account_id) {
            // Verify account belongs to user and is a securities account
            $stmt = $conn->prepare("
                SELECT a.id FROM accounts a
                JOIN account_types at ON a.type_id = at.id
                WHERE a.id = :account_id AND a.user_id = :user_id AND at.code = 'SECURITIES'
            ");
            $stmt->execute([':account_id' => $account_id, ':user_id' => $user_id]);
            if (!$stmt->fetch()) {
                http_response_code(404);
                echo json_encode(['error' => 'Securities account not found']);
                exit;
            }
            
            $stmt = $conn->prepare("
                SELECT s.*, COALESCE(s.transaction_type, 'PURCHASE') as transaction_type
                FROM securities s
                WHERE s.account_id = :account_id
                ORDER BY s.purchase_date DESC, s.symbol
            ");
            $stmt->execute([':account_id' => $account_id]);
        } else {
            // Get all securities for user
            $stmt = $conn->prepare("
                SELECT s.*, s.transaction_type, a.name as account_name
                FROM securities s
                JOIN accounts a ON s.account_id = a.id
                WHERE a.user_id = :user_id
                ORDER BY s.purchase_date DESC, s.symbol
            ");
            $stmt->execute([':user_id' => $user_id]);
        }
        
        $securities = $stmt->fetchAll();
        
        // Get attachments for each security
        foreach ($securities as &$security) {
            $stmt = $conn->prepare("SELECT * FROM securities_attachments WHERE security_id = :security_id");
            $stmt->execute([':security_id' => $security['id']]);
            $security['attachments'] = $stmt->fetchAll();
        }
        
        echo json_encode($securities);
        break;
        
    case 'POST':
        // Create new security
        $data = json_decode(file_get_contents('php://input'), true);
        
        $required = ['account_id', 'symbol', 'quantity', 'purchase_price', 'purchase_date', 'currency'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                exit;
            }
        }
        
        // Verify account belongs to user and is a securities account
        $stmt = $conn->prepare("
            SELECT a.id FROM accounts a
            JOIN account_types at ON a.type_id = at.id
            WHERE a.id = :account_id AND a.user_id = :user_id AND at.code = 'SECURITIES'
        ");
        $stmt->execute([':account_id' => $data['account_id'], ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Securities account not found']);
            exit;
        }
        
        $transaction_type = $data['transaction_type'] ?? 'PURCHASE';
        
        $stmt = $conn->prepare("
            INSERT INTO securities (account_id, symbol, quantity, purchase_price, purchase_date, currency, transaction_type, notes)
            VALUES (:account_id, :symbol, :quantity, :purchase_price, :purchase_date, :currency, :transaction_type, :notes)
        ");
        
        $stmt->execute([
            ':account_id' => $data['account_id'],
            ':symbol' => $data['symbol'],
            ':quantity' => $data['quantity'],
            ':purchase_price' => $data['purchase_price'],
            ':purchase_date' => $data['purchase_date'],
            ':currency' => $data['currency'],
            ':transaction_type' => $transaction_type,
            ':notes' => $data['notes'] ?? null
        ]);
        
        $security_id = $conn->lastInsertId();
        
        echo json_encode(['success' => true, 'id' => $security_id]);
        break;
        
    case 'PUT':
        // Update existing security
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Security ID is required']);
            break;
        }
        
        $security_id = (int)$data['id'];
        
        // Verify security belongs to user
        $stmt = $conn->prepare("
            SELECT s.id FROM securities s
            JOIN accounts a ON s.account_id = a.id
            WHERE s.id = :id AND a.user_id = :user_id
        ");
        $stmt->execute([':id' => $security_id, ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Security not found']);
            break;
        }
        
        $required = ['account_id', 'symbol', 'quantity', 'purchase_price', 'purchase_date', 'currency'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        // Verify account belongs to user and is a securities account
        $stmt = $conn->prepare("
            SELECT a.id FROM accounts a
            JOIN account_types at ON a.type_id = at.id
            WHERE a.id = :account_id AND a.user_id = :user_id AND at.code = 'SECURITIES'
        ");
        $stmt->execute([':account_id' => $data['account_id'], ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Securities account not found']);
            break;
        }
        
        $transaction_type = $data['transaction_type'] ?? 'PURCHASE';
        
        $stmt = $conn->prepare("
            UPDATE securities 
            SET account_id = :account_id, symbol = :symbol, quantity = :quantity, 
                purchase_price = :purchase_price, purchase_date = :purchase_date, 
                currency = :currency, transaction_type = :transaction_type, notes = :notes
            WHERE id = :id
        ");
        
        $stmt->execute([
            ':id' => $security_id,
            ':account_id' => $data['account_id'],
            ':symbol' => $data['symbol'],
            ':quantity' => $data['quantity'],
            ':purchase_price' => $data['purchase_price'],
            ':purchase_date' => $data['purchase_date'],
            ':currency' => $data['currency'],
            ':transaction_type' => $transaction_type,
            ':notes' => $data['notes'] ?? null
        ]);
        
        echo json_encode(['success' => true, 'id' => $security_id]);
        break;
        
    case 'DELETE':
        // Delete security
        $security_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$security_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Security ID is required']);
            break;
        }
        
        // Verify security belongs to user
        $stmt = $conn->prepare("
            SELECT s.id FROM securities s
            JOIN accounts a ON s.account_id = a.id
            WHERE s.id = :id AND a.user_id = :user_id
        ");
        $stmt->execute([':id' => $security_id, ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Security not found']);
            break;
        }
        
        // Delete all attachment files before deleting security
        $stmt = $conn->prepare("
            SELECT file_path 
            FROM securities_attachments 
            WHERE security_id = :security_id AND file_path IS NOT NULL
        ");
        $stmt->execute([':security_id' => $security_id]);
        $attachments = $stmt->fetchAll();
        
        foreach ($attachments as $attachment) {
            if ($attachment['file_path']) {
                $file_path = UPLOAD_DIR . $attachment['file_path'];
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
            }
        }
        
        // Delete the security directory if it exists
        $security_dir = UPLOAD_DIR . 'securities/' . $security_id . '/';
        if (is_dir($security_dir)) {
            // Remove all files in directory
            $files = glob($security_dir . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
            // Remove directory
            rmdir($security_dir);
        }
        
        // Delete security
        $stmt = $conn->prepare("DELETE FROM securities WHERE id = :id");
        $stmt->execute([':id' => $security_id]);
        
        echo json_encode(['success' => true]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}

