<?php
/**
 * Prepaid Cards API
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$card_id = isset($_GET['id']) ? intval($_GET['id']) : null;

switch ($method) {
    case 'GET':
        // Get all prepaid cards for user (owned + shared, optionally filter by enabled status and archived status)
        $enabled_only = isset($_GET['enabled_only']) && $_GET['enabled_only'] === 'true';
        $archived_only = isset($_GET['archived_only']) && $_GET['archived_only'] === 'true';
        
        $query = "
            SELECT 
                pc.*,
                NULL as sharing_mode,
                NULL as owner_name,
                NULL as owner_email,
                NULL as share_id,
                'owned' as card_status,
                CASE WHEN EXISTS (
                    SELECT 1 FROM prepaid_card_shares pcs WHERE pcs.prepaid_card_id = pc.id
                    UNION
                    SELECT 1 FROM prepaid_card_share_invitations pcsi WHERE pcsi.prepaid_card_id = pc.id
                ) THEN 1 ELSE 0 END as has_shares
            FROM prepaid_cards pc
            WHERE pc.user_id = :user_id1
        ";
        if ($enabled_only) {
            $query .= " AND pc.enabled = 1";
        }
        if ($archived_only) {
            $query .= " AND pc.archived = 1";
        } else {
            // By default, exclude archived cards from normal view
            $query .= " AND pc.archived = 0";
        }
        
        $query .= "
            UNION ALL
            
            SELECT 
                pc.*,
                'write' as sharing_mode,
                owner.name as owner_name,
                owner.email as owner_email,
                pcs.id as share_id,
                'shared' as card_status,
                0 as has_shares
            FROM prepaid_card_shares pcs
            JOIN prepaid_cards pc ON pcs.prepaid_card_id = pc.id
            JOIN users owner ON pcs.owner_user_id = owner.id
            WHERE pcs.shared_with_user_id = :user_id2
        ";
        if ($enabled_only) {
            $query .= " AND pc.enabled = 1";
        }
        if ($archived_only) {
            $query .= " AND pc.archived = 1";
        } else {
            // By default, exclude archived cards from normal view
            $query .= " AND pc.archived = 0";
        }
        
        $query .= " ORDER BY card_status DESC, name";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([':user_id1' => $user_id, ':user_id2' => $user_id]);
        $cards = $stmt->fetchAll();
        
        // Calculate remaining balance for each card
        foreach ($cards as &$card) {
            $nominal_amount = $card['nominal_amount'] ? floatval($card['nominal_amount']) : 0;
            
            // Sum all expenses for this prepaid card
            $stmt_expenses = $conn->prepare("
                SELECT COALESCE(SUM(amount), 0) as total_expenses
                FROM expenses
                WHERE prepaid_card_id = :prepaid_card_id
            ");
            $stmt_expenses->execute([':prepaid_card_id' => $card['id']]);
            $expenses_result = $stmt_expenses->fetch();
            $total_expenses = floatval($expenses_result['total_expenses']);
            
            // Remaining balance = nominal amount + expenses (expenses are already signed: negative for expenses, positive for income)
            $card['remaining_balance'] = $nominal_amount + $total_expenses;
        }
        unset($card); // Break reference
        
        echo json_encode($cards);
        break;
        
    case 'POST':
        // Create or update prepaid card - handle both JSON and FormData (for file uploads)
        // POST is used for both create and update when FormData is involved (because PUT doesn't populate $_POST)
        $data = [];
        $photo_path = null;
        $is_update = false;
        
        // Check if it's FormData or JSON
        $content_type = $_SERVER['CONTENT_TYPE'] ?? '';
        $is_multipart = strpos($content_type, 'multipart/form-data') !== false || !empty($_POST) || !empty($_FILES);
        
        if ($is_multipart) {
            // Handle FormData request
            $data = $_POST;
            
            // Check if this is an update (has 'id' field)
            if (!empty($data['id'])) {
                $is_update = true;
                $card_id = intval($data['id']);
                
                // Verify card belongs to user
                $stmt = $conn->prepare("SELECT id FROM prepaid_cards WHERE id = :card_id AND user_id = :user_id");
                $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
                if (!$stmt->fetch()) {
                    http_response_code(404);
                    echo json_encode(['error' => 'Card not found']);
                    break;
                }
            }
            
            // Handle photo upload if provided
            if (isset($_FILES['photo']) && $_FILES['photo']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['photo'];
                
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'prepaid_cards/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Delete old photo if updating
                if ($is_update) {
                    $stmt = $conn->prepare("SELECT photo_path FROM prepaid_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $old_card = $stmt->fetch();
                    if ($old_card && $old_card['photo_path']) {
                        $old_file_path = UPLOAD_DIR . $old_card['photo_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $photo_path = 'prepaid_cards/' . $filename;
                }
            } else if ($is_update) {
                // Get current photo path if not uploading new one
                $remove_photo = isset($data['remove_photo']) && ($data['remove_photo'] === true || $data['remove_photo'] === 'on');
                
                if ($remove_photo) {
                    // Delete old photo
                    $stmt = $conn->prepare("SELECT photo_path FROM prepaid_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $old_card = $stmt->fetch();
                    if ($old_card && $old_card['photo_path']) {
                        $old_file_path = UPLOAD_DIR . $old_card['photo_path'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                    $photo_path = null;
                } else {
                    // Keep current photo
                    $stmt = $conn->prepare("SELECT photo_path FROM prepaid_cards WHERE id = :card_id");
                    $stmt->execute([':card_id' => $card_id]);
                    $current_card = $stmt->fetch();
                    $photo_path = $current_card['photo_path'] ?? null;
                }
            }
        } else {
            // Handle JSON request
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
        }
        
        $required = ['name', 'base_currency', 'opening_date', 'expire_date'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || $data[$field] === '') {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        $nominal_amount = null;
        if (isset($data['nominal_amount']) && $data['nominal_amount'] !== '' && $data['nominal_amount'] !== null) {
            $nominal_amount = floatval($data['nominal_amount']);
        }
        
        // Handle card_number: convert empty string to NULL
        $card_number = null;
        if (isset($data['card_number']) && $data['card_number'] !== '' && $data['card_number'] !== null && trim($data['card_number']) !== '') {
            $card_number = trim($data['card_number']);
        }
        
        if ($is_update) {
            // Update existing card
            $enabled = isset($data['enabled']) ? ($data['enabled'] === true || $data['enabled'] === 'true' || $data['enabled'] === '1' || $data['enabled'] === 1) : true;
            
            $stmt = $conn->prepare("
                UPDATE prepaid_cards 
                SET name = :name, 
                    card_number = :card_number, 
                    base_currency = :base_currency, 
                    nominal_amount = :nominal_amount, 
                    opening_date = :opening_date, 
                    expire_date = :expire_date, 
                    enabled = :enabled,
                    photo_path = :photo_path
                WHERE id = :card_id AND user_id = :user_id
            ");
            
            $stmt->execute([
                ':card_id' => $card_id,
                ':user_id' => $user_id,
                ':name' => $data['name'],
                ':card_number' => $card_number,
                ':base_currency' => $data['base_currency'],
                ':nominal_amount' => $nominal_amount,
                ':opening_date' => $data['opening_date'],
                ':expire_date' => $data['expire_date'],
                ':enabled' => $enabled ? 1 : 0,
                ':photo_path' => $photo_path
            ]);
            
            echo json_encode(['success' => true]);
        } else {
            // Create new card
            $enabled = isset($data['enabled']) ? ($data['enabled'] === true || $data['enabled'] === 'true' || $data['enabled'] === '1' || $data['enabled'] === 1) : true;
            
            $stmt = $conn->prepare("
                INSERT INTO prepaid_cards (user_id, name, card_number, base_currency, nominal_amount, opening_date, expire_date, enabled, photo_path)
                VALUES (:user_id, :name, :card_number, :base_currency, :nominal_amount, :opening_date, :expire_date, :enabled, :photo_path)
            ");
            
            $stmt->execute([
                ':user_id' => $user_id,
                ':name' => $data['name'],
                ':card_number' => $card_number,
                ':base_currency' => $data['base_currency'],
                ':nominal_amount' => $nominal_amount,
                ':opening_date' => $data['opening_date'],
                ':expire_date' => $data['expire_date'],
                ':enabled' => $enabled ? 1 : 0,
                ':photo_path' => $photo_path
            ]);
            
            $card_id = $conn->lastInsertId();
            
            echo json_encode(['success' => true, 'id' => $card_id]);
        }
        break;
        
    case 'PUT':
        // Update prepaid card
        if (!$card_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Card ID is required']);
            break;
        }
        
        // Verify card belongs to user
        $stmt = $conn->prepare("SELECT id FROM prepaid_cards WHERE id = :card_id AND user_id = :user_id");
        $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Card not found']);
            break;
        }
        
        // Handle FormData (for file uploads) or JSON
        // Check Content-Type to determine if it's FormData or JSON
        $content_type = $_SERVER['CONTENT_TYPE'] ?? '';
        $is_multipart = strpos($content_type, 'multipart/form-data') !== false || !empty($_POST) || !empty($_FILES);
        
        if ($is_multipart) {
            // FormData was sent - for PUT requests, $_POST might not be populated automatically
            // Try $_POST first, then $_REQUEST as fallback
            if (!empty($_POST)) {
                $data = $_POST;
            } else if (!empty($_REQUEST)) {
                $data = $_REQUEST;
            } else {
                // For PUT with FormData, PHP doesn't populate $_POST automatically
                // We need to parse it manually or return an error
                http_response_code(400);
                echo json_encode(['error' => 'FormData fields not received. Please ensure all fields are included in FormData.']);
                break;
            }
        } else {
            // JSON was sent
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
        }
        
        $required = ['name', 'base_currency', 'opening_date', 'expire_date'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || $data[$field] === '') {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        $photo_path = null;
        $remove_photo = isset($data['remove_photo']) && $data['remove_photo'] === true;
        
        // Handle photo upload if provided
        if (isset($_FILES['photo']) && $_FILES['photo']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['photo'];
            
            if ($file['size'] > MAX_UPLOAD_SIZE) {
                http_response_code(400);
                echo json_encode(['error' => 'File too large']);
                break;
            }
            
            $upload_dir = UPLOAD_DIR . 'prepaid_cards/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = uniqid() . '.' . $extension;
            $file_path = $upload_dir . $filename;
            
            if (move_uploaded_file($file['tmp_name'], $file_path)) {
                $photo_path = 'prepaid_cards/' . $filename;
            }
        }
        
        // Get current photo path if not removing and not uploading new one
        if (!$remove_photo && !$photo_path) {
            $stmt = $conn->prepare("SELECT photo_path FROM prepaid_cards WHERE id = :card_id");
            $stmt->execute([':card_id' => $card_id]);
            $current_card = $stmt->fetch();
            $photo_path = $current_card['photo_path'];
        }
        
        // Delete old photo if removing or uploading new one
        if ($remove_photo || ($photo_path && isset($_FILES['photo']))) {
            $stmt = $conn->prepare("SELECT photo_path FROM prepaid_cards WHERE id = :card_id");
            $stmt->execute([':card_id' => $card_id]);
            $old_card = $stmt->fetch();
            if ($old_card && $old_card['photo_path']) {
                $old_file_path = UPLOAD_DIR . $old_card['photo_path'];
                if (file_exists($old_file_path)) {
                    unlink($old_file_path);
                }
            }
        }
        
        $nominal_amount = null;
        if (isset($data['nominal_amount']) && $data['nominal_amount'] !== '' && $data['nominal_amount'] !== null) {
            $nominal_amount = floatval($data['nominal_amount']);
        }
        
        $stmt = $conn->prepare("
            UPDATE prepaid_cards 
            SET name = :name, 
                card_number = :card_number, 
                base_currency = :base_currency, 
                nominal_amount = :nominal_amount, 
                opening_date = :opening_date, 
                expire_date = :expire_date, 
                photo_path = :photo_path
            WHERE id = :card_id AND user_id = :user_id
        ");
        
        $stmt->execute([
            ':card_id' => $card_id,
            ':user_id' => $user_id,
            ':name' => $data['name'],
            ':card_number' => $data['card_number'] ?? null,
            ':base_currency' => $data['base_currency'],
            ':nominal_amount' => $nominal_amount,
            ':opening_date' => $data['opening_date'],
            ':expire_date' => $data['expire_date'],
            ':photo_path' => $photo_path
        ]);
        
        echo json_encode(['success' => true]);
        break;
        
    case 'DELETE':
        // Delete or archive prepaid card
        if (!$card_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Card ID is required']);
            break;
        }
        
        // Verify card belongs to user and check if it has associated expenses
        $stmt = $conn->prepare("
            SELECT pc.photo_path, 
                   (SELECT COUNT(*) FROM expenses WHERE prepaid_card_id = pc.id) as expense_count
            FROM prepaid_cards pc 
            WHERE pc.id = :card_id AND pc.user_id = :user_id
        ");
        $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
        $card = $stmt->fetch();
        
        if (!$card) {
            http_response_code(404);
            echo json_encode(['error' => 'Card not found']);
            break;
        }
        
        $expense_count = intval($card['expense_count']);
        
        if ($expense_count > 0) {
            // Card has expenses - archive it instead of deleting
            $stmt = $conn->prepare("UPDATE prepaid_cards SET archived = 1 WHERE id = :card_id AND user_id = :user_id");
            $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
            
            echo json_encode(['success' => true, 'archived' => true, 'message' => 'Carta archiviata perché ha movimenti associati']);
        } else {
            // No expenses - delete the card completely
            // Delete photo file if exists
            if ($card['photo_path']) {
                $photo_file_path = UPLOAD_DIR . $card['photo_path'];
                if (file_exists($photo_file_path)) {
                    unlink($photo_file_path);
                }
            }
            
            // Delete card
            $stmt = $conn->prepare("DELETE FROM prepaid_cards WHERE id = :card_id AND user_id = :user_id");
            $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
            
            echo json_encode(['success' => true, 'archived' => false, 'message' => 'Carta eliminata']);
        }
        break;
        
    case 'PATCH':
        // Archive or unarchive prepaid card
        if (!$card_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Card ID is required']);
            break;
        }
        
        $raw_input = file_get_contents('php://input');
        $data = json_decode($raw_input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
            break;
        }
        
        if (!isset($data['archived'])) {
            http_response_code(400);
            echo json_encode(['error' => 'archived field is required']);
            break;
        }
        
        // Verify card belongs to user
        $stmt = $conn->prepare("SELECT id FROM prepaid_cards WHERE id = :card_id AND user_id = :user_id");
        $stmt->execute([':card_id' => $card_id, ':user_id' => $user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Card not found']);
            break;
        }
        
        $archived = ($data['archived'] === true || $data['archived'] === 'true' || $data['archived'] === 1 || $data['archived'] === '1');
        
        $stmt = $conn->prepare("UPDATE prepaid_cards SET archived = :archived WHERE id = :card_id AND user_id = :user_id");
        $stmt->execute([
            ':card_id' => $card_id,
            ':user_id' => $user_id,
            ':archived' => $archived ? 1 : 0
        ]);
        
        echo json_encode(['success' => true, 'archived' => $archived]);
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}

