<?php
/**
 * Notifications API
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$user_id = requireAuth();
$db = new Database();
$conn = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$notification_id = $_GET['id'] ?? null;

switch ($method) {
    case 'GET':
        // Get notifications for current user
        $unread_only = isset($_GET['unread_only']) && $_GET['unread_only'] === 'true';
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : null;
        
        $sql = "SELECT id, type, title, message, related_entity_type, related_entity_id, is_read, created_at, read_at
                FROM notifications
                WHERE user_id = :user_id";
        
        if ($unread_only) {
            $sql .= " AND is_read = FALSE";
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        if ($limit) {
            $sql .= " LIMIT :limit";
        }
        
        $stmt = $conn->prepare($sql);
        $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        if ($limit) {
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        }
        $stmt->execute();
        
        $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get unread count
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM notifications WHERE user_id = :user_id AND is_read = FALSE");
        $stmt->execute([':user_id' => $user_id]);
        $unread_count = $stmt->fetch()['count'];
        
        echo json_encode([
            'notifications' => $notifications,
            'unread_count' => (int)$unread_count
        ]);
        break;
        
    case 'PUT':
        // Mark notification(s) as read
        $data = json_decode(file_get_contents('php://input'), true);
        
        if ($notification_id) {
            // Mark single notification as read
            $mark_read = isset($data['is_read']) ? (bool)$data['is_read'] : true;
            
            $stmt = $conn->prepare("
                UPDATE notifications 
                SET is_read = :is_read, read_at = " . ($mark_read ? "NOW()" : "NULL") . "
                WHERE id = :id AND user_id = :user_id
            ");
            $stmt->execute([
                ':id' => $notification_id,
                ':user_id' => $user_id,
                ':is_read' => $mark_read ? 1 : 0
            ]);
            
            if ($stmt->rowCount() > 0) {
                echo json_encode(['success' => true, 'message' => 'Notification updated']);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Notification not found']);
            }
        } else if (isset($data['mark_all_read']) && $data['mark_all_read'] === true) {
            // Mark all notifications as read
            $stmt = $conn->prepare("
                UPDATE notifications 
                SET is_read = TRUE, read_at = NOW()
                WHERE user_id = :user_id AND is_read = FALSE
            ");
            $stmt->execute([':user_id' => $user_id]);
            
            echo json_encode([
                'success' => true,
                'message' => 'All notifications marked as read',
                'updated_count' => $stmt->rowCount()
            ]);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid request']);
        }
        break;
        
    case 'DELETE':
        // Delete notification(s)
        if ($notification_id) {
            // Delete single notification
            $stmt = $conn->prepare("DELETE FROM notifications WHERE id = :id AND user_id = :user_id");
            $stmt->execute([
                ':id' => $notification_id,
                ':user_id' => $user_id
            ]);
            
            if ($stmt->rowCount() > 0) {
                echo json_encode(['success' => true, 'message' => 'Notification deleted']);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Notification not found']);
            }
        } else if (isset($_GET['delete_all_read']) && $_GET['delete_all_read'] === 'true') {
            // Delete all read notifications
            $stmt = $conn->prepare("DELETE FROM notifications WHERE user_id = :user_id AND is_read = TRUE");
            $stmt->execute([':user_id' => $user_id]);
            
            echo json_encode([
                'success' => true,
                'message' => 'All read notifications deleted',
                'deleted_count' => $stmt->rowCount()
            ]);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid request']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}
