<?php
/**
 * Check Prepaid Card Expirations and Create Notifications
 * This endpoint should be called periodically (e.g., via cron) to check for expiring cards
 * and create notifications based on each user's expiration_notification_days setting
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/notifications_helper.php';

header('Content-Type: application/json');

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get all users with their expiration_notification_days setting
    $stmt = $conn->prepare("
        SELECT id, expiration_notification_days 
        FROM users 
        WHERE expiration_notification_days > 0
    ");
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $notifications_created = 0;
    $cards_checked = 0;
    
    foreach ($users as $user) {
        $user_id = $user['id'];
        $days_ahead = (int)$user['expiration_notification_days'];
        
        // Calculate the expiration date threshold (today + days_ahead)
        $threshold_date = date('Y-m-d', strtotime("+{$days_ahead} days"));
        
        // Get all prepaid cards that expire on or before the threshold date
        // Include cards owned by the user and cards shared with the user
        $stmt = $conn->prepare("
            SELECT DISTINCT pc.id, pc.name, pc.expire_date, pc.user_id as owner_id
            FROM prepaid_cards pc
            LEFT JOIN prepaid_card_shares pcs ON pc.id = pcs.prepaid_card_id
            WHERE pc.expire_date IS NOT NULL
            AND pc.expire_date <= :threshold_date
            AND pc.expire_date >= CURDATE()
            AND pc.archived = 0
            AND (
                pc.user_id = :user_id 
                OR pcs.shared_with_user_id = :user_id
            )
            AND NOT EXISTS (
                SELECT 1 FROM notifications n
                WHERE n.user_id = :user_id
                AND n.type = 'card_expiring'
                AND n.related_entity_type = 'prepaid_card'
                AND n.related_entity_id = pc.id
                AND DATE(n.created_at) = CURDATE()
            )
        ");
        $stmt->execute([
            ':user_id' => $user_id,
            ':threshold_date' => $threshold_date
        ]);
        $expiring_cards = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($expiring_cards as $card) {
            $cards_checked++;
            
            // Calculate days until expiration
            $expire_date = new DateTime($card['expire_date']);
            $today = new DateTime();
            $days_until_expiration = $today->diff($expire_date)->days;
            
            // Create notification
            $days_text = $days_until_expiration == 0 ? 'oggi' : 
                        ($days_until_expiration == 1 ? 'domani' : "tra {$days_until_expiration} giorni");
            
            $title = 'Carta prepagata in scadenza';
            $message = "La carta prepagata \"{$card['name']}\" scade {$days_text} ({$card['expire_date']})";
            
            if (createNotification(
                $conn,
                $user_id,
                'card_expiring',
                $title,
                $message,
                'prepaid_card',
                $card['id']
            )) {
                $notifications_created++;
            }
        }
    }
    
    echo json_encode([
        'success' => true,
        'users_checked' => count($users),
        'cards_checked' => $cards_checked,
        'notifications_created' => $notifications_created,
        'message' => "Checked {$cards_checked} cards, created {$notifications_created} notifications"
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Server error: ' . $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    error_log('Check expirations error: ' . $e->getMessage());
}
