<?php
/**
 * Database Session Handler
 * 
 * Custom session handler that stores sessions in the database instead of files.
 * This makes sessions independent from PHP file-based sessions and system cleanup timers.
 */

require_once __DIR__ . '/../../config/database.php';

class DatabaseSessionHandler implements SessionHandlerInterface {
    private $db;
    private $conn;
    
    public function __construct() {
        try {
            $this->db = new Database();
            $this->conn = $this->db->getConnection();
            
            // Verify table exists
            if (defined('SESSION_DEBUG_ENABLED') && SESSION_DEBUG_ENABLED) {
                $tableCheck = $this->conn->query("SHOW TABLES LIKE 'user_sessions'");
                if ($tableCheck->rowCount() === 0) {
                    error_log("DatabaseSessionHandler::__construct - WARNING: Table user_sessions does not exist. Please run migration: database/migration_add_user_sessions.sql");
                } else {
                    error_log("DatabaseSessionHandler::__construct - Database session handler initialized successfully");
                }
            }
        } catch (Exception $e) {
            error_log("DatabaseSessionHandler::__construct error: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Initialize session
     */
    public function open($save_path, $session_name) {
        return true;
    }
    
    /**
     * Close session
     */
    public function close() {
        return true;
    }
    
    /**
     * Read session data
     */
    public function read($session_id) {
        try {
            // Check if table exists first
            $tableCheck = $this->conn->query("SHOW TABLES LIKE 'user_sessions'");
            if ($tableCheck->rowCount() === 0) {
                error_log("DatabaseSessionHandler::read - Table user_sessions does not exist. Please run migration.");
                return '';
            }
            
            $stmt = $this->conn->prepare("
                SELECT data, expires_at, user_id 
                FROM user_sessions 
                WHERE session_id = :session_id 
                AND expires_at > NOW()
            ");
            $stmt->execute([':session_id' => $session_id]);
            $session = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($session) {
                // Update last_activity
                $updateStmt = $this->conn->prepare("
                    UPDATE user_sessions 
                    SET last_activity = NOW() 
                    WHERE session_id = :session_id
                ");
                $updateStmt->execute([':session_id' => $session_id]);
                
                if (defined('SESSION_DEBUG_ENABLED') && SESSION_DEBUG_ENABLED) {
                    error_log("DatabaseSessionHandler::read - Session found: " . substr($session_id, 0, 16) . "... (user_id: " . ($session['user_id'] ?? 'null') . ")");
                }
                
                return $session['data'] ?: '';
            }
            
            if (defined('SESSION_DEBUG_ENABLED') && SESSION_DEBUG_ENABLED) {
                error_log("DatabaseSessionHandler::read - Session not found or expired: " . substr($session_id, 0, 16) . "...");
            }
            
            return '';
        } catch (Exception $e) {
            error_log("DatabaseSessionHandler::read error: " . $e->getMessage());
            error_log("DatabaseSessionHandler::read stack trace: " . $e->getTraceAsString());
            return '';
        }
    }
    
    /**
     * Write session data
     */
    public function write($session_id, $session_data) {
        try {
            // Check if table exists first
            $tableCheck = $this->conn->query("SHOW TABLES LIKE 'user_sessions'");
            if ($tableCheck->rowCount() === 0) {
                error_log("DatabaseSessionHandler::write - Table user_sessions does not exist. Please run migration.");
                return false;
            }
            
            // Get user_id from session data if available
            // PHP serializes session data in format: "key1|s:5:"value1";key2|i:123;"
            $user_id = null;
            $first_login_at = null;
            
            if (!empty($session_data)) {
                // PHP uses serialize() format for session data
                // Format: "user_id|i:123;first_login_at|i:1234567890;"
                // Try to extract user_id and first_login_at from serialized data
                if (preg_match('/user_id\|i:(\d+)/', $session_data, $matches)) {
                    $user_id = (int)$matches[1];
                }
                if (preg_match('/first_login_at\|i:(\d+)/', $session_data, $matches)) {
                    $first_login_at = (int)$matches[1];
                }
            }
            
            // Calculate expiration: 7 days from now (sliding expiration)
            // But check for first_login_at in session data to respect 180-day absolute expiration
            $expires_at = date('Y-m-d H:i:s', time() + (7 * 24 * 60 * 60)); // 7 days default
            
            if ($first_login_at) {
                $absolute_expiration = $first_login_at + (180 * 24 * 60 * 60); // 180 days
                $sliding_expiration = time() + (7 * 24 * 60 * 60); // 7 days
                $expires_at = date('Y-m-d H:i:s', min($absolute_expiration, $sliding_expiration));
            }
            
            // Get IP and user agent
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;
            $user_agent = substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 255);
            
            // Build the query - only update user_id if it's not null (to preserve existing user_id)
            $stmt = $this->conn->prepare("
                INSERT INTO user_sessions (session_id, user_id, ip_address, user_agent, data, expires_at, last_activity)
                VALUES (:session_id, :user_id, :ip_address, :user_agent, :data, :expires_at, NOW())
                ON DUPLICATE KEY UPDATE
                    data = VALUES(data),
                    expires_at = VALUES(expires_at),
                    last_activity = NOW(),
                    ip_address = VALUES(ip_address),
                    user_agent = VALUES(user_agent),
                    user_id = COALESCE(VALUES(user_id), user_id)
            ");
            
            $result = $stmt->execute([
                ':session_id' => $session_id,
                ':user_id' => $user_id ?: null, // Explicitly set to NULL if empty
                ':ip_address' => $ip_address,
                ':user_agent' => $user_agent,
                ':data' => $session_data,
                ':expires_at' => $expires_at
            ]);
            
            if (defined('SESSION_DEBUG_ENABLED') && SESSION_DEBUG_ENABLED && $result) {
                error_log("DatabaseSessionHandler::write - Session written: " . substr($session_id, 0, 16) . "... (user_id: " . ($user_id ?? 'null') . ", expires_at: $expires_at)");
            }
            
            return $result !== false;
        } catch (Exception $e) {
            error_log("DatabaseSessionHandler::write error: " . $e->getMessage());
            error_log("DatabaseSessionHandler::write stack trace: " . $e->getTraceAsString());
            return false;
        }
    }
    
    /**
     * Destroy session
     */
    public function destroy($session_id) {
        try {
            $stmt = $this->conn->prepare("DELETE FROM user_sessions WHERE session_id = :session_id");
            return $stmt->execute([':session_id' => $session_id]);
        } catch (Exception $e) {
            error_log("DatabaseSessionHandler::destroy error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Garbage collection - clean up expired sessions
     */
    public function gc($maxlifetime) {
        try {
            $stmt = $this->conn->prepare("DELETE FROM user_sessions WHERE expires_at < NOW()");
            return $stmt->execute() !== false;
        } catch (Exception $e) {
            error_log("DatabaseSessionHandler::gc error: " . $e->getMessage());
            return false;
        }
    }
}
