<?php
/**
 * Fidelity Card Order API
 * Handles saving and retrieving the order of fidelity cards for each user
 */

ob_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

ini_set('display_errors', 0);
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
    case 'POST':
        // Save fidelity card order
        $raw_input = file_get_contents('php://input');
        $data = json_decode($raw_input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
            break;
        }
        
        if (!isset($data['card_orders']) || !is_array($data['card_orders'])) {
            http_response_code(400);
            echo json_encode(['error' => 'card_orders array is required']);
            break;
        }
        
        // Verify all cards belong to user or are shared with user
        $card_ids = array_column($data['card_orders'], 'fidelity_card_id');
        if (empty($card_ids)) {
            http_response_code(400);
            echo json_encode(['error' => 'No card IDs provided']);
            break;
        }
        
        // Use positional placeholders for verification query
        $verify_placeholders = str_repeat('?,', count($card_ids) - 1) . '?';
        $verify_params = array_merge($card_ids, [$user_id, $user_id]);
        
        // Verify user has access to all cards
        $stmt = $conn->prepare("
            SELECT fc.id
            FROM fidelity_cards fc
            WHERE fc.id IN ($verify_placeholders)
            AND (
                fc.user_id = ?
                OR EXISTS (
                    SELECT 1 FROM fidelity_card_shares fcs 
                    WHERE fcs.fidelity_card_id = fc.id 
                    AND fcs.shared_with_user_id = ?
                )
            )
        ");
        $stmt->execute($verify_params);
        $accessible_cards = $stmt->fetchAll();
        $accessible_card_ids = array_column($accessible_cards, 'id');
        
        if (count($accessible_card_ids) !== count($card_ids)) {
            http_response_code(403);
            echo json_encode(['error' => 'You do not have access to all specified cards']);
            break;
        }
        
        // Begin transaction
        $conn->beginTransaction();
        
        try {
            // Delete existing orders for these cards
            $delete_placeholders = str_repeat('?,', count($card_ids) - 1) . '?';
            $delete_params = array_merge([$user_id], $card_ids);
            
            $delete_query = "
                DELETE FROM user_fidelity_card_order 
                WHERE user_id = ? AND fidelity_card_id IN ($delete_placeholders)
            ";
            
            $delete_stmt = $conn->prepare($delete_query);
            $delete_stmt->execute($delete_params);
            
            // Insert new orders
            $stmt = $conn->prepare("
                INSERT INTO user_fidelity_card_order (user_id, fidelity_card_id, display_order)
                VALUES (:user_id, :fidelity_card_id, :display_order)
                ON DUPLICATE KEY UPDATE display_order = :display_order_update
            ");
            
            foreach ($data['card_orders'] as $order) {
                $stmt->execute([
                    ':user_id' => $user_id,
                    ':fidelity_card_id' => $order['fidelity_card_id'],
                    ':display_order' => $order['display_order'],
                    ':display_order_update' => $order['display_order']
                ]);
            }
            
            $conn->commit();
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            $conn->rollBack();
            throw $e;
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error processing fidelity card order: ' . $e->getMessage()]);
    error_log('Fidelity card order API error: ' . $e->getMessage());
}
