<?php
/**
 * Check if user is authenticated
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

header('Content-Type: application/json');

$sessionName = session_name();
$currentSessionId = session_id();
$cookieSessionId = $_COOKIE[$sessionName] ?? null;
$cookieReceived = isset($_COOKIE[$sessionName]);

// Check if session exists in database (for database session handler)
$sessionInDb = false;
$sessionDbExpiresAt = null;
try {
    $db = new Database();
    $conn = $db->getConnection();
    $stmt = $conn->prepare("
        SELECT expires_at 
        FROM user_sessions 
        WHERE session_id = :session_id 
        AND expires_at > NOW()
    ");
    $stmt->execute([':session_id' => $currentSessionId]);
    $dbSession = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($dbSession) {
        $sessionInDb = true;
        $sessionDbExpiresAt = $dbSession['expires_at'];
    }
} catch (Exception $e) {
    // Table might not exist yet, ignore error
    error_log("check_session: Could not check database session: " . $e->getMessage());
}

// Legacy: Check if session file exists on server (for file-based sessions)
$sessionFileExists = false;
$sessionFileAge = null;
$sessionSavePath = session_save_path();
if (empty($sessionSavePath)) {
    $sessionSavePath = sys_get_temp_dir();
}
$sessionFilePath = $sessionSavePath . '/sess_' . $currentSessionId;
if ($currentSessionId && file_exists($sessionFilePath)) {
    $sessionFileExists = true;
    $sessionFileAge = time() - filemtime($sessionFilePath);
}

// Check if cookie session ID matches current session ID
$sessionIdMismatch = false;
if ($cookieReceived && $cookieSessionId !== $currentSessionId) {
    $sessionIdMismatch = true;
}

logSessionDebug('check_session_called', [
    'has_session' => isset($_SESSION),
    'has_user_id' => isset($_SESSION['user_id']),
    'session_id' => $currentSessionId ?: 'no-session-id',
    'cookie_received' => $cookieReceived,
    'cookie_session_id' => $cookieSessionId ? substr($cookieSessionId, 0, 16) . '...' : null,
    'session_id_mismatch' => $sessionIdMismatch,
    'session_in_db' => $sessionInDb,
    'session_db_expires_at' => $sessionDbExpiresAt,
    'session_file_exists' => $sessionFileExists,
    'session_file_age_seconds' => $sessionFileAge,
    'session_file_age_hours' => $sessionFileAge ? round($sessionFileAge / 3600, 2) : null,
    'session_save_path' => $sessionSavePath,
    'session_gc_maxlifetime' => ini_get('session.gc_maxlifetime'),
    'session_cookie_lifetime' => ini_get('session.cookie_lifetime'),
    'session_count' => isset($_SESSION) ? count($_SESSION) : 0,
    'session_handler' => 'database'
]);

if (isset($_SESSION['user_id'])) {
    // Update session expiration (sliding expiration + check absolute expiration)
    // Don't exit on expiration, return authenticated: false instead
    if (!updateSessionExpiration($_SESSION['user_id'], false)) {
        // Session expired (absolute expiration reached)
        logSessionDebug('check_session_expired', [
            'reason' => 'Session expired during check',
            'user_id' => $_SESSION['user_id'] ?? 'unknown'
        ]);
        http_response_code(401);
        echo json_encode(['authenticated' => false]);
        exit;
    }
    
    // Get user admin status from database
    $db = new Database();
    $conn = $db->getConnection();
    $stmt = $conn->prepare("SELECT is_admin FROM users WHERE id = :user_id");
    $stmt->execute([':user_id' => $_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    logSessionDebug('check_session_success', [
        'user_id' => $_SESSION['user_id'],
        'is_admin' => $user ? (bool)$user['is_admin'] : false
    ]);
    
    echo json_encode([
        'authenticated' => true,
        'user_id' => $_SESSION['user_id'],
        'email' => $_SESSION['email'] ?? null,
        'name' => $_SESSION['name'] ?? null,
        'is_admin' => $user ? (bool)$user['is_admin'] : false
    ]);
} else {
    // Additional diagnostics for empty session
    $sessionIsNew = false;
    if ($cookieReceived && $cookieSessionId && !$sessionFileExists) {
        $sessionIsNew = true;
    }
    
    // Read session file content to see what's actually stored
    $sessionFileContent = null;
    $sessionFileSize = null;
    $sessionFileReadable = false;
    if ($sessionFileExists && $currentSessionId) {
        $sessionFileSize = filesize($sessionFilePath);
        if ($sessionFileSize > 0) {
            $sessionFileContent = @file_get_contents($sessionFilePath);
            $sessionFileReadable = ($sessionFileContent !== false);
            // Don't log full content for security, but log if it's empty or contains data
            if ($sessionFileContent !== false) {
                $sessionFileContent = strlen($sessionFileContent) > 0 ? 'has_content_' . strlen($sessionFileContent) . '_bytes' : 'empty';
            }
        } else {
            $sessionFileContent = 'file_empty';
        }
    }
    
    // Check file permissions
    $sessionFilePermissions = null;
    $sessionFileWritable = false;
    if ($sessionFileExists) {
        $sessionFilePermissions = substr(sprintf('%o', fileperms($sessionFilePath)), -4);
        $sessionFileWritable = is_writable($sessionFilePath);
    }
    
    logSessionDebug('check_session_no_user_id', [
        'reason' => 'No user_id in session',
        'session_exists' => isset($_SESSION),
        'session_keys' => isset($_SESSION) ? array_keys($_SESSION) : [],
        'cookie_name' => session_name(),
        'cookie_exists' => $cookieReceived,
        'cookie_session_id' => $cookieSessionId ? substr($cookieSessionId, 0, 16) . '...' : null,
        'current_session_id' => $currentSessionId ? substr($currentSessionId, 0, 16) . '...' : null,
        'session_id_mismatch' => $sessionIdMismatch,
        'session_file_exists' => $sessionFileExists,
        'session_file_age_seconds' => $sessionFileAge,
        'session_file_age_hours' => $sessionFileAge ? round($sessionFileAge / 3600, 2) : null,
        'session_file_size' => $sessionFileSize,
        'session_file_content' => $sessionFileContent,
        'session_file_readable' => $sessionFileReadable,
        'session_file_permissions' => $sessionFilePermissions,
        'session_file_writable' => $sessionFileWritable,
        'session_is_new' => $sessionIsNew,
        'diagnosis' => $sessionIsNew ? 'Cookie exists but session file was deleted (likely garbage collected)' : 
                      ($sessionIdMismatch ? 'Cookie session ID does not match current session ID' : 
                      ($sessionFileExists && $sessionFileSize === 0 ? 'Session file exists but is empty (0 bytes)' :
                      ($sessionFileExists && $sessionFileReadable && $sessionFileContent === 'empty' ? 'Session file exists but contains no data' :
                      'Session exists but is empty'))),
        'session_gc_maxlifetime' => ini_get('session.gc_maxlifetime'),
        'session_gc_maxlifetime_hours' => round(ini_get('session.gc_maxlifetime') / 3600, 2),
        'session_save_path_writable' => is_writable($sessionSavePath)
    ]);
    
    // If cookie exists but session file doesn't, destroy the cookie to force new login
    if ($sessionIsNew) {
        logSessionDebug('session_cleanup_empty_session', [
            'action' => 'Destroying empty session and cookie',
            'reason' => 'Cookie exists but session file was garbage collected'
        ]);
        // Destroy the session and clear the cookie
        session_destroy();
        // Clear the cookie by setting it to expire in the past
        $cookieParams = session_get_cookie_params();
        setcookie(
            $sessionName,
            '',
            time() - 3600,
            $cookieParams['path'],
            $cookieParams['domain'],
            $cookieParams['secure'],
            $cookieParams['httponly']
        );
    }
    
    http_response_code(401);
    echo json_encode(['authenticated' => false]);
}

