<?php
/**
 * Account Totals API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints to prevent HTML in JSON responses
// This must be set before including config.php which may enable display_errors
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    // Get all accounts with balances (exclude "No Account" and accounts with show_in_dashboard = false) - owned + shared
    $stmt = $conn->prepare("
        SELECT 
            a.id,
            a.name,
            a.base_currency,
            a.logo_path,
            at.code as type_code,
            at.name as type_name,
            NULL as sharing_mode,
            NULL as owner_name,
            NULL as owner_email,
            NULL as share_id,
            'owned' as account_status,
            CASE WHEN EXISTS (
                SELECT 1 FROM account_shares ash WHERE ash.account_id = a.id
                UNION
                SELECT 1 FROM account_share_invitations asi WHERE asi.account_id = a.id
            ) THEN 1 ELSE 0 END as has_shares
        FROM accounts a
        JOIN account_types at ON a.type_id = at.id
        WHERE a.user_id = :user_id1 
          AND a.name != 'No Account'
          AND COALESCE(a.show_in_dashboard, TRUE) = TRUE
        
        UNION ALL
        
        SELECT 
            a.id,
            a.name,
            a.base_currency,
            a.logo_path,
            at.code as type_code,
            at.name as type_name,
            ash.sharing_mode,
            owner.name as owner_name,
            owner.email as owner_email,
            ash.id as share_id,
            'shared' as account_status,
            0 as has_shares
        FROM account_shares ash
        JOIN accounts a ON ash.account_id = a.id
        JOIN account_types at ON a.type_id = at.id
        JOIN users owner ON ash.owner_user_id = owner.id
        WHERE ash.shared_with_user_id = :user_id2 
          AND a.name != 'No Account'
          AND COALESCE(a.show_in_dashboard, TRUE) = TRUE
    ");
    $stmt->execute([':user_id1' => $user_id, ':user_id2' => $user_id]);
    $accounts = $stmt->fetchAll();
    
    // Get account order for this user
    $stmt = $conn->prepare("
        SELECT account_id, display_order
        FROM user_account_order
        WHERE user_id = :user_id
    ");
    $stmt->execute([':user_id' => $user_id]);
    $account_orders = $stmt->fetchAll();
    
    // Create a map of account_id => display_order
    $order_map = [];
    foreach ($account_orders as $order) {
        $order_map[$order['account_id']] = intval($order['display_order']);
    }
    
    // Add display_order to each account and determine max order
    $max_order = 0;
    foreach ($accounts as &$account) {
        if (isset($order_map[$account['id']])) {
            $account['display_order'] = $order_map[$account['id']];
            $max_order = max($max_order, $account['display_order']);
        } else {
            // New account: assign order at the end
            $account['display_order'] = null;
        }
    }
    unset($account);
    
    // Sort accounts: first by display_order (nulls last), then by account_status DESC, then by name
    usort($accounts, function($a, $b) {
        // If both have display_order, sort by it
        if (isset($a['display_order']) && isset($b['display_order'])) {
            return $a['display_order'] <=> $b['display_order'];
        }
        // If only a has display_order, it comes first
        if (isset($a['display_order']) && !isset($b['display_order'])) {
            return -1;
        }
        // If only b has display_order, it comes first
        if (!isset($a['display_order']) && isset($b['display_order'])) {
            return 1;
        }
        // If neither has display_order, sort by account_status DESC, then name
        $status_order = ['owned' => 1, 'shared' => 0];
        $a_status = $status_order[$a['account_status']] ?? 0;
        $b_status = $status_order[$b['account_status']] ?? 0;
        if ($a_status !== $b_status) {
            return $b_status <=> $a_status;
        }
        return strcmp($a['name'], $b['name']);
    });
    
    // Assign order to new accounts (those without display_order)
    $current_order = $max_order + 1;
    foreach ($accounts as &$account) {
        if (!isset($account['display_order'])) {
            $account['display_order'] = $current_order++;
        }
    }
    unset($account);

    // Calculate balance for each account
    foreach ($accounts as &$account) {
        if ($account['type_code'] === 'SECURITIES') {
            // For securities accounts, calculate total value by currency
            // purchase_price is the total amount, so we sum it directly
            // Sales subtract from the total
            $stmt = $conn->prepare("
                SELECT 
                    currency,
                    SUM(CASE 
                        WHEN transaction_type = 'PURCHASE' OR transaction_type IS NULL THEN purchase_price 
                        WHEN transaction_type = 'SALE' THEN -purchase_price 
                        ELSE 0 
                    END) as total_value
                FROM securities
                WHERE account_id = :account_id
                GROUP BY currency
            ");
            $stmt->execute([':account_id' => $account['id']]);
            $securities_by_currency = $stmt->fetchAll();
            
            // Ensure total_value is a float
            foreach ($securities_by_currency as &$sec) {
                $sec['total_value'] = floatval($sec['total_value']);
            }
            unset($sec); // Break reference
            
            $account['securities_by_currency'] = $securities_by_currency;
            $account['balance'] = 0; // Will be calculated on frontend with currency conversion
            $account['expenses'] = 0;
            $account['transfers_in'] = 0;
            $account['transfers_out'] = 0;
        } else {
            // Expenses: stored as negative values for expenses, positive for income
            $stmt = $conn->prepare("
                SELECT COALESCE(SUM(amount), 0) as total
                FROM expenses
                WHERE account_id = :account_id
            ");
            $stmt->execute([':account_id' => $account['id']]);
            $expenses = $stmt->fetch();
            $total_expenses = floatval($expenses['total']);
            
            // Transfers out
            $stmt = $conn->prepare("
                SELECT COALESCE(SUM(amount), 0) as total
                FROM fund_transfers
                WHERE from_account_id = :account_id
            ");
            $stmt->execute([':account_id' => $account['id']]);
            $transfers_out = $stmt->fetch();
            $total_out = floatval($transfers_out['total']);
            
            // Transfers in (converted to account currency)
            $stmt = $conn->prepare("
                SELECT COALESCE(SUM(converted_amount), 0) as total
                FROM fund_transfers
                WHERE to_account_id = :account_id
            ");
            $stmt->execute([':account_id' => $account['id']]);
            $transfers_in = $stmt->fetch();
            $total_in = floatval($transfers_in['total']);
            
            // Expenses are already signed (negative for expenses, positive for income), so we add them
            $account['balance'] = $total_in - $total_out + $total_expenses;
            $account['expenses'] = $total_expenses;
            $account['transfers_in'] = $total_in;
            $account['transfers_out'] = $total_out;
            
            // Calculate period summaries (last 30 days, 180 days, 365 days)
            $now = new DateTime();
            
            // Last 30 days
            $last30_start = clone $now;
            $last30_start->modify('-30 days');
            $prev30_start = clone $last30_start;
            $prev30_start->modify('-30 days');
            
            // Last 180 days
            $last180_start = clone $now;
            $last180_start->modify('-180 days');
            $prev180_start = clone $last180_start;
            $prev180_start->modify('-180 days');
            
            // Last year (365 days)
            $last365_start = clone $now;
            $last365_start->modify('-365 days');
            $prev365_start = clone $last365_start;
            $prev365_start->modify('-365 days');
            
            // Helper function to calculate total movements for a period
            $calculatePeriodMovements = function($account_id, $start_date, $end_date) use ($conn) {
                // Expenses (already signed: negative for expenses, positive for income)
                $stmt = $conn->prepare("
                    SELECT COALESCE(SUM(amount), 0) as total
                    FROM expenses
                    WHERE account_id = :account_id
                    AND date >= :start_date
                    AND date <= :end_date
                ");
                $stmt->execute([
                    ':account_id' => $account_id,
                    ':start_date' => $start_date->format('Y-m-d'),
                    ':end_date' => $end_date->format('Y-m-d')
                ]);
                $expenses = $stmt->fetch();
                $total_expenses = floatval($expenses['total']);
                
                // Transfers out (negative)
                $stmt = $conn->prepare("
                    SELECT COALESCE(SUM(amount), 0) as total
                    FROM fund_transfers
                    WHERE from_account_id = :account_id
                    AND transfer_date >= :start_date
                    AND transfer_date <= :end_date
                ");
                $stmt->execute([
                    ':account_id' => $account_id,
                    ':start_date' => $start_date->format('Y-m-d'),
                    ':end_date' => $end_date->format('Y-m-d')
                ]);
                $transfers_out = $stmt->fetch();
                $total_out = floatval($transfers_out['total']);
                
                // Transfers in (positive, converted to account currency)
                $stmt = $conn->prepare("
                    SELECT COALESCE(SUM(converted_amount), 0) as total
                    FROM fund_transfers
                    WHERE to_account_id = :account_id
                    AND COALESCE(arrival_date, transfer_date) >= :start_date
                    AND COALESCE(arrival_date, transfer_date) <= :end_date
                ");
                $stmt->execute([
                    ':account_id' => $account_id,
                    ':start_date' => $start_date->format('Y-m-d'),
                    ':end_date' => $end_date->format('Y-m-d')
                ]);
                $transfers_in = $stmt->fetch();
                $total_in = floatval($transfers_in['total']);
                
                // Total movements: transfers_in - transfers_out + expenses
                return $total_in - $total_out + $total_expenses;
            };
            
            // Last 30 days
            $last30_total = $calculatePeriodMovements($account['id'], $last30_start, $now);
            $prev30_total = $calculatePeriodMovements($account['id'], $prev30_start, $last30_start);
            $account['last_month_total'] = $last30_total;
            $account['last_month_delta'] = $last30_total - $prev30_total;
            
            // Last 180 days
            $last180_total = $calculatePeriodMovements($account['id'], $last180_start, $now);
            $prev180_total = $calculatePeriodMovements($account['id'], $prev180_start, $last180_start);
            $account['last_semester_total'] = $last180_total;
            $account['last_semester_delta'] = $last180_total - $prev180_total;
            
            // Last year
            $last365_total = $calculatePeriodMovements($account['id'], $last365_start, $now);
            $prev365_total = $calculatePeriodMovements($account['id'], $prev365_start, $last365_start);
            $account['last_year_total'] = $last365_total;
            $account['last_year_delta'] = $last365_total - $prev365_total;
        }
    }
    
    echo json_encode($accounts);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error loading account totals: ' . $e->getMessage()]);
    exit;
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
    exit;
}

