<?php
/**
 * Account Order API
 * Handles saving and retrieving the order of accounts for dashboard display
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
    case 'POST':
        // Save account order
        $raw_input = file_get_contents('php://input');
        error_log('Account order API - Raw input: ' . $raw_input);
        $data = json_decode($raw_input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('Account order API - JSON decode error: ' . json_last_error_msg());
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
            break;
        }
        
        error_log('Account order API - Decoded data: ' . print_r($data, true));
        
        if (!isset($data['account_orders']) || !is_array($data['account_orders'])) {
            error_log('Account order API - Missing or invalid account_orders array');
            http_response_code(400);
            echo json_encode(['error' => 'account_orders array is required']);
            break;
        }
        
        // Verify all accounts belong to user or are shared with user
        $account_ids = array_column($data['account_orders'], 'account_id');
        if (empty($account_ids)) {
            http_response_code(400);
            echo json_encode(['error' => 'No account IDs provided']);
            break;
        }
        
        // Use positional placeholders for verification query
        $verify_placeholders = str_repeat('?,', count($account_ids) - 1) . '?';
        $verify_params = array_merge($account_ids, [$user_id, $user_id]);
        
        // Verify user has access to all accounts
        $stmt = $conn->prepare("
            SELECT a.id
            FROM accounts a
            WHERE a.id IN ($verify_placeholders)
            AND (
                a.user_id = ?
                OR EXISTS (
                    SELECT 1 FROM account_shares ash 
                    WHERE ash.account_id = a.id 
                    AND ash.shared_with_user_id = ?
                )
            )
        ");
        $stmt->execute($verify_params);
        $accessible_accounts = $stmt->fetchAll();
        $accessible_account_ids = array_column($accessible_accounts, 'id');
        
        if (count($accessible_account_ids) !== count($account_ids)) {
            http_response_code(403);
            echo json_encode(['error' => 'You do not have access to all specified accounts']);
            break;
        }
        
        // Begin transaction
        $conn->beginTransaction();
        
        try {
            // Delete existing orders for these accounts
            // Use positional placeholders for IN clause to avoid parameter issues
            $delete_placeholders = str_repeat('?,', count($account_ids) - 1) . '?';
            $delete_params = array_merge([$user_id], $account_ids);
            
            if (empty($account_ids)) {
                throw new Exception('No account IDs to delete');
            }
            
            $delete_query = "
                DELETE FROM user_account_order 
                WHERE user_id = ? AND account_id IN ($delete_placeholders)
            ";
            error_log('DELETE query: ' . $delete_query);
            error_log('DELETE query params: ' . print_r($delete_params, true));
            
            // Use a new statement variable to avoid conflicts
            $delete_stmt = $conn->prepare($delete_query);
            $result = $delete_stmt->execute($delete_params);
            
            if (!$result) {
                $error_info = $delete_stmt->errorInfo();
                error_log('DELETE query failed: ' . print_r($error_info, true));
                throw new Exception('DELETE query failed: ' . print_r($error_info, true));
            }
            
            error_log('DELETE query executed successfully, rows affected: ' . $delete_stmt->rowCount());
            
            // Insert new orders
            $stmt = $conn->prepare("
                INSERT INTO user_account_order (user_id, account_id, display_order)
                VALUES (:user_id, :account_id, :display_order)
                ON DUPLICATE KEY UPDATE display_order = :display_order_update
            ");
            
            foreach ($data['account_orders'] as $order) {
                error_log('Account order API - Inserting order: user_id=' . $user_id . ', account_id=' . $order['account_id'] . ', display_order=' . $order['display_order']);
                $result = $stmt->execute([
                    ':user_id' => $user_id,
                    ':account_id' => $order['account_id'],
                    ':display_order' => $order['display_order'],
                    ':display_order_update' => $order['display_order']
                ]);
                if (!$result) {
                    $error_info = $stmt->errorInfo();
                    error_log('Account order API - Insert failed: ' . print_r($error_info, true));
                    throw new Exception('Failed to insert account order: ' . print_r($error_info, true));
                }
            }
            
            $conn->commit();
            error_log('Account order API - Successfully saved ' . count($data['account_orders']) . ' account orders');
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            $conn->rollBack();
            throw $e;
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Error processing account order: ' . $e->getMessage()]);
    error_log('Account order API error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
} catch (Error $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error: ' . $e->getMessage()]);
    error_log('Account order API fatal error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
}

