<?php
/**
 * Accounts API
 */

// Start output buffering to catch any accidental output
ob_start();

// Disable error display for API endpoints to prevent HTML in JSON responses
// This must be set before including config.php which may enable display_errors
ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../middleware/auth.php';

// Ensure display_errors is still disabled after config includes
ini_set('display_errors', 0);

// Clear any output that might have been generated
ob_clean();

header('Content-Type: application/json');

try {
    $user_id = requireAuth();
    $db = new Database();
    $conn = $db->getConnection();

    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
    case 'GET':
        // Get all accounts for user (owned + shared)
        $stmt = $conn->prepare("
            SELECT 
                a.*, 
                at.code as type_code, 
                at.name as type_name,
                NULL as sharing_mode,
                NULL as owner_name,
                NULL as owner_email,
                NULL as share_id,
                'owned' as account_status,
                CASE WHEN EXISTS (
                    SELECT 1 FROM account_shares ash WHERE ash.account_id = a.id
                    UNION
                    SELECT 1 FROM account_share_invitations asi WHERE asi.account_id = a.id
                ) THEN 1 ELSE 0 END as has_shares
            FROM accounts a
            JOIN account_types at ON a.type_id = at.id
            WHERE a.user_id = :user_id1
            
            UNION ALL
            
            SELECT 
                a.*,
                at.code as type_code,
                at.name as type_name,
                ash.sharing_mode,
                owner.name as owner_name,
                owner.email as owner_email,
                ash.id as share_id,
                'shared' as account_status,
                0 as has_shares
            FROM account_shares ash
            JOIN accounts a ON ash.account_id = a.id
            JOIN account_types at ON a.type_id = at.id
            JOIN users owner ON ash.owner_user_id = owner.id
            WHERE ash.shared_with_user_id = :user_id2
            
            ORDER BY account_status DESC, name
        ");
        $stmt->execute([':user_id1' => $user_id, ':user_id2' => $user_id]);
        $accounts = $stmt->fetchAll();
        
        echo json_encode($accounts);
        break;
        
    case 'POST':
        // Create or update account - handle both JSON and FormData (for logo upload)
        // POST is used for both create and update when FormData is involved (because PUT doesn't populate $_POST)
        $data = [];
        $logo_path = null;
        $is_update = false;
        
        // Check if it's FormData or JSON
        $content_type = $_SERVER['CONTENT_TYPE'] ?? '';
        $is_multipart = strpos($content_type, 'multipart/form-data') !== false || !empty($_POST) || !empty($_FILES);
        
        if ($is_multipart) {
            // Handle FormData request
            $data = $_POST;
            
            // Check if this is an update (has 'id' field)
            if (!empty($data['id'])) {
                $is_update = true;
            }
            
            // Handle logo upload if file is provided
            if (!empty($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['logo'];
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'accounts/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Delete old logo if updating
                if ($is_update && !empty($data['id'])) {
                    $stmt = $conn->prepare("SELECT logo_path FROM accounts WHERE id = :id AND user_id = :user_id");
                    $stmt->execute([':id' => $data['id'], ':user_id' => $user_id]);
                    $old_account = $stmt->fetch();
                    if ($old_account && $old_account['logo_path']) {
                        $old_file = UPLOAD_DIR . $old_account['logo_path'];
                        if (file_exists($old_file)) {
                            unlink($old_file);
                        }
                    }
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $logo_path = 'accounts/' . $filename;
                }
            }
        } else {
            // Handle JSON request
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
        }
        
        if (!is_array($data)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid request data']);
            break;
        }
        
        // If this is an update, handle it differently
        if ($is_update) {
            // Handle update logic (similar to PUT case)
            $account_id = (int)$data['id'];
            
            // Verify account belongs to user
            $stmt = $conn->prepare("SELECT id, logo_path FROM accounts WHERE id = :id AND user_id = :user_id");
            $stmt->execute([':id' => $account_id, ':user_id' => $user_id]);
            $account = $stmt->fetch();
            
            if (!$account) {
                http_response_code(404);
                echo json_encode(['error' => 'Account not found']);
                break;
            }
            
            // If no new logo uploaded, keep existing logo_path
            if ($logo_path === null) {
                $logo_path = $account['logo_path'];
            }
            
            $required = ['name', 'type_id', 'base_currency', 'opening_date'];
            foreach ($required as $field) {
                if (!isset($data[$field]) || $data[$field] === '') {
                    http_response_code(400);
                    echo json_encode(['error' => "Missing required field: $field"]);
                    break 2;
                }
            }
            
            // Check if account name already exists for this user (excluding current account)
            $stmt = $conn->prepare("SELECT id FROM accounts WHERE user_id = :user_id AND name = :name AND id != :account_id");
            $stmt->execute([':user_id' => $user_id, ':name' => $data['name'], ':account_id' => $account_id]);
            if ($stmt->fetch()) {
                http_response_code(400);
                echo json_encode(['error' => 'An account with this name already exists']);
                break;
            }
            
            // Ensure type_id is an integer
            $data['type_id'] = (int)$data['type_id'];
            
            // Verify account type exists
            $stmt = $conn->prepare("SELECT id, requires_tax_specification FROM account_types WHERE id = :type_id");
            $stmt->execute([':type_id' => $data['type_id']]);
            $account_type = $stmt->fetch();
            
            if (!$account_type) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid account type']);
                break;
            }
            
            // If type is OTHER and requires tax specification, validate tax_type
            if ($account_type['requires_tax_specification'] && (!isset($data['tax_type']) || 
                !in_array($data['tax_type'], ['CURRENT_ACCOUNT_TAX', 'DEPOSIT_ACCOUNT_TAX']))) {
                http_response_code(400);
                echo json_encode(['error' => 'Tax type required for this account type']);
                break;
            }
            
            // Convert empty strings, null, or missing values to null for optional fields
            $id_number = (isset($data['id_number']) && $data['id_number'] !== '' && $data['id_number'] !== null) ? $data['id_number'] : null;
            $expire_date = (isset($data['expire_date']) && $data['expire_date'] !== '' && $data['expire_date'] !== null) ? $data['expire_date'] : null;
            $closure_date = (isset($data['closure_date']) && $data['closure_date'] !== '' && $data['closure_date'] !== null) ? $data['closure_date'] : null;
            $tax_type = (isset($data['tax_type']) && $data['tax_type'] !== '' && $data['tax_type'] !== null) ? $data['tax_type'] : null;
            
            // Handle show_in_dashboard checkbox (default to true if not provided)
            $show_in_dashboard = isset($data['show_in_dashboard']) 
                ? ($data['show_in_dashboard'] === 'on' || $data['show_in_dashboard'] === true || $data['show_in_dashboard'] === '1' || $data['show_in_dashboard'] === 1)
                : true;
            
            $stmt = $conn->prepare("
                UPDATE accounts 
                SET name = :name, type_id = :type_id, base_currency = :base_currency, 
                    opening_date = :opening_date, id_number = :id_number, 
                    expire_date = :expire_date, closure_date = :closure_date, tax_type = :tax_type, logo_path = :logo_path,
                    show_in_dashboard = :show_in_dashboard
                WHERE id = :id AND user_id = :user_id
            ");
            
            try {
                $stmt->bindValue(':id', $account_id, PDO::PARAM_INT);
                $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->bindValue(':name', $data['name'], PDO::PARAM_STR);
                $stmt->bindValue(':type_id', $data['type_id'], PDO::PARAM_INT);
                $stmt->bindValue(':base_currency', $data['base_currency'], PDO::PARAM_STR);
                $stmt->bindValue(':opening_date', $data['opening_date'], PDO::PARAM_STR);
                $stmt->bindValue(':id_number', $id_number, ($id_number === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
                $stmt->bindValue(':expire_date', $expire_date, ($expire_date === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
                $stmt->bindValue(':closure_date', $closure_date, ($closure_date === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
                $stmt->bindValue(':tax_type', $tax_type, ($tax_type === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
                $stmt->bindValue(':logo_path', $logo_path, ($logo_path === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
                $stmt->bindValue(':show_in_dashboard', $show_in_dashboard, PDO::PARAM_BOOL);
                
                $stmt->execute();
                
                echo json_encode(['success' => true, 'id' => $account_id]);
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => 'Database error: Failed to update account']);
                error_log('PDO Error updating account: ' . $e->getMessage());
            }
            break;
        }
        
        // Otherwise, handle as create
        $required = ['name', 'type_id', 'base_currency', 'opening_date'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || $data[$field] === '') {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        // Check if account name already exists for this user
        $stmt = $conn->prepare("SELECT id FROM accounts WHERE user_id = :user_id AND name = :name");
        $stmt->execute([':user_id' => $user_id, ':name' => $data['name']]);
        if ($stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'An account with this name already exists']);
            break;
        }
        
        // Ensure type_id is an integer
        $data['type_id'] = (int)$data['type_id'];
        
        // Verify account type exists
        $stmt = $conn->prepare("SELECT id, requires_tax_specification FROM account_types WHERE id = :type_id");
        $stmt->execute([':type_id' => $data['type_id']]);
        $account_type = $stmt->fetch();
        
        if (!$account_type) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid account type']);
            break;
        }
        
        // If type is OTHER and requires tax specification, validate tax_type
        if ($account_type['requires_tax_specification'] && (!isset($data['tax_type']) || 
            !in_array($data['tax_type'], ['CURRENT_ACCOUNT_TAX', 'DEPOSIT_ACCOUNT_TAX']))) {
            http_response_code(400);
            echo json_encode(['error' => 'Tax type required for this account type']);
            break;
        }
        
        // Handle show_in_dashboard checkbox (default to true if not provided)
        $show_in_dashboard = isset($data['show_in_dashboard']) 
            ? ($data['show_in_dashboard'] === 'on' || $data['show_in_dashboard'] === true || $data['show_in_dashboard'] === '1' || $data['show_in_dashboard'] === 1)
            : true;
        
        $stmt = $conn->prepare("
            INSERT INTO accounts (user_id, name, type_id, base_currency, opening_date, id_number, expire_date, closure_date, tax_type, logo_path, show_in_dashboard)
            VALUES (:user_id, :name, :type_id, :base_currency, :opening_date, :id_number, :expire_date, :closure_date, :tax_type, :logo_path, :show_in_dashboard)
        ");
        
        // Convert empty strings, null, or missing values to null for optional fields
        $id_number = (isset($data['id_number']) && $data['id_number'] !== '' && $data['id_number'] !== null) ? $data['id_number'] : null;
        $expire_date = (isset($data['expire_date']) && $data['expire_date'] !== '' && $data['expire_date'] !== null) ? $data['expire_date'] : null;
        $closure_date = (isset($data['closure_date']) && $data['closure_date'] !== '' && $data['closure_date'] !== null) ? $data['closure_date'] : null;
        $tax_type = (isset($data['tax_type']) && $data['tax_type'] !== '' && $data['tax_type'] !== null) ? $data['tax_type'] : null;
        
        try {
            // Bind parameters with proper null handling
            $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->bindValue(':name', $data['name'], PDO::PARAM_STR);
            $stmt->bindValue(':type_id', $data['type_id'], PDO::PARAM_INT);
            $stmt->bindValue(':base_currency', $data['base_currency'], PDO::PARAM_STR);
            $stmt->bindValue(':opening_date', $data['opening_date'], PDO::PARAM_STR);
            $stmt->bindValue(':id_number', $id_number, ($id_number === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':expire_date', $expire_date, ($expire_date === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':closure_date', $closure_date, ($closure_date === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':tax_type', $tax_type, ($tax_type === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':logo_path', $logo_path, ($logo_path === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':show_in_dashboard', $show_in_dashboard, PDO::PARAM_BOOL);
            
            $stmt->execute();
            
            $account_id = $conn->lastInsertId();
            
            echo json_encode(['success' => true, 'id' => $account_id]);
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Database error: Failed to create account']);
            error_log('PDO Error in accounts API: ' . $e->getMessage());
            error_log('Data being inserted: ' . print_r([
                'user_id' => $user_id,
                'name' => $data['name'],
                'type_id' => $data['type_id'],
                'expire_date' => $expire_date,
                'closure_date' => $closure_date
            ], true));
        }
        break;
        
    case 'PUT':
        // Update existing account - handle both JSON and FormData (for logo upload)
        $data = [];
        $logo_path = null;
        
        // Check Content-Type to determine if it's FormData or JSON
        $content_type = $_SERVER['CONTENT_TYPE'] ?? '';
        $is_multipart = strpos($content_type, 'multipart/form-data') !== false;
        
        if ($is_multipart) {
            // For PUT requests with FormData, PHP doesn't populate $_POST automatically
            // We need to parse it manually or use $_REQUEST
            // However, if the request is sent as POST with _method=PUT, $_POST will work
            // For true PUT with FormData, we need to parse multipart data manually
            
            // Try $_POST first (works if server rewrites PUT to POST)
            if (!empty($_POST)) {
                $data = $_POST;
            } else {
                // Parse multipart/form-data manually for PUT requests
                // This is complex, so we'll use a workaround: parse from php://input
                // Actually, for PUT with multipart, $_FILES should still work
                // Let's check if we can get form fields from the request
                
                // Workaround: For PUT requests, we'll require the client to send
                // form fields as part of the FormData, which should be accessible
                // Let's try $_REQUEST as fallback
                $data = !empty($_REQUEST) ? $_REQUEST : [];
                
                // If still empty, try parsing from raw input (complex multipart parsing)
                if (empty($data)) {
                    // For now, return error asking to use POST or include fields differently
                    http_response_code(400);
                    echo json_encode(['error' => 'PUT with FormData requires form fields. Please ensure all fields are included in FormData.']);
                    break;
                }
            }
            
            // Handle logo upload if file is provided
            if (!empty($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['logo'];
                if ($file['size'] > MAX_UPLOAD_SIZE) {
                    http_response_code(400);
                    echo json_encode(['error' => 'File too large']);
                    break;
                }
                
                $upload_dir = UPLOAD_DIR . 'accounts/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Delete old logo if exists
                if (!empty($data['id'])) {
                    $stmt = $conn->prepare("SELECT logo_path FROM accounts WHERE id = :id AND user_id = :user_id");
                    $stmt->execute([':id' => $data['id'], ':user_id' => $user_id]);
                    $old_account = $stmt->fetch();
                    if ($old_account && $old_account['logo_path']) {
                        $old_file = UPLOAD_DIR . $old_account['logo_path'];
                        if (file_exists($old_file)) {
                            unlink($old_file);
                        }
                    }
                }
                
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = uniqid() . '.' . $extension;
                $file_path = $upload_dir . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    $logo_path = 'accounts/' . $filename;
                }
            }
        } else {
            // Handle JSON request
            $raw_input = file_get_contents('php://input');
            $data = json_decode($raw_input, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON: ' . json_last_error_msg()]);
                break;
            }
        }
        
        if (!isset($data['id']) || empty($data['id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Account ID is required']);
            break;
        }
        
        $account_id = (int)$data['id'];
        
        // Verify account belongs to user
        $stmt = $conn->prepare("SELECT id, logo_path FROM accounts WHERE id = :id AND user_id = :user_id");
        $stmt->execute([':id' => $account_id, ':user_id' => $user_id]);
        $account = $stmt->fetch();
        
        if (!$account) {
            http_response_code(404);
            echo json_encode(['error' => 'Account not found']);
            break;
        }
        
        // Check if logo should be removed
        $remove_logo = isset($data['remove_logo']) && ($data['remove_logo'] === '1' || $data['remove_logo'] === 'on' || $data['remove_logo'] === true);
        
        if ($remove_logo) {
            // Delete old logo file if exists
            if ($account['logo_path']) {
                $old_file = UPLOAD_DIR . $account['logo_path'];
                if (file_exists($old_file)) {
                    unlink($old_file);
                }
            }
            $logo_path = null;
        } elseif ($logo_path === null) {
            // If no new logo uploaded and not removing, keep existing logo_path
            $logo_path = $account['logo_path'];
        }
        
        $required = ['name', 'type_id', 'base_currency', 'opening_date'];
        foreach ($required as $field) {
            if (!isset($data[$field]) || $data[$field] === '') {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                break 2;
            }
        }
        
        // Check if account name already exists for this user (excluding current account)
        $stmt = $conn->prepare("SELECT id FROM accounts WHERE user_id = :user_id AND name = :name AND id != :account_id");
        $stmt->execute([':user_id' => $user_id, ':name' => $data['name'], ':account_id' => $account_id]);
        if ($stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'An account with this name already exists']);
            break;
        }
        
        // Ensure type_id is an integer (FormData sends strings)
        $data['type_id'] = (int)$data['type_id'];
        $data['id'] = (int)$data['id'];
        
        // Verify account type exists
        $stmt = $conn->prepare("SELECT id, requires_tax_specification FROM account_types WHERE id = :type_id");
        $stmt->execute([':type_id' => $data['type_id']]);
        $account_type = $stmt->fetch();
        
        if (!$account_type) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid account type']);
            break;
        }
        
        // If type is OTHER and requires tax specification, validate tax_type
        if ($account_type['requires_tax_specification'] && (!isset($data['tax_type']) || 
            !in_array($data['tax_type'], ['CURRENT_ACCOUNT_TAX', 'DEPOSIT_ACCOUNT_TAX']))) {
            http_response_code(400);
            echo json_encode(['error' => 'Tax type required for this account type']);
            break;
        }
        
        // Convert empty strings, null, or missing values to null for optional fields
        $id_number = (isset($data['id_number']) && $data['id_number'] !== '' && $data['id_number'] !== null) ? $data['id_number'] : null;
        $expire_date = (isset($data['expire_date']) && $data['expire_date'] !== '' && $data['expire_date'] !== null) ? $data['expire_date'] : null;
        $closure_date = (isset($data['closure_date']) && $data['closure_date'] !== '' && $data['closure_date'] !== null) ? $data['closure_date'] : null;
        $tax_type = (isset($data['tax_type']) && $data['tax_type'] !== '' && $data['tax_type'] !== null) ? $data['tax_type'] : null;
        
        // Handle show_in_dashboard checkbox (default to true if not provided)
        $show_in_dashboard = isset($data['show_in_dashboard']) 
            ? ($data['show_in_dashboard'] === 'on' || $data['show_in_dashboard'] === true || $data['show_in_dashboard'] === '1' || $data['show_in_dashboard'] === 1)
            : true;
        
        $stmt = $conn->prepare("
            UPDATE accounts 
            SET name = :name, type_id = :type_id, base_currency = :base_currency, 
                opening_date = :opening_date, id_number = :id_number, 
                expire_date = :expire_date, closure_date = :closure_date, tax_type = :tax_type, logo_path = :logo_path,
                show_in_dashboard = :show_in_dashboard
            WHERE id = :id AND user_id = :user_id
        ");
        
        try {
            $stmt->bindValue(':id', $account_id, PDO::PARAM_INT);
            $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->bindValue(':name', $data['name'], PDO::PARAM_STR);
            $stmt->bindValue(':type_id', $data['type_id'], PDO::PARAM_INT);
            $stmt->bindValue(':base_currency', $data['base_currency'], PDO::PARAM_STR);
            $stmt->bindValue(':opening_date', $data['opening_date'], PDO::PARAM_STR);
            $stmt->bindValue(':id_number', $id_number, ($id_number === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':expire_date', $expire_date, ($expire_date === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':closure_date', $closure_date, ($closure_date === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':tax_type', $tax_type, ($tax_type === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':logo_path', $logo_path, ($logo_path === null ? PDO::PARAM_NULL : PDO::PARAM_STR));
            $stmt->bindValue(':show_in_dashboard', $show_in_dashboard, PDO::PARAM_BOOL);
            
            $stmt->execute();
            
            echo json_encode(['success' => true, 'id' => $account_id]);
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Database error: Failed to update account']);
            error_log('PDO Error updating account: ' . $e->getMessage());
        }
        break;
        
    case 'DELETE':
        // Delete account (only if no expenses exist)
        $account_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$account_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Account ID is required']);
            break;
        }
        
        // Verify account belongs to user
        $stmt = $conn->prepare("SELECT id FROM accounts WHERE id = :id AND user_id = :user_id");
        $stmt->execute([':id' => $account_id, ':user_id' => $user_id]);
        $account = $stmt->fetch();
        
        if (!$account) {
            http_response_code(404);
            echo json_encode(['error' => 'Account not found']);
            break;
        }
        
        // Check if account has any expenses
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM expenses WHERE account_id = :account_id");
        $stmt->execute([':account_id' => $account_id]);
        $result = $stmt->fetch();
        
        if ($result['count'] > 0) {
            http_response_code(400);
            echo json_encode(['error' => 'Cannot delete account: it has associated expenses']);
            break;
        }
        
        // Get account logo_path before deletion
        $stmt = $conn->prepare("SELECT logo_path FROM accounts WHERE id = :id AND user_id = :user_id");
        $stmt->execute([':id' => $account_id, ':user_id' => $user_id]);
        $account = $stmt->fetch();
        
        // Delete logo file if exists
        if ($account && $account['logo_path']) {
            $logo_file = UPLOAD_DIR . $account['logo_path'];
            if (file_exists($logo_file)) {
                unlink($logo_file);
            }
        }
        
        // Delete the account
        try {
            $stmt = $conn->prepare("DELETE FROM accounts WHERE id = :id AND user_id = :user_id");
            $stmt->execute([':id' => $account_id, ':user_id' => $user_id]);
            
            echo json_encode(['success' => true]);
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Database error: Failed to delete account']);
            error_log('PDO Error deleting account: ' . $e->getMessage());
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    ob_clean(); // Clear any output
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
    error_log('Accounts API Error: ' . $e->getMessage());
}

// End output buffering
ob_end_flush();

