# Sistema di Debug Sessioni

Questo documento spiega come utilizzare il sistema di debug per tracciare i problemi di sessione.

## Abilitazione del Debug

### Backend (PHP)

Per abilitare il debug delle sessioni sul backend, modifica `config/config.php`:

```php
define('SESSION_DEBUG_ENABLED', true); // Cambia da false a true
```

I log verranno scritti in `logs/session_debug.log`.

### Frontend (JavaScript)

Per abilitare il debug nel browser, apri la console del browser e esegui:

```javascript
localStorage.setItem('sessionDebugEnabled', 'true');
```

Poi ricarica la pagina. I log appariranno nella console del browser e verranno salvati in `localStorage` con la chiave `sessionDebugLogs`.

Per disabilitare:

```javascript
localStorage.removeItem('sessionDebugEnabled');
localStorage.removeItem('sessionDebugLogs');
```

## Eventi Tracciati

Il sistema traccia i seguenti eventi:

### Backend

- `session_created_login` - Quando viene creata una nuova sessione dopo il login
- `check_session_called` - Quando viene chiamato l'endpoint di verifica sessione
- `check_session_success` - Quando la verifica sessione ha successo
- `check_session_expired` - Quando la sessione è scaduta durante la verifica
- `check_session_no_user_id` - Quando non c'è user_id nella sessione
- `cookie_updated` - Quando viene aggiornato il cookie di sessione
- `session_expired_absolute` - Quando la sessione scade per scadenza assoluta (180 giorni)
- `first_login_at_set` - Quando viene impostato il timestamp del primo login
- `requireAuth_called` - Quando viene chiamata la funzione requireAuth
- `auth_failed_no_user_id` - Quando l'autenticazione fallisce per mancanza di user_id
- `auth_success` - Quando l'autenticazione ha successo
- `session_destroyed_user_blocked` - Quando la sessione viene distrutta perché l'utente è bloccato
- `logout_called` - Quando viene chiamato il logout
- `session_destroyed_logout` - Quando la sessione viene distrutta per logout

### Frontend

- `checkAuth_called` - Quando viene chiamata la funzione checkAuth
- `checkSession_response` - Risposta ricevuta da checkSession
- `checkAuth_success` - Quando checkAuth ha successo
- `checkAuth_failed_not_authenticated` - Quando checkAuth fallisce per mancanza di autenticazione
- `checkAuth_error` - Quando si verifica un errore durante checkAuth
- `logout_called` - Quando viene chiamato il logout dal frontend
- `logout_success` - Quando il logout ha successo
- `logout_error` - Quando si verifica un errore durante il logout

## Visualizzazione dei Log

### Log Backend

I log backend sono salvati in formato JSON, una riga per evento, nel file `logs/session_debug.log`.

Puoi visualizzarli con:

```bash
tail -f logs/session_debug.log
```

Oppure usando l'endpoint API (solo admin):

```javascript
// Nel browser console o nel codice
const logs = await API.getSessionDebugLogs({
    limit: 100,        // Numero di log da visualizzare
    offset: 0,         // Offset per paginazione
    event: 'check_session_called',  // Filtra per evento specifico
    user_id: 123       // Filtra per user_id specifico
});
console.log(logs);
```

### Log Frontend

I log frontend sono salvati in `localStorage` con la chiave `sessionDebugLogs`.

Per visualizzarli nella console:

```javascript
const logs = JSON.parse(localStorage.getItem('sessionDebugLogs') || '[]');
console.table(logs);
```

## Informazioni Tracciate

Ogni log contiene:

- `timestamp` - Data e ora dell'evento
- `event` - Nome dell'evento
- `session_id` - ID della sessione (troncato per sicurezza)
- `user_id` - ID dell'utente
- `ip` - Indirizzo IP della richiesta
- `user_agent` - User agent del browser
- `data` - Dati aggiuntivi specifici dell'evento

## Analisi dei Problemi Comuni

### Sessione chiusa dopo poche ore

Cerca nei log:
1. Eventi `session_expired_absolute` - indica scadenza assoluta (180 giorni)
2. Eventi `cookie_updated` - verifica le date di scadenza del cookie
3. Eventi `check_session_no_user_id` - indica che la sessione non contiene user_id
4. Verifica `days_until_cookie_expiration` e `days_until_absolute_expiration` nei log

### Cookie non viene salvato

Cerca:
- Eventi `cookie_updated` e verifica `cookie_secure`, `cookie_samesite`, `cookie_domain`
- Se `cookie_secure` è `true` ma stai usando HTTP, il cookie non verrà salvato
- Se `cookie_samesite` è `Strict`, potrebbe causare problemi con redirect

### Sessione persa dopo riavvio browser

Verifica:
- `cookie_expiration` nei log - se è nel passato, il cookie è scaduto
- `cookie_received` nei log `check_session_called` - se è `false`, il cookie non è stato ricevuto

## Disabilitazione

Per disabilitare il debug:

1. Backend: imposta `SESSION_DEBUG_ENABLED` a `false` in `config/config.php`
2. Frontend: rimuovi `sessionDebugEnabled` da `localStorage`

## Note di Sicurezza

- I log contengono informazioni sensibili (session_id, user_id, IP)
- I log backend sono accessibili solo agli admin tramite l'endpoint API
- I log frontend sono salvati solo localmente nel browser
- Non condividere i log pubblicamente
