# Migrazione a Database Session Handler

## Problema Risolto

Il timer di sistema `phpsessionclean.timer` pulisce i file di sessione PHP prima che scadano secondo le impostazioni dell'applicazione, causando la disconnessione degli utenti.

## Soluzione Implementata

Abbiamo implementato un **Database Session Handler** che memorizza le sessioni nel database invece che nei file. Questo rende le sessioni completamente indipendenti dal timer di sistema.

## Installazione

### 1. Eseguire la Migration SQL

```bash
mysql -u root -p personal_expenses < database/migration_add_user_sessions.sql
```

Oppure eseguire manualmente:

```sql
CREATE TABLE IF NOT EXISTS user_sessions (
    session_id VARCHAR(128) PRIMARY KEY,
    user_id INT NOT NULL,
    ip_address VARCHAR(45),
    user_agent VARCHAR(255),
    data TEXT,
    last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP NOT NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_expires_at (expires_at),
    INDEX idx_last_activity (last_activity)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

### 2. Verificare la Configurazione

Il file `config/config.php` è già stato modificato per usare il database session handler. Non sono necessarie altre modifiche.

### 3. Testare

1. Fai login su un browser
2. Chiudi il browser
3. Riapri il browser dopo alcune ore
4. Dovresti essere ancora loggato

## Vantaggi

1. **Indipendente dal timer di sistema**: Le sessioni non vengono più eliminate dal `phpsessionclean.timer`
2. **Controllo completo**: Possiamo gestire la scadenza delle sessioni direttamente nel database
3. **Tracciabilità**: Possiamo vedere tutte le sessioni attive nel database
4. **Performance**: Le sessioni vengono caricate direttamente dal database, più veloce dei file

## Pulizia Automatica

Il garbage collection delle sessioni scadute viene gestito automaticamente da PHP tramite il metodo `gc()` del session handler. Le sessioni vengono eliminate quando:
- Scadono (`expires_at < NOW()`)
- Viene chiamato il garbage collection di PHP

## Monitoraggio

Puoi vedere tutte le sessioni attive nel database:

```sql
SELECT 
    session_id,
    user_id,
    ip_address,
    created_at,
    last_activity,
    expires_at,
    TIMESTAMPDIFF(HOUR, NOW(), expires_at) as hours_until_expiration
FROM user_sessions
WHERE expires_at > NOW()
ORDER BY last_activity DESC;
```

## Rollback (se necessario)

Se vuoi tornare ai file di sessione, rimuovi queste righe da `config/config.php`:

```php
// Use database session handler instead of file-based sessions
require_once __DIR__ . '/../api/middleware/db_session_handler.php';
$dbSessionHandler = new DatabaseSessionHandler();
session_set_save_handler($dbSessionHandler, true);
```

E ripristina il codice originale per i file di sessione.
